/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Mathias Panzenböck <grosser.meister.morti@gmx.net>
* Copyright (C) 2009-2010 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "SmoothTasks/ByShapeTaskbarLayout.h"

#include <QApplication>

#include <cmath>

namespace SmoothTasks {

int ByShapeTaskbarLayout::optimumCapacity() const {
	const QRectF effectiveRect(effectiveGeometry());
	
	const bool isVertical = orientation() == Qt::Vertical;
	const qreal availableWidth = isVertical ? effectiveRect.height() : effectiveRect.width();
	
	const qreal spacing = this->spacing();
	
	const int itemsPerRow = std::ceil((availableWidth + spacing) / (averagePreferredItemRdWidthStatic(m_cellHeight, true) + spacing));
	
	return itemsPerRow * maximumRows();
}

void ByShapeTaskbarLayout::setRowAspectRatio(qreal rowAspectRatio) {
	if (rowAspectRatio <= 0.0) {
		qWarning("ByShapeTaskbarLayout::setRowAspectRatio: illegal rowAspectRatio: %g", rowAspectRatio);
	}

	if (m_rowAspectRatio != rowAspectRatio) {
		m_rowAspectRatio = rowAspectRatio;
		invalidate();
	}
}

void ByShapeTaskbarLayout::updateLayoutStatic() {
	const bool isVertical = orientation() == Qt::Vertical;
	
	const QList<TaskbarItem*>& items = this->items();
	const int N = items.size();
	
	const QRectF effectiveRect(effectiveGeometry());
	const qreal availableWidth  = isVertical ? effectiveRect.height() : effectiveRect.width();
	const qreal availableHeight = isVertical ? effectiveRect.width()  : effectiveRect.height();
	const qreal spacing         = this->spacing();
	
#define CELL_HEIGHT(ROWS) (((availableHeight + spacing) / ((qreal) (ROWS))) - spacing)
	
	// if there is nothing to layout fill in some dummy data and leave
	if (N == 0) {
		m_rows = minimumRows();
		m_cellHeight = CELL_HEIGHT(m_rows);
		m_rowInfos.clear();
		return;
	}
	
	qreal cellHeight;
	
	int rows = minimumRows();
	
	// the row width has to be bigger than the row height for a nice looking layout
	while ((cellHeight = CELL_HEIGHT(rows)) * m_rowAspectRatio > availableWidth) {
		++ rows;
	}
	
	if (rows > maximumRows()) {
		rows = maximumRows();
	}
	
	int itemsPerRow = std::ceil((availableWidth + spacing) / (averagePreferredItemRdWidthStatic(cellHeight) + spacing));
	
	if (itemsPerRow * rows < N) {
		itemsPerRow = std::ceil(((qreal) N) / rows);
	}
	
	QList<RowInfo> rowInfos;
	
	buildRows(itemsPerRow, rowInfos, rows);
	
	m_rows = rows;
	m_cellHeight = cellHeight;
	m_rowInfos = rowInfos;
	updateItemsRowCache();
	
#undef CELL_HEIGHT
}

} // namespace SmoothTasks
