/******************************************************************************
* Copyright (c) 2014, Pete Gadomski (pete.gadomski@gmail.com)
*
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following
* conditions are met:
*
*     * Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     * Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in
*       the documentation and/or other materials provided
*       with the distribution.
*     * Neither the name of Hobu, Inc. or Flaxen Geo Consulting nor the
*       names of its contributors may be used to endorse or promote
*       products derived from this software without specific prior
*       written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
* FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
* COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
* OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
* AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
* OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
* OF SUCH DAMAGE.
****************************************************************************/

#include <pdal/pdal_test_main.hpp>

#include <pdal/Writer.hpp>
#include <pdal/StageFactory.hpp>
#include <pdal/util/Algorithm.hpp>

#include "Support.hpp"
#include "Pgtest-Support.hpp"
#include "../io/PgCommon.hpp"

using namespace pdal;

namespace { // anonymous

std::string getTestConnBase()
{
    std::string s;
    if ( ! testDbPort.empty() )
        s += " port='" + testDbPort + "'";
    if ( ! testDbHost.empty() )
        s += " host='" + testDbHost + "'";
    if ( ! testDbUser.empty() )
        s += " user='" + testDbUser + "'";
    return s;
}


std::string getConnectionString(const std::string& dbname)
{
    return getTestConnBase() + " dbname='" + dbname + "'";
}


std::string getTestDBTempConn()
{
    return getConnectionString(testDbTempname);
}

std::string getMasterDBConn()
{
    return getConnectionString(testDbName);
}

} // anonymous namespace

Options getDbOptions()
{
    Options options;

    options.add(Option("connection", getTestDBTempConn()));
    options.add(Option("table", "4dal-\"test\"-table")); // intentional quotes
    options.add(Option("column", "p\"a")); // intentional quotes

    return options;
}

class PgpointcloudWriterTest : public testing::Test
{
public:
    PgpointcloudWriterTest() : m_masterConnection(0), m_testConnection(0),
                               m_bSkipTests(false) {};
protected:
    virtual void SetUp()
    {
        std::string connstr = getMasterDBConn();
        m_masterConnection = pg_connect( connstr );
        m_testConnection = NULL;

        // Silence those pesky notices
        executeOnMasterDb("SET client_min_messages TO WARNING");

        dropTestDb();

        std::stringstream createDbSql;
        createDbSql << "CREATE DATABASE " <<
            testDbTempname << " TEMPLATE template0";
        try
        {
            executeOnMasterDb(createDbSql.str());
        }
        catch( const pdal_error& )
        {
            m_bSkipTests = true;
            return;
        }

        m_testConnection = pg_connect( getTestDBTempConn() );

        try
        {
            executeOnTestDb("CREATE EXTENSION pointcloud");
        }
        catch( const pdal_error& )
        {
            m_bSkipTests = true;
            return;
        }
    }

    void executeOnTestDb(const std::string& sql)
    {
        pg_execute(m_testConnection, sql);
    }

    virtual void TearDown()
    {
        if (!m_testConnection || !m_masterConnection) return;
        if (m_testConnection)
        {
            PQfinish(m_testConnection);
        }
        dropTestDb();
        if (m_masterConnection)
        {
            PQfinish(m_masterConnection);
        }
    }

    bool shouldSkipTests() const { return m_bSkipTests; }

private:

    void executeOnMasterDb(const std::string& sql)
    {
        pg_execute(m_masterConnection, sql);
    }

    void execute(PGconn* connection, const std::string& sql)
    {
        pg_execute(connection, sql);
    }

    void dropTestDb()
    {
        std::stringstream dropDbSql;
        dropDbSql << "DROP DATABASE IF EXISTS " << testDbTempname;
        executeOnMasterDb(dropDbSql.str());
    }

    PGconn* m_masterConnection;
    PGconn* m_testConnection;
    bool m_bSkipTests;
};

namespace
{

void optionsWrite(const Options& writerOps)
{
    StageFactory f;
    Stage* reader(f.createStage("readers.las"));

    const std::string file(Support::datapath("las/1.2-with-color.las"));
    Options options;
    options.add("filename", file);
    reader->setOptions(options);

    Stage* writer(f.createStage("writers.pgpointcloud"));
    writer->setOptions(writerOps);
    writer->setInput(*reader);

    PointTable table;
    writer->prepare(table);

    PointViewSet written = writer->execute(table);

    point_count_t count(0);
    for(auto i = written.begin(); i != written.end(); ++i)
	    count += (*i)->size();
    EXPECT_EQ(written.size(), 1U);
    EXPECT_EQ(count, 1065U);
}

} // unnamed namespace

TEST_F(PgpointcloudWriterTest, write)
{
    if (shouldSkipTests())
    {
        return;
    }

    optionsWrite(getDbOptions());
}

TEST_F(PgpointcloudWriterTest, writeScaled)
{
    if (shouldSkipTests())
    {
        return;
    }

    Options ops = getDbOptions();
    ops.add("scale_x", .01);
    ops.add("scale_y", .01);
    ops.add("scale_z", .01);

    optionsWrite(ops);
}

TEST_F(PgpointcloudWriterTest, writeXYZ)
{
    if (shouldSkipTests())
    {
        return;
    }

    Options ops = getDbOptions();
    ops.add("output_dims", "X,Y,Z");

    optionsWrite(ops);

    PointTable table;
    StageFactory factory;
    Stage* reader(factory.createStage("readers.pgpointcloud"));
    reader->setOptions(getDbOptions());

    reader->prepare(table);
    Dimension::IdList dims = table.layout()->dims();
    EXPECT_EQ(dims.size(), (size_t)3);
    EXPECT_TRUE(Utils::contains(dims, Dimension::Id::X));
    EXPECT_TRUE(Utils::contains(dims, Dimension::Id::Y));
    EXPECT_TRUE(Utils::contains(dims, Dimension::Id::Z));
}

TEST_F(PgpointcloudWriterTest, writetNoPointcloudExtension)
{
    if (shouldSkipTests())
    {
        return;
    }

    StageFactory f;
    Stage* writer(f.createStage("writers.pgpointcloud"));
    EXPECT_TRUE(writer);

    executeOnTestDb("DROP EXTENSION pointcloud");

    const std::string file(Support::datapath("las/1.2-with-color.las"));

    const Option opt_filename("filename", file);

    Stage* reader(f.createStage("readers.las"));
    EXPECT_TRUE(reader);
    Options options;
    options.add(opt_filename);
    reader->setOptions(options);
    writer->setOptions(getDbOptions());
    writer->setInput(*reader);

    PointTable table;
    writer->prepare(table);

    EXPECT_THROW(writer->execute(table), pdal_error);
}

TEST_F(PgpointcloudWriterTest, writeDeleteTable)
{
    if (shouldSkipTests())
    {
        return;
    }

    executeOnTestDb("CREATE SCHEMA \"4dal-\"\"test\"\"-schema\"");
    executeOnTestDb("CREATE TABLE \"4dal-\"\"test\"\"-schema\"."
                    "\"4dal-\"\"test\"\"-table\" (p PCPATCH)");
    Options ops = getDbOptions();
    ops.add("overwrite", true);
    ops.add("schema", "4dal-\"test\"-schema");

    optionsWrite(ops);
}
