///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/input/XFormManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/viewport/input/MoveMode.h>
#include <core/viewport/input/RotationMode.h>
#include <core/viewport/input/ScalingMode.h>
#include <core/viewport/input/SelectionMode.h>
#include <core/scene/SceneNode.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/animation/AnimManager.h>
#include <core/scene/animation/TimeInterval.h>
#include <core/data/DataSetManager.h>

namespace Core {

/// The singleton instance of this class.
XFormManager* XFormManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the xform manager.
******************************************************************************/
XFormManager::XFormManager() :
	tripodSize(1.6f), tripodArrowSize(0.1f),
	_xformSystem(CONSTRUCTION_GRID_SYS),
	_centerMode(SELECTION_CENTER)
{
	// Reset the xform manager when a new scene has been loaded.
	connect(&DATASET_MANAGER, SIGNAL(dataSetReset(DataSet*)), this, SLOT(reset()));

	_objectSelectionMode = new SelectionMode();
	_objectMoveMode = new MoveMode();
	_objectRotationMode = new RotationMode();
	_objectScalingMode = new ScalingMode();
}

/******************************************************************************
* Sets the coordinate system to use for transformation of scene nodes.
******************************************************************************/
void XFormManager::setXFormSystem(XFormSystem sys)
{
	if(_xformSystem == sys) return;
    _xformSystem = sys;
	// Raise event.
    xformSystemChanged();
	// Update viewports
	VIEWPORT_MANAGER.updateViewports();
}

/******************************************************************************
* Sets the center mode to use for transformation of scene nodes.
* The center mode controls which point to use as origin for scaling and rotation of nodes.
******************************************************************************/
void XFormManager::setCenterMode(XFormCenterMode mode)
{
	if(_centerMode == mode) return;
	_centerMode = mode;
	// Raise event.
    xformCenterChanged();
	// Update viewports.
	VIEWPORT_MANAGER.updateViewports();
}

/******************************************************************************
* Resets the transformation settings to their default values.
******************************************************************************/
void XFormManager::reset()
{
    setXFormSystem(CONSTRUCTION_GRID_SYS);
	setCenterMode(SELECTION_CENTER);
}

/******************************************************************************
* Returns the transformation matrix of coordinate system to use for
* xform manipulation modes.
*******************************************************************************/
void XFormManager::getTransformationSystem(SceneNode* contextNode, AffineTransformation& sysMat)
{
	Viewport* vp = VIEWPORT_MANAGER.activeViewport();
	if(vp == NULL) {
		sysMat = IDENTITY;
		return;
	}

	switch(xformSystem()) {

		case CONSTRUCTION_GRID_SYS:
            sysMat = vp->grid().gridMatrix();
			break;

		case LOCAL_SYS:	{
			TimeInterval interval;
			sysMat = contextNode->getWorldTransform(ANIM_MANAGER.time(), interval);
			if(centerMode() == LOCAL_ORIGIN) return;
			}
			break;

		case SCREEN_SYS:
			sysMat = vp->inverseViewMatrix();
			break;

		default:
			sysMat = IDENTITY;
			break;
	}

	// Set transformation origin.
	if(centerMode() != SYSTEM_ORIGIN) {
		Point3 center = getTransformationCenter(contextNode);
        sysMat.setTranslation(center - ORIGIN);
	}
}

/******************************************************************************
* Returns the origin of the transformation system to use for xform modes.
*    contextNode - The scene node whose transformation center should be returned.
* Returns the origin (in world coordinates) of the transformation system.
******************************************************************************/
Point3 XFormManager::getTransformationCenter(SceneNode* contextNode)
{
	switch(centerMode()) {

		case SELECTION_CENTER: {
			SelectionSet* sel = DATASET_MANAGER.currentSelection();
			if(sel->empty()) return ORIGIN;
			TimeInterval interval;
			Vector3 center = NULL_VECTOR;
			Q_FOREACH(SceneNode* node, sel->nodes()) {
				const AffineTransformation& nodeTM = node->getWorldTransform(ANIM_MANAGER.time(), interval);
				center += nodeTM.getTranslation();
			}
			center /= (FloatType)sel->count();
			return ORIGIN + center;
		}

		case LOCAL_ORIGIN: {
			TimeInterval interval;
			const AffineTransformation& nodeTM = contextNode->getWorldTransform(ANIM_MANAGER.time(), interval);
			return ORIGIN + nodeTM.getTranslation();
		}

		case SYSTEM_ORIGIN: {
			AffineTransformation sysTM;
			getTransformationSystem(contextNode, sysTM);
			return ORIGIN + sysTM.getTranslation();
		}

		default:
			return ORIGIN;
	}
}

/******************************************************************************
* Renders the tripods for all selected nodes in the scene.
******************************************************************************/
void XFormManager::renderTripods(Viewport* vp)
{
	// Render tripods only when a transformation mode is active.
	if(VIEWPORT_INPUT_MANAGER.currentHandler() != objectMoveMode() &&
		VIEWPORT_INPUT_MANAGER.currentHandler() != objectRotationMode() &&
		VIEWPORT_INPUT_MANAGER.currentHandler() != objectScalingMode())
		return;

	CHECK_POINTER(vp);
	SelectionSet* selection = DATASET_MANAGER.currentSelection();
    if(selection->empty()) return;

    // Disable z-buffer.
	vp->setDepthTest(false);
	vp->setLightingEnabled(false);

	if(centerMode() == LOCAL_ORIGIN) {
		// Render one tripod for every selected node.
		Q_FOREACH(SceneNode* node, selection->nodes()) {

			// Do not render tripod for a node if it is the view node of the viewport or the target of the view node.
			if(vp->settings()->viewNode() && (vp->settings()->viewNode() == node || vp->settings()->viewNode()->targetNode() == node))
				continue;

			AffineTransformation tm;
			getTransformationSystem(node, tm);
			renderTripod(vp, tm);
		}
	}
	else {
		Q_FOREACH(SceneNode* node, selection->nodes()) {

			// Do not render tripod for a node if it is the view node of the viewport or the target of the view node.
			if(vp->settings()->viewNode() && (vp->settings()->viewNode() == node || vp->settings()->viewNode()->targetNode() == node))
				continue;

			// Render only one tripod because they are all the same for every node.
			AffineTransformation tm;
			getTransformationSystem(node, tm);
			renderTripod(vp, tm);
			break;

		}
	}
}

/******************************************************************************
* Renders a single tripod into the viewport.
******************************************************************************/
void XFormManager::renderTripod(Viewport* vp, const AffineTransformation& tm, bool onlyRotationComponent)
{
	// Compute tripod size.
	Point3 origin = ORIGIN + tm.getTranslation();
	FloatType size = vp->nonScalingSize(origin) * tripodSize;
	if(size <= 0) return;
	if(abs(tm.determinant()) < 1e-5) return;

	if(onlyRotationComponent) {
		// Remove scaling part of transformation matrix and apply scaling.
		Quaternion quat(tm);
		vp->setWorldMatrix(AffineTransformation::translation(tm.getTranslation()) * AffineTransformation::rotation(quat) * AffineTransformation::scaling(size));
	}
	else {
		vp->setWorldMatrix(tm * AffineTransformation::scaling(size));
	}

	// Render lines of the tripod.
	Point3 verts[18];
	verts[0] = ORIGIN;
	verts[1] = Point3(1,0,0);
	verts[2] = Point3(1,0,0);
	verts[3] = Point3(1.0-tripodArrowSize,tripodArrowSize,0);
	verts[4] = Point3(1,0,0);
	verts[5] = Point3(1.0-tripodArrowSize,-tripodArrowSize,0);
	verts[6] = ORIGIN;
	verts[7] = Point3(0,1,0);
	verts[8] = Point3(0,1,0);
	verts[9] = Point3(tripodArrowSize,1.0-tripodArrowSize,0);
	verts[10] = Point3(0,1,0);
	verts[11] = Point3(-tripodArrowSize,1.0-tripodArrowSize,0);
	verts[12] = ORIGIN;
	verts[13] = Point3(0,0,1);
	verts[14] = Point3(0,0,1);
	verts[15] = Point3(tripodArrowSize,0,1.0-tripodArrowSize);
	verts[16] = Point3(0,0,1);
	verts[17] = Point3(-tripodArrowSize,0,1.0-tripodArrowSize);
	vp->setRenderingColor(Color(1,0,0));
	vp->renderLines(18, Box3(Point3(-tripodArrowSize), Point3(1)), verts);

	// Render x,y,z axis labels.
	Point2 sp;
	if(vp->projectObjectPoint(Point3(1.2f,0,0), sp)) {
		vp->setRenderingColor(Color(1,0,0));
		vp->renderText((int)sp.X, (int)sp.Y, "x");
	}
	if(vp->projectObjectPoint(Point3(0,1.2f,0), sp)) {
		vp->setRenderingColor(Color(0,1,0));
		vp->renderText((int)sp.X, (int)sp.Y, "y");
	}
	if(vp->projectObjectPoint(Point3(0,0,1.2f), sp)) {
		vp->setRenderingColor(Color(0,0,1));
		vp->renderText((int)sp.X, (int)sp.Y, "z");
	}
}


};
