///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/SceneObject.h>
#include <core/scene/objects/ModifiedObject.h>
#include <core/scene/objects/Modifier.h>
#include <core/scene/ObjectNode.h>
#include <core/viewport/ViewportManager.h>
#include <core/undo/UndoManager.h>
#include <core/actions/ActionManager.h>
#include <core/plugins/branding/Branding.h>
#include <core/plugins/PluginManager.h>
#include "ModifierStack.h"
#include "ModifyCommandPage.h"

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ModifierStackEntry, RefTarget)
DEFINE_FLAGS_REFERENCE_FIELD(ModifierStackEntry, RefTarget, "Object", PROPERTY_FIELD_NO_UNDO, object)
DEFINE_FLAGS_VECTOR_REFERENCE_FIELD(ModifierStackEntry, ModifierApplication, "ModifierApplications", PROPERTY_FIELD_NO_UNDO, modApps)

/******************************************************************************
* Constructor.
******************************************************************************/
ModifierStackEntry::ModifierStackEntry(ModifierStack* _stack, RefTarget* commonObject) : RefTarget(false), stack(_stack)
{
	INIT_PROPERTY_FIELD(ModifierStackEntry, object);
	INIT_PROPERTY_FIELD(ModifierStackEntry, modApps);

	this->object = commonObject;
}


IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ModifierStack, RefMaker)
DEFINE_FLAGS_VECTOR_REFERENCE_FIELD(ModifierStack, ModifierStackEntry, "StackEntries", PROPERTY_FIELD_NO_UNDO, stackEntries)
DEFINE_FLAGS_VECTOR_REFERENCE_FIELD(ModifierStack, ObjectNode, "SelectedNodes", PROPERTY_FIELD_NO_UNDO, selectedNodes)

/******************************************************************************
* Initializes the object.
******************************************************************************/
ModifierStack::ModifierStack(ModifyCommandPage* modifyPage)
	: RefMaker(), page(modifyPage), nextObjectToSelect(NULL), needStackUpdate(false)
{
	INIT_PROPERTY_FIELD(ModifierStack, stackEntries);
	INIT_PROPERTY_FIELD(ModifierStack, selectedNodes);

	loadModifierCategories();

	_listModel = new ModifierStackModel(this);
	setParent(modifyPage);
	connect(this, SIGNAL(internalStackUpdate()), this, SLOT(onInternalStackUpdate()), Qt::QueuedConnection);
}

/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool ModifierStackEntry::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	// The modifier stack display must be updated if a modifier has been added or removed
	// from a ModifiedObject.
	if((msg->type() == REFERENCE_FIELD_ADDED || msg->type() == REFERENCE_FIELD_REMOVED || msg->type() == REFERENCE_FIELD_CHANGED)
		&& source == object && dynamic_object_cast<ModifiedObject>((RefTarget*)object))
	{
		stack->invalidate();
	}
	/// Update entry if the modifier has been enabled or disabled.
	else if(msg->type() == MODIFIER_ENABLED && source == object && msg->sender() == object) {
		stack->listModel()->refreshStackEntry(this);
	}
	/// Update entry if the evaluation status of the modifiers has changed.
	else if(msg->type() == REFTARGET_STATUS_CHANGED) {
		stack->listModel()->refreshStackEntry(this);
	}

	return RefTarget::onRefTargetMessage(source, msg);
}

/******************************************************************************
* Returns the internal selection model.
******************************************************************************/
QItemSelectionModel* ModifierStack::selectionModel() const
{
	return page->stackBox->selectionModel();
}

/******************************************************************************
* Clears the displayed modification stack.
******************************************************************************/
void ModifierStack::clearStack()
{
	listModel()->clear();
	UndoSuspender noUndo;
	stackEntries.clear();
	selectedNodes.clear();
	updatePropertiesPanel();
}

/******************************************************************************
* Filters the given input list for ObjectNodes.
******************************************************************************/
void ModifierStack::collectObjectNodes(const QVector<SceneNode*>& in)
{
	Q_FOREACH(SceneNode* node, in) {
		if(node->isObjectNode()) {
			selectedNodes.push_back(static_object_cast<ObjectNode>(node));
		}
		else if(node->isGroupNode()) {
            // Step recursively into the group node.
			collectObjectNodes(node->children());
		}
	}
}

/******************************************************************************
* If all selected object nodes reference the same SceneObject
* then it is returned by this method; otherwise NULL is returned.
******************************************************************************/
SceneObject* ModifierStack::commonObject()
{
	SceneObject* obj = NULL;
	foreach_ref(ObjectNode* objNode, selectedNodes) {
		if(obj == NULL) obj = objNode->sceneObject();
		else if(obj != objNode->sceneObject()) return NULL;	// The scene nodes are not compatible.
	}
	return obj;
}

/******************************************************************************
* Completely rebuilds the modifier stack list.
******************************************************************************/
void ModifierStack::refreshModifierStack()
{
	UndoSuspender noUndo;

	// Determine the currently selected object and
	// try to select it again after the stack has been rebuilt.
	// If nextObjectToSelect is already non-NULL then the caller
	// has specified an object to be selected.
	if(nextObjectToSelect == NULL) {
		QModelIndexList selection = page->stackBox->selectionModel()->selectedRows();
		if(!selection.empty()) {
			ModifierStackEntry* entry = (ModifierStackEntry*)selection.front().data(Qt::UserRole).value<void*>();
			CHECK_OBJECT_POINTER(entry);
			nextObjectToSelect = entry->commonObject();
		}
	}

	// Remove old stuff.
	clearStack();

	// Collect all selected ObjectNodes.
	selectedNodes.clear();
	collectObjectNodes(DATASET_MANAGER.currentSelection()->nodes());

    SceneObject* cmnObject = commonObject();
	if(cmnObject != NULL) {
		// The user has selected a set of identical instances.

		// Walk up the pipeline.
		do {
			CHECK_OBJECT_POINTER(cmnObject);

			// Create entries for the modifier applications if this is a ModifiedObject.
			ModifiedObject* modObj = dynamic_object_cast<ModifiedObject>(cmnObject);
			if(modObj != NULL) {
				for(int i=modObj->modifierApplications().size(); i--; ) {
					ModifierApplication* app = modObj->modifierApplications()[i];
				    ModifierStackEntry* entry = new ModifierStackEntry(this, app->modifier());
					entry->addModifierApplication(app);
					stackEntries.push_back(entry);
				}
			}

			// Create an entry for the scene object.
            stackEntries.push_back(new ModifierStackEntry(this, cmnObject));

			// In case there are multiple input slots, determine if they all point to the same object.
			SceneObject* nextObj = NULL;
			for(int i=0; i < cmnObject->inputObjectCount(); i++) {
				if(nextObj == NULL)
					nextObj = cmnObject->inputObject(i);
				else if(nextObj != cmnObject->inputObject(i)) {
					nextObj = NULL;  // The input objects do not match.
					break;
				}
			}
			cmnObject = nextObj;
		}
		while(cmnObject != NULL);
	}
	else {

		// Several different scene nodes are selected that are not instances.
		// But maybe they have been applied the same modifier which should be
		// displayed in the modifier stack.

		// Collect the ModifiedObjects of all selected nodes.
        QVector<ModifiedObject*> modObjs;
		foreach_ref(ObjectNode* objNode, selectedNodes) {
			ModifiedObject* modObj = dynamic_object_cast<ModifiedObject>(objNode->sceneObject());
			if(modObj == NULL) {
				modObjs.clear();
				break;
			}
			modObjs.push_back(modObj);
		}

		if(!modObjs.empty()) {

			// Each of the selected nodes has a ModifiedObject, now try to find
			// identicial modifiers in them.

			bool done = false;
			for(size_t i=0; !done; i++) {
                QVector<ModifierApplication*> apps;
				Q_FOREACH(ModifiedObject* modObj, modObjs) {
					if(modObj->modifierApplications().size() == i) {
						done = true;
						break;
					}
					ModifierApplication* app = modObj->modifierApplications()[modObj->modifierApplications().size() - i - 1];
					if(!apps.empty() && apps.front()->modifier() != app->modifier()) {
						done = true;
						break;
					}
					apps.push_back(app);
				}
				if(!done) {
					ModifierStackEntry* entry = new ModifierStackEntry(this, apps.front()->modifier());
					Q_FOREACH(ModifierApplication* app, apps)
						entry->addModifierApplication(app);
					stackEntries.push_back(entry);
				}
			}
		}
	}

	// The internal list of ModifierStackEntries is now complete.
	// Fill the list model with them.
	int selIndex = 0;
	QVector<ModifierStackEntry*> listBoxEntries;
	foreach_ref(ModifierStackEntry* entry, stackEntries) {
		if(nextObjectToSelect == entry->commonObject())
			selIndex = listBoxEntries.size();
		if(dynamic_object_cast<ModifiedObject>(entry->commonObject())) {
			// Skip ModifiedObject if it is empty and on the top of the stack.
			if(listBoxEntries.empty()) continue;
		}
		listBoxEntries.push_back(entry);
	}
	listModel()->setEntries(listBoxEntries);
	nextObjectToSelect = NULL;
	page->stackBox->setEnabled(!listBoxEntries.empty());

	// Select the first entry in the list or the one given by "nextObjectToSelect".
	if(!listBoxEntries.empty())
		selectionModel()->select(listModel()->index(selIndex), QItemSelectionModel::SelectCurrent|QItemSelectionModel::Clear);

	// Show the properties of the selected object.
	updatePropertiesPanel();
}

/******************************************************************************
* Updates a single entry in the modifier stack list box.
******************************************************************************/
void ModifierStackModel::refreshStackEntry(ModifierStackEntry* entry)
{
	CHECK_OBJECT_POINTER(entry);
	int i = entries.indexOf(entry);
	OVITO_ASSERT(i != -1);
	if(i != -1) {
		dataChanged(index(i), index(i));

		// Also update actions if the changed entry is currently selected.
		QModelIndexList selection = stack()->selectionModel()->selectedRows();
		ModifierStackEntry* selectedEntry = NULL;
		if(!selection.empty()) {
			ModifierStackEntry* selectedEntry = (ModifierStackEntry*)selection.front().data(Qt::UserRole).value<void*>();
			if(selectedEntry == entry) {
				stack()->updateAvailableActions(selectedEntry);
			}
		}
	}
}

/******************************************************************************
* Shows the properties of the selected item in the modifier stack box in the
* properties panel of the page.
******************************************************************************/
void ModifierStack::updatePropertiesPanel()
{
	QModelIndexList selection = selectionModel()->selectedRows();
	ModifierStackEntry* selectedEntry = NULL;
	if(!selection.empty()) {
		selectedEntry = (ModifierStackEntry*)selection.front().data(Qt::UserRole).value<void*>();
		CHECK_OBJECT_POINTER(selectedEntry);
	}

	if(selectedEntry == NULL) {
		page->propertiesPanel->setEditObject(NULL);
		updateAvailableModifiers(NULL);
		updateAvailableActions(NULL);

		// Whenever no object is selected, show the about panel with information about the program.
		BrandingManager::primaryBranding()->showAboutPanel(page->propertiesPanel);
	}
	else {
		// Hide about panel first.
		BrandingManager::primaryBranding()->hideAboutPanel(page->propertiesPanel);

		page->propertiesPanel->setEditObject(selectedEntry->commonObject());
		updateAvailableModifiers(selectedEntry);
		updateAvailableActions(selectedEntry);
	}

	VIEWPORT_MANAGER.updateViewports();
}

/******************************************************************************
* Defines a comparison operator for the modifier list.
* This ensures an alphabetical ordering.
******************************************************************************/
bool ModifierStack::modifierOrdering(const PluginClassDescriptor* a, const PluginClassDescriptor* b)
{
    return QString::compare(a->schematicTitle(), b->schematicTitle(), Qt::CaseInsensitive) < 0;
}

/******************************************************************************
* Defines a comparison operator for the modifier category list.
* This ensures an alphabetical ordering.
******************************************************************************/
bool ModifierStack::modifierCategoryOrdering(const ModifierStack::ModifierCategory& a, const ModifierStack::ModifierCategory& b)
{
    return QString::compare(a.label, b.label, Qt::CaseInsensitive) < 0;
}

/******************************************************************************
* Gathers all defined modifier categories from the plugin manifests.
******************************************************************************/
void ModifierStack::loadModifierCategories()
{
	// Gather category names.
	Q_FOREACH(Plugin* plugin, PLUGIN_MANAGER.plugins()) {
		QDomNodeList list = plugin->manifest().elementsByTagName("Modifier-Category-Definition");
		for(int i = 0; i < list.count(); i++) {
			QDomElement element = list.at(i).toElement();
			ModifierCategory category;
			category.id = element.attribute("Id");
			category.label = element.text();
			_modifierCategories.push_back(category);
		}
	}

	// Sort category list alphabetically.
	std::sort(_modifierCategories.begin(), _modifierCategories.end(), modifierCategoryOrdering);

	// Assign modifiers to categories.
	ModifierCategory nullCategory;
	nullCategory.label = tr("Others");
	Q_FOREACH(PluginClassDescriptor* clazz, page->modifierClasses) {
		QDomElement categoryElement = clazz->getMetaData("Modifier-Category");
		ModifierCategory* category = &nullCategory;
		if(categoryElement.isNull() == false) {
			QString categoryId = categoryElement.attribute("Id");
			for(int i=0; i<_modifierCategories.size(); i++) {
				if(_modifierCategories[i].id == categoryId) {
					category = &_modifierCategories[i];
					break;
				}
			}
		}
		category->modifierClasses.push_back(clazz);
	}
	if(nullCategory.modifierClasses.empty() == false)
		_modifierCategories.push_back(nullCategory);

	// Sort modifier sub-lists alphabetically.
	for(int i=0; i<_modifierCategories.size(); i++) {
		std::sort(_modifierCategories[i].modifierClasses.begin(), _modifierCategories[i].modifierClasses.end(), modifierOrdering);
	}
}

/******************************************************************************
* Updates the list box of modifier classes that can be applied to the current selected
* item in the modifier stack.
******************************************************************************/
void ModifierStack::updateAvailableModifiers(ModifierStackEntry* currentEntry)
{
	page->modifierSelector->clear();
	page->modifierSelector->addItem(tr("Modifier List"));
	page->modifierSelector->addItem("-------------");
	page->modifierSelector->setCurrentIndex(0);

	if(currentEntry == NULL) {
		if(selectedNodes.empty()) {
			// Empty node selection.
			page->modifierSelector->setEnabled(false);
			return;
		}
	}

	QFont categoryFont = page->modifierSelector->font();
	categoryFont.setBold(true);

	Q_FOREACH(const ModifierCategory& category, _modifierCategories) {
		page->modifierSelector->addItem(category.label);
		page->modifierSelector->setItemData(page->modifierSelector->count()-1, categoryFont, Qt::FontRole);
		Q_FOREACH(PluginClassDescriptor* descriptor, category.modifierClasses) {
			page->modifierSelector->addItem("    " + descriptor->schematicTitle(), qVariantFromValue((void*)descriptor));
		}
	}

    page->modifierSelector->setEnabled(true);
    page->modifierSelector->setMaxVisibleItems(page->modifierSelector->count());
}

/******************************************************************************
* Updates the state of the actions that can be invoked on the currently selected
* item in the modifier stack.
******************************************************************************/
void ModifierStack::updateAvailableActions(ModifierStackEntry* currentEntry)
{
	ActionProxy* deleteModifierAction = ACTION_MANAGER.findActionProxy(ACTION_MODIFIER_DELETE);
	ActionProxy* moveModifierUpAction = ACTION_MANAGER.findActionProxy(ACTION_MODIFIER_MOVE_UP);
	ActionProxy* moveModifierDownAction = ACTION_MANAGER.findActionProxy(ACTION_MODIFIER_MOVE_DOWN);
	ActionProxy* toggleModifierStateAction = ACTION_MANAGER.findActionProxy(ACTION_MODIFIER_TOGGLE_STATE);
	Modifier* modifier = currentEntry ? dynamic_object_cast<Modifier>(currentEntry->commonObject()) : NULL;
	if(modifier != NULL) {
		deleteModifierAction->setEnabled(true);
		if(currentEntry->modifierApplications().size() == 1) {
			ModifierApplication* modApp = currentEntry->modifierApplications()[0];
			ModifiedObject* modObj = modApp->modifiedObject();
			if(modObj != NULL) {
				OVITO_ASSERT(modObj->modifierApplications().contains(modApp));
				moveModifierUpAction->setEnabled(modApp != modObj->modifierApplications().back());
				moveModifierDownAction->setEnabled(modApp != modObj->modifierApplications().front());
			}
		}
		else {
			moveModifierUpAction->setEnabled(false);
			moveModifierDownAction->setEnabled(false);
		}
		if(modifier) {
			toggleModifierStateAction->setEnabled(true);
			toggleModifierStateAction->setChecked(modifier->isModifierEnabled() == false);
		}
		else {
			toggleModifierStateAction->setChecked(false);
			toggleModifierStateAction->setEnabled(false);
		}
	}
	else {
		deleteModifierAction->setEnabled(false);
		moveModifierUpAction->setEnabled(false);
		moveModifierDownAction->setEnabled(false);
		toggleModifierStateAction->setChecked(false);
		toggleModifierStateAction->setEnabled(false);
	}
}

/******************************************************************************
* Inserts the given modifier into the modification stack of the selected scene nodes.
******************************************************************************/
void ModifierStack::applyModifier(Modifier* modifier)
{
	// Get the selected stack entry. The new modifier is inserted just behind it.
	QModelIndexList selection = selectionModel()->selectedRows();
	ModifierStackEntry* selEntry = NULL;
	if(!selection.empty()) {
		selEntry = (ModifierStackEntry*)selection.front().data(Qt::UserRole).value<void*>();
		CHECK_OBJECT_POINTER(selEntry);
	}

	// On the next stack update the new modifier should be selected.
	nextObjectToSelect = modifier;

	if(selEntry != NULL) {
		if(dynamic_object_cast<Modifier>(selEntry->commonObject())) {
			Q_FOREACH(ModifierApplication* modApp, selEntry->modifierApplications()) {
				ModifiedObject* modObj = modApp->modifiedObject();
				CHECK_OBJECT_POINTER(modObj);
				modObj->insertModifier(modifier, modObj->modifierApplications().indexOf(modApp)+1);
			}
			return;
		}
		else if(dynamic_object_cast<ModifiedObject>(selEntry->commonObject())) {
			ModifiedObject* modObj = static_object_cast<ModifiedObject>(selEntry->commonObject());
			CHECK_OBJECT_POINTER(modObj);
			modObj->insertModifier(modifier, 0);
			return;
		}
		else {
			OVITO_ASSERT(stackEntries.contains(selEntry));
			int index = stackEntries.indexOf(selEntry);
			if(index != 0) {
				selEntry = stackEntries[index-1];
				if(dynamic_object_cast<ModifiedObject>(selEntry->commonObject())) {
					ModifiedObject* modObj = static_object_cast<ModifiedObject>(selEntry->commonObject());
					CHECK_OBJECT_POINTER(modObj);
					modObj->insertModifier(modifier, 0);
					return;
				}
			}
		}
	}

	// Apply modifier to each node separately.
	foreach_ref(ObjectNode* objNode, selectedNodes)
		objNode->applyModifier(modifier);
}

/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool ModifierStack::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(msg->type() == REFERENCE_FIELD_CHANGED) {
		ObjectNode* targetNode = dynamic_object_cast<ObjectNode>(source);
		if(targetNode) {
			OVITO_ASSERT(selectedNodes.contains(targetNode));
			invalidate();
		}
	}
	return RefMaker::onRefTargetMessage(source, msg);
}

/******************************************************************************
* Returns the data for the QListView widget.
******************************************************************************/
QVariant ModifierStackModel::data(const QModelIndex& index, int role) const
{
	OVITO_ASSERT(index.row() >= 0 && index.row() < (int)entries.size());

	ModifierStackEntry* entry = entries[index.row()];
	CHECK_OBJECT_POINTER(entry);

	if(role == Qt::DisplayRole) {
		if(dynamic_object_cast<ModifiedObject>(entry->commonObject()))
			return qVariantFromValue(QString("----------"));
		else
			return qVariantFromValue(entry->commonObject()->schematicTitle());
	}
	else if(role == Qt::UserRole) {
		return qVariantFromValue((void*)entry);
	}
	else if(role == Qt::DecorationRole) {
		Modifier* modifier = dynamic_object_cast<Modifier>(entry->commonObject());
		if(modifier) {
			if(!modifier->isModifierEnabled()) {
				return qVariantFromValue(modifierDisabledIcon);
			}
			else {
				EvaluationStatus status;
				Q_FOREACH(ModifierApplication* modApp, entry->modifierApplications()) {
					status = modApp->status();
					if(status.type() == EvaluationStatus::EVALUATION_ERROR) break;
				}
				if(status.type() == EvaluationStatus::EVALUATION_SUCCESS) {
					if(status.shortMessage().isEmpty())
						return qVariantFromValue(modifierEnabledIcon);
					else
						return qVariantFromValue(modifierStatusInfoIcon);
				}
				else if(status.type() == EvaluationStatus::EVALUATION_WARNING)
					return qVariantFromValue(modifierStatusWarningIcon);
				else if(status.type() == EvaluationStatus::EVALUATION_ERROR)
					return qVariantFromValue(modifierStatusErrorIcon);
				else
					return qVariantFromValue(modifierEnabledIcon);
			}
		}
	}
	else if(role == Qt::ToolTipRole) {
		Modifier* modifier = dynamic_object_cast<Modifier>(entry->commonObject());
		if(modifier && modifier->isModifierEnabled()) {
			EvaluationStatus status;
			Q_FOREACH(ModifierApplication* modApp, entry->modifierApplications()) {
				status = modApp->status();
				if(status.type() == EvaluationStatus::EVALUATION_ERROR) break;
			}
			if(status.shortMessage().isEmpty() == false)
				return qVariantFromValue(status.shortMessage());
		}
	}


	return QVariant();
}

};
