///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __SLICE_MODIFIER_H
#define __SLICE_MODIFIER_H

#include <core/Core.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/viewport/input/ViewportInputHandler.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/actions/ViewportModeAction.h>
#include <core/actions/ActionProxy.h>

#include <atomviz/AtomViz.h>
#include <atomviz/modifier/AtomsObjectModifierBase.h>
#include <atomviz/utils/AtomPicker.h>

namespace AtomViz {

/******************************************************************************
* The slice modifier rejects every atom on one side of a plane.
******************************************************************************/
class ATOMVIZ_DLLEXPORT SliceModifier : public AtomsObjectModifierBase
{
public:
	/// Default constructor.
	SliceModifier(bool isLoading = false);

	/// Asks the modifier for its validity interval at the given time.
	virtual TimeInterval modifierValidity(TimeTicks time);

	/// Makes the modifier render itself into the viewport.
	/// The viewport transformation is already set up, when this method is called by the
	/// system. The modifier has to be rendered in the local object coordinate system.
	///   time - The time at which to render the modifier.
	///   contextNode - The node context used to render the modifier.
	///   modApp - The modifier application associated with the modifier.
	///   vp - The viewport to render in.
	virtual void renderModifier(TimeTicks time, ObjectNode* contextNode, ModifierApplication* modApp, Viewport* vp);

	/// \brief This virtual method is called by the system when the modifier has been inserted into a ModifiedObject.
	virtual void initializeModifier(ModifiedObject* modObject, ModifierApplication* modApp);

	// Property access functions:

	/// Returns the plane's distance from the origin.
	FloatType distance() const { return _distanceCtrl ? _distanceCtrl->getCurrentValue() : (FloatType)0; }
	/// Sets the plane's distance from the origin.
	void setDistance(FloatType newDistance) { if(_distanceCtrl) _distanceCtrl->setCurrentValue(newDistance); }
	/// Returns the controller for the plane distance.
	FloatController* distanceController() const { return _distanceCtrl; }
	/// Sets the controller for the plane distance.
	void setDistanceController(const FloatController::SmartPtr& ctrl) { _distanceCtrl = ctrl; }

	/// Returns the plane's normal vector.
	Vector3 normal() const { return _normalCtrl ? _normalCtrl->getCurrentValue() : Vector3(0,0,1); }
	/// Sets the plane's distance from the origin.
	void setNormal(const Vector3& newNormal) { if(_normalCtrl) _normalCtrl->setCurrentValue(newNormal); }
	/// Returns the controller for the plane normal.
	VectorController* normalController() const { return _normalCtrl; }
	/// Sets the controller for the plane normal.
	void setVectorController(const VectorController::SmartPtr& ctrl) { _normalCtrl = ctrl; }

	/// Returns the slice width.
	FloatType sliceWidth() const { return _widthCtrl ? _widthCtrl->getCurrentValue() : (FloatType)0; }
	/// Sets the slice width.
	void setSliceWidth(FloatType newWidth) { if(_widthCtrl) _widthCtrl->setCurrentValue(newWidth); }
	/// Returns the controller for the slice width.
	FloatController* sliceWidthController() const { return _widthCtrl; }
	/// Sets the controller for the slice width.
	void setSliceWidthController(const FloatController::SmartPtr& ctrl) { _widthCtrl = ctrl; }

	/// Returns whether the plane's orientation should be flipped.
	bool inverse() const { return _inverse; }
	/// Sets whether the plane's orientation should be flipped.
	void setInverse(bool inverse) { _inverse = inverse; }

	/// Returns whether the atoms are only selected instead of deleted.
	bool createSelection() const { return _createSelection; }
	/// Sets whether the atoms are only selected instead of deleted.
	void setCreateSelection(bool select) { _createSelection = select; }

	/// Returns whether the modifier is only applied to the currently selected atoms.
	bool applyToSelection() const { return _applyToSelection; }
	/// Sets whether the modifier should only be applied to the currently selected atoms.
	void setApplyToSelection(bool flag) { _applyToSelection = flag; }

	/// Returns the slicing plane.
	Plane3 slicingPlane(TimeTicks time, TimeInterval& validityInterval);

public:

	Q_PROPERTY(FloatType distance READ distance WRITE setDistance)
	Q_PROPERTY(Vector3 normal READ normal WRITE setNormal)
	Q_PROPERTY(FloatType sliceWidth READ sliceWidth WRITE setSliceWidth)
	Q_PROPERTY(bool inverse READ inverse WRITE setInverse)
	Q_PROPERTY(bool createSelection READ createSelection WRITE setCreateSelection)
	Q_PROPERTY(bool applyToSelection READ applyToSelection WRITE setApplyToSelection)

protected:

	/// Modifies the atoms object. The time interval passed
	/// to the function is reduced to the interval where the modified object is valid/constant.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// Performs the actual rejection of atoms.
	size_t filterAtoms(dynamic_bitset<>& mask, TimeTicks time, TimeInterval& validityInterval);

	/// Renders a plane in the viewport with the given color.
	void renderPlane(Viewport* vp, const Plane3& plane, const Box3& box, const Color& color) const;

	/// Computes the intersection lines of a plane and a quad.
	void planeQuadIntersesction(const Ray3& r1, const Ray3& r2, const Ray3& r3, const Ray3& r4, const Plane3& plane, QVector<Point3>& lines) const;

	/// This controller stores the normal of the slicing plane.
	ReferenceField<VectorController> _normalCtrl;

	/// This controller stores the distance of the slicing plane from the origin.
	ReferenceField<FloatController> _distanceCtrl;

	/// Controls the slice width.
	ReferenceField<FloatController> _widthCtrl;

	/// Controls whether the atoms should only be selected instead of deleted.
	PropertyField<bool> _createSelection;

	/// Controls whether the selection/plane orientation should be inverted.
	PropertyField<bool> _inverse;

	/// Controls whether the modifier should only be applied to the currently selected atoms.
	PropertyField<bool> _applyToSelection;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(SliceModifier)
	DECLARE_REFERENCE_FIELD(_normalCtrl)
	DECLARE_REFERENCE_FIELD(_distanceCtrl)
	DECLARE_REFERENCE_FIELD(_widthCtrl)
	DECLARE_PROPERTY_FIELD(_createSelection)
	DECLARE_PROPERTY_FIELD(_inverse)
	DECLARE_PROPERTY_FIELD(_applyToSelection)
};

/******************************************************************************
* The viewport input mode that lets the user select three atoms
* to define the slicing plane.
******************************************************************************/
class ATOMVIZ_DLLEXPORT PickAtomPlaneInputMode : public ViewportInputHandler, private AtomPicker
{
public:

	/// Constructor.
	PickAtomPlaneInputMode() {}

	/// Returns the activation behaviour of this input handler.
	virtual InputHandlerType handlerActivationType() {
		return ViewportInputHandler::NORMAL;
	}

	/// Handles the mouse down events for a Viewport.
	virtual void onMouseDown(Viewport& vp, QMouseEvent* event);

	/// \brief Lets the input mode render its overlay content in a viewport.
	virtual void renderOverlay(Viewport* vp, bool isActive);

	/// \brief Indicates whether this input mode renders into the viewports.
	virtual bool hasOverlay() { return true; }

protected:

	/// \brief This is called by the system after the input handler has become the active handler.
	virtual void onActivated();

	/// \brief This is called by the system after the input handler is no longer the active handler.
	virtual void onDeactivated();

private:

	/// \brief Aligns the modifier's slicing plane to the three selected atoms.
	void alignPlane(SliceModifier* mod);

	/// The list of atoms picked by the user so far.
	QVector<PickAtomResult> pickedAtoms;
};

/******************************************************************************
* A properties editor for the SliceModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT SliceModifierEditor : public AtomsObjectModifierEditorBase
{
public:

	virtual ~SliceModifierEditor() {
		// Deactivate the editor's input mode.
		VIEWPORT_INPUT_MANAGER.removeInputHandler(pickAtomPlaneInputMode.get());
	}

protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Aligns the slicing plane to the viewing direction.
	void onAlignPlaneToView();

	/// Aligns the current viewing direction to the slicing plane.
	void onAlignViewToPlane();

	/// Aligns the normal of the slicing plane with the X, Y, or Z axis.
	void onXYZNormal(const QString& link);

private:

	PickAtomPlaneInputMode::SmartPtr pickAtomPlaneInputMode;
	ViewportModeAction::SmartPtr pickAtomPlaneInputModeAction;
	ActionProxy* pickAtomPlaneInputModeActionProxy;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(SliceModifierEditor)
};

};	// End of namespace AtomViz

#endif // __SLICE_MODIFIER_H
