//                                               -*- C++ -*-
/**
 *  @file  NatafIndependentCopulaGradient.cxx
 *  @brief Class for the Nataf transformation evaluation for independent
 *
 *  Copyright 2005-2015 Airbus-EDF-IMACS-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  @author schueller
 *  @date   2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012)
 */
#include <cmath>
#include "NatafIndependentCopulaGradient.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS

/*
 * @class NatafIndependentCopulaGradient
 *
 * This class offers an interface for the Nataf function for independent copula
 */


CLASSNAMEINIT(NatafIndependentCopulaGradient);

static Factory<NatafIndependentCopulaGradient> RegisteredFactory("NatafIndependentCopulaGradient");

/* Default constructor */
NatafIndependentCopulaGradient::NatafIndependentCopulaGradient()
  : NumericalMathGradientImplementation()
  , dimension_()
{
  // Nothing to do
}

/* Parameter constructor */
NatafIndependentCopulaGradient::NatafIndependentCopulaGradient(const UnsignedInteger dimension)
  : NumericalMathGradientImplementation()
  , dimension_(dimension)
{
  // Nothing to do
}

/* Virtual constructor */
NatafIndependentCopulaGradient * NatafIndependentCopulaGradient::clone() const
{
  return new NatafIndependentCopulaGradient(*this);
}

/* String converter */
String NatafIndependentCopulaGradient::__repr__() const
{
  OSS oss;
  oss << "class=" << NatafIndependentCopulaGradient::GetClassName()
      << " dimension=" << dimension_;

  return oss;
}

/*
 * Evaluation
 * The Nataf transform T reads:
 * Ti(xi) = Q(xi), where Q = Phi^{-1} and Phi is the CDF of the standard normal distribution
 * We have:
 * Jij = dTi/dxj = Q'(xi) if i = j
 *               = 0 else
 */
Matrix NatafIndependentCopulaGradient::gradient(const NumericalPoint & inP) const
{
  Matrix result(dimension_, dimension_);
  for (UnsignedInteger i = 0; i < dimension_; ++i)
  {
    const NumericalScalar x(inP[i]);
    if ((x < 0.0) || (x > 1.0)) throw InvalidArgumentException(HERE) << "Error: cannot evaluate the NatafIndependentCopulaGradient if all the components are not in [0, 1], here in=" << inP;
    // q = Normal(0,1).computeQuantile(x)
    const NumericalScalar q(DistFunc::qNormal(x));
    // 2.506628274631000502415765 = sqrt(2*Pi)
    result(i, i) = 2.506628274631000502415765 * exp(0.5 * q * q);
  }
  return result;
}

/* Accessor for input point dimension */
UnsignedInteger NatafIndependentCopulaGradient::getInputDimension() const
{
  return dimension_;
}

/* Accessor for output point dimension */
UnsignedInteger NatafIndependentCopulaGradient::getOutputDimension() const
{
  return dimension_;
}

/* Method save() stores the object through the StorageManager */
void NatafIndependentCopulaGradient::save(Advocate & adv) const
{
  NumericalMathGradientImplementation::save(adv);
  adv.saveAttribute( "dimension_", dimension_ );
}

/* Method load() reloads the object from the StorageManager */
void NatafIndependentCopulaGradient::load(Advocate & adv)
{
  NumericalMathGradientImplementation::load(adv);
  adv.loadAttribute( "dimension_", dimension_ );
}

END_NAMESPACE_OPENTURNS
