!***********************************************************************
! This file is part of OpenMolcas.                                     *
!                                                                      *
! OpenMolcas is free software; you can redistribute it and/or modify   *
! it under the terms of the GNU Lesser General Public License, v. 2.1. *
! OpenMolcas is distributed in the hope that it will be useful, but it *
! is provided "as is" and without any express or implied warranties.   *
! For more details see the full text of the license in the file        *
! LICENSE or in <http://www.gnu.org/licenses/>.                        *
!                                                                      *
! Copyright (C) 2006, Luca De Vico                                     *
!               Valera Veryazov                                        *
!***********************************************************************
!  NAT_BOND_ORDER
!
!> @brief
!>   Compute Natural Bond Order
!> @author Luca De Vico
!> @modified_by V. Veryazov
!>
!> @details
!> For a given set of atomic orbitals compute the
!> Natural Bond Order (NBO) as described in
!> Jensen's "Introduction to Computational Chemistry" \cite Jensen-1999.
!> The atomic orbitals are taken as generated by LoProp,
!> in order to be localised. In the future, hopefully, it
!> will consider also triatomic bonds
!> and then put everything into a molden file and/or gv.
!> Treatment of UHF should be included.
!>
!> - \p iCase = ``0``: First UHF run, no print just dump
!> - \p iCase = ``1``: Second UHF run, final print
!> - \p iCase = ``2``: Normal run, print
!>
!> @param[in] NSYM  Number of irreducible representations
!> @param[in] NBAS  Number of basis functions per irred. rep.
!> @param[in] BNAME Center and function type label
!> @param[in] iCase Type of run
!***********************************************************************

subroutine NAT_BOND_ORDER(NSYM,NBAS,BNAME,iCase)
!***********************************************************************
!                                                                      *
!     Author: Luca De Vico, University of Lund, 2006                   *
!                                                                      *
!***********************************************************************
!***********************************************************************
!                                                                      *
! Subroutine Scheme                                                    *
!      Initialization                                                  *
!      Retrieve of the LoProp density matrix from the RunFile          *
!      Loop over atoms i                                               *
!        generate submatrix SUBDNAO from DS(i)                         *
!        if (eigenvalues > thr_CO) => core orbital                     *
!           depletion of orbital                                       *
!        if (eigenvalues > thr_LP) => lone pair orbital                *
!           depletion of orbital                                       *
!      Double loop over atoms i,j with i<j                             *
!        generate submatrix SUBDNAO from DS(i) and DS(j)               *
!        diagonalize SUBDNAO                                           *
!        if (eigenvalues > threshold) => NBO                           *
!      if NBO-OCCN + CORE-OCCN + LP-OCCN << electrons                  *
!        search for single occupied orbitals                           *
!        same as core orbitals, with thr_SO                            *
!        or                                                            *
!        (search for three-center bonds)                               *
!      print out                                                       *
!                                                                      *
!***********************************************************************

use stdalloc, only: mma_allocate, mma_deallocate
use Constants, only: Zero, One, Half
use Definitions, only: wp, iwp, u6

implicit none
#include "Molcas.fh"
integer(kind=iwp), intent(in) :: NSYM, NBAS(*), iCase
character(len=LenIn8), intent(in) :: BNAME(*)
integer(kind=iwp) :: AtomA, AtomB, I, i_Component, i_Opt, i_Rc, i_SymLbl, iANr, IAtom, IB, iBlo, iBondNumb, ICNT, iDummy(1), &
                     iElToAsgn, iErr, iHalf, IMN, iNoNuc, iOdd, iPL, IS, isAtom, iSElem, ISING, iSingNumb, isThereAtLeastABond, &
                     iSum, iSyLbl, ISYM, iTriplBondNumb, iTry, ix_Single, ix_Triple, J, jANr, JAtom, k, KAtom, mSub, MY, NB, &
                     nBas2, nBasAtoms, nBasAtomsA, nBasAtomsB, nBasAtomsC, nBasMax, NBAST, nDens, nNUC, NPBonds, nScr, nSub, NY, &
                     tNUC, tRealNUC
real(kind=wp) :: coeff, covij, Covrad1, Covrad2, DET, Dummy(1), ElecNonAssgn, rij, rij2, thr_BO, thr_CO, thr_Decr, thr_DecrStep, &
                 thr_Diff, thr_Dummy, thr_Dummy1, thr_Dummy2, thr_LP, thr_LP_Orig, thr_MIN, thr_NA, thr_Orig, thr_SO, TotBondElec, &
                 TotCoreElec, TotEl, TotLoneElec, TotSingleElec, TotTriplBondElec, x, y, z
logical(kind=iwp) :: Exists, SearchedSingle, SearchedTriple
character(len=LenIn4) :: Atom_A, Atom_B, Atom_C, Atom_D
character(len=49) :: Label3ANoSym, Label3ASym
character(len=36) :: LabelNoSym, LabelSym
character(len=16) :: Label
integer(kind=iwp), allocatable :: ANr(:), BondAtomA(:), BondAtomB(:), center(:), NBFpA(:), SingElAtom(:), SubDNAOindex(:,:,:), &
                                  TriplAtomA(:), TriplAtomB(:), TriplAtomC(:)
real(kind=wp), allocatable :: Allc(:,:), Bonds(:), CM(:), DNAO(:,:), DS(:,:), DS_orig(:,:), DS_tmp(:,:), P(:,:), PInv(:,:), &
                              S(:,:), S_blo(:), S_orig(:), S_tmp(:,:), Scr(:), SingEl(:), SubDNAO(:,:), SubIVal(:), SubVal(:), &
                              SubVec(:,:), Tmp(:), Tripl(:)
character(len=LenIn4), allocatable :: LblCnt4(:)
character(len=LenIn), allocatable :: CNAME(:), TLbl(:) !, LblCnt(:)
character, parameter :: cSign = '-'
integer(kind=iwp), external :: iPrintLevel
real(kind=wp), external :: Covrad
logical(kind=iwp), external :: Reduce_Prt

!define _DEBUGPRINT_
!                                                                      *
!***********************************************************************
!                                                                      *
iPL = iPrintLevel(-1)
if (Reduce_Prt() .and. (iPL < 3)) iPL = 0
!                                                                      *
!***********************************************************************
!                                                                      *
if (iPL < 2) return
!                                                                      *
!***********************************************************************
!                                                                      *
! Since LoProp gives a complete density matrix after the second
! run in UHF case, we simply skip the first run
if (iCase == 0) return

!----------------------------------------------------------------------*
! Parameters                                                           *
!----------------------------------------------------------------------*

! Core Orbitals threshold, default 1.999
thr_CO = 1.999_wp
! Lone Pairs threshold .le. Core Orbitals threshold, default 1.90
!thr_LP = 1.90_wp
!thr_LP = 1.85_wp
thr_LP = 1.80_wp
thr_LP_Orig = thr_LP
! Bond occupation threshold .le. Lone Pairs Orbitals threshold
! Guessed to 1.90. Minimum possible threshold thr_MIN guessed to 1.50
thr_BO = 1.999_wp
thr_Orig = thr_BO
thr_MIN = 1.50_wp
! Non-assigned threshold, guessed to 0.90
thr_NA = 0.90_wp
! Single occupied orbitals threshold, guessed to 0.999
thr_SO = 0.90_wp
! Not so Dummy thresholds
thr_Dummy = 4.0_wp
thr_Dummy1 = 2.02_wp
thr_Dummy2 = 1.02_wp

! Decreasing threshold and decrease step
thr_DecrStep = 0.01_wp
thr_Decr = Zero

SearchedSingle = .false.
SearchedTriple = .false.
ix_Triple = 0
ix_Single = 0

!----------------------------------------------------------------------*
! GET THE TOTAL NUMBER OF BASIS FUNCTIONS AND CHECK LIMITS             *
!----------------------------------------------------------------------*

NBAST = 0
do I=1,NSYM
  NBAST = NBAST+NBAS(I)
end do

!----------------------------------------------------------------------*
! Find the list of unique center labels                                *
!----------------------------------------------------------------------*

call Get_iScalar('Unique atoms',nNuc)
call mma_allocate(CNAME,nNuc,label='CNAME')
call Get_cArray('Unique Atom Names',CNAME,LenIn*nNuc)

!----------------------------------------------------------------------*
! Find the center label for each basis function                        *
!----------------------------------------------------------------------*

do I=1,NBAST
  ICNT = -1
  do J=1,NNUC
    if (BNAME(I)(1:LenIn) == CNAME(J)) ICNT = J
  end do
  if (ICNT < 0) then
    write(u6,*)
    write(u6,*) 'NBO analysis not implemented with pseudo atoms'
    write(u6,*) '(yet). Continuing normal execution.'
    write(u6,*)
    call mma_deallocate(CNAME)
    return
  end if
end do

!----------------------------------------------------------------------*
! Get the total number of atoms tNUC, regardless of symmetry           *
!----------------------------------------------------------------------*

call Get_iScalar('LP_nCenter',tNUC)
if (tNUC == 1) then
  call mma_deallocate(CNAME)
  return
end if

!----------------------------------------------------------------------*
! Find the center label for each atom, regardless of symmetry          *
!----------------------------------------------------------------------*

! Just atom label. It's a double of the next one,
! but someone could find it usefull in the future

call mma_allocate(TLbl,tNUC,label='Tlbl')
call Get_LblCnt_All(TLbl)

! Atom label plus symmetry generator

call mma_allocate(LblCnt4,tNUC,label='LblCnt4')
call Get_cArray('LP_L',LblCnt4,LenIn4*tNUC)
!do i=1,tNUC
!  LblCnt(i)(1:LenIn) = LblCnt4(i)(1:LenIn)
!end do

!----------------------------------------------------------------------*
! Check for extra atoms in tnuc, caused by Mr. Choleski                *
!----------------------------------------------------------------------*

tRealNUC = tNUC
if (tNUC > NNUC) then
  iNoNuc = 0
  do i=1,tNUC
    iTry = 0
    do j=1,NNuc
      if (Tlbl(i) == CNAME(J)) iTry = 1
    end do
    if (iTry == 0) iNoNuc = iNoNuc+1
  end do
  tRealNUC = tNUC-iNoNuc
end if

call mma_deallocate(TLbl)
call mma_deallocate(CNAME)

!----------------------------------------------------------------------*
! Get the coordinates of the atoms                                     *
!----------------------------------------------------------------------*

call mma_allocate(Allc,3,tRealNUC,label='Allc')
Allc(:,:) = Zero

call Get_Coord_All(Allc,tRealNUC)

#ifdef _DEBUGPRINT_
call RecPrt('Coordinates',' ',Allc,3,tRealNUC)
#endif

!----------------------------------------------------------------------*
! Get the atoms numbers                                                *
!----------------------------------------------------------------------*

call mma_allocate(CM,tRealNUC,label='CM')
call Get_Nuc_Charge_All(CM,tRealNUC)
#ifdef _DEBUGPRINT_
call RecPrt(' Charges',' ',CM,tRealNUC,1)
#endif

! Transform charges to masses (C=12)

call mma_allocate(ANr,tRealNUC,label='ANr')
do isAtom=1,tRealNUC
  ANr(isAtom) = int(CM(isAtom))
end do

#ifdef _DEBUGPRINT_
call ivcPrt(' Atoms',' ',ANr,tRealNUC)
#endif

call mma_deallocate(CM)

!----------------------------------------------------------------------*
! Initialize bond order vectors                                        *
! + 10 added to stay on the safe side                                  *
!----------------------------------------------------------------------*

NPBonds = tRealNUC*(tRealNUC-1)/2+10
call mma_allocate(Bonds,NPBonds,label='Bonds')
call mma_allocate(BondAtomA,NPBonds,label='BondAtomA')
call mma_allocate(BondAtomB,NPBonds,label='BondAtomB')

Bonds(:) = Zero
BondAtomA(:) = 0
BondAtomB(:) = 0

!----------------------------------------------------------------------*
! In case of symmetry we need the desymmetrization matrix              *
!----------------------------------------------------------------------*

if (nSym > 1) then

  call mma_allocate(P,NBAST,NBAST,label='P')
  call mma_allocate(PInv,NBAST,NBAST,label='PInv')

  call Get_dArray('SM',P,NBAST*NBAST)
# ifdef _DEBUGPRINT_
  call RecPrt('SM',' ',P,NBAST,NBAST)
# endif
  call MINV(P,PInv,ISING,DET,NBAST)
# ifdef _DEBUGPRINT_
  call RecPrt('SMInv',' ',PInv,NBAST,NBAST)
# endif
  call DGeTMi(PInv,NBAST,NBAST)
end if

!----------------------------------------------------------------------*
! Pick up index array of which center a basis function belongs to      *
!----------------------------------------------------------------------*

call mma_allocate(center,NBAST,label='center')
call Get_iArray('Center Index',center,NBAST)

#ifdef _DEBUGPRINT_
write(u6,*) 'center ='
do I=1,NBAST
  write(u6,*) center(I)
end do
#endif

!----------------------------------------------------------------------*
! Get the Number of Basis Functions per Atom                           *
!----------------------------------------------------------------------*

call mma_allocate(NBFpA,tNUC,label='NBFpA')

NBFpA(:) = 0
do I=1,NBAST
  NBFpA(center(I)) = NBFpA(center(I))+1
end do

#ifdef _DEBUGPRINT_
write(u6,*) 'number of basis per atom NBFpA ='
do I=1,tNUC
  write(u6,*) NBFpA(I)
end do
#endif

!----------------------------------------------------------------------*
! Allocation and Initialization of matrices                            *
!----------------------------------------------------------------------*

call mma_allocate(S,NBAST,NBAST,label='S')
call mma_allocate(S_tmp,NBAST,NBAST,label='S_tmp')

S(:,:) = Zero
S_tmp(:,:) = Zero

nBas2 = 0
do I=1,nsym
  nBas2 = nBas2+nBas(i)*nBas(i)
end do

if (nSym > 1) then
  call mma_allocate(S_blo,nBas2,label='S_blo')

  S_blo(:) = Zero
end if

iSElem = 0
do ISYM=1,NSYM
  iSElem = iSElem+(NBAS(ISYM)*(NBAS(ISYM)+1)/2)
end do
call mma_allocate(S_orig,iSElem,label='S_orig')
S_orig(:) = Zero

!----------------------------------------------------------------------*
! Read overlap matrix SMAT                                             *
!----------------------------------------------------------------------*

i_Rc = 0
i_Opt = 6
i_Component = 1
i_SymLbl = 1
call RdOne(i_Rc,i_Opt,'Mltpl  0',i_Component,S_orig,i_SymLbl)
if (i_Rc /= 0) then
  write(u6,*) 'NBO Error: Could not read overlaps from ONEINT.'
  call Abend()
end if

#ifdef _DEBUGPRINT_
write(u6,*)
write(u6,*) 'Original Overlap Matrix'
write(u6,'(2X,8F12.8)') (S_orig(J),J=1,iSElem)
#endif

!----------------------------------------------------------------------*
! Overlap matrix is retrieved in matrix form                           *
!----------------------------------------------------------------------*

IB = 0
IS = 1
do ISYM=1,NSYM
  NB = NBAS(ISYM)
  if (NB /= 0) then
    IMN = 0
    do MY=1,NB
      !if (ICNT(IB+MY) <= 0) cycle    ! skip pseudo center
      do NY=1,MY
        !if (ICNT(IB+NY) <= 0) cycle  ! skip pseudo center
        !  Save the Overlap matrix element (my,ny) and (ny,my) in S_orig
        S_tmp(MY+IB,NY+IB) = S_orig(IMN+IS)
        S_tmp(NY+IB,MY+IB) = S_orig(IMN+IS)
        IMN = IMN+1
      end do
    end do
    IB = IB+NB
    IS = IS+(NB+NB**2)/2
  end if
end do

#ifdef _DEBUGPRINT_
write(u6,*)
write(u6,*) 'Before desymmetrization'
call RecPrt('Overlap Matrix = ',' ',S_tmp,NBAST,NBAST)
#endif

!----------------------------------------------------------------------*
! In case of symmetry, we desymmetrize S through S_blo                 *
!----------------------------------------------------------------------*

if (nSym > 1) then
  iBlo = 1
  iSum = 0
  do i=1,NSYM
    if (nbas(i) /= 0) then
      do j=1,nbas(i)
        do k=1,nbas(i)
          S_blo(iBlo) = S_tmp(iSum+k,iSum+j)
          iBlo = iBlo+1
        end do
      end do
      iSum = iSum+nbas(i)
    end if
  end do

# ifdef _DEBUGPRINT_
  write(u6,*) 'S_blo = '
  do i=1,nBas2
    write(u6,*) S_blo(I)
  end do
# endif

  nBasMax = 0
  do i=1,nSym
    nBasMax = max(nBasMax,nBas(i))
  end do

# ifdef _DEBUGPRINT_
  write(u6,*) 'nBasMax = ',nBasMax
# endif

  nScr = nBasMax*NBAST
  iSyLbl = 1

  call mma_allocate(Scr,nScr,label='Scr')
  call Desymmetrize(S_blo,nBas2,Scr,nScr,S,nBas,NBAST,PInv,nSym,iSyLbl)
  call mma_deallocate(Scr)

!----------------------------------------------------------------------*
! Otherwise we simply copy S_tmp into S                                *
!----------------------------------------------------------------------*

else
  S(:,:) = S_tmp(:,:)
end if

#ifdef _DEBUGPRINT_
write(u6,*)
write(u6,*) 'After desymmetrization'
call RecPrt('S Matrix = ',' ',S,NBAST,NBAST)
#endif

!----------------------------------------------------------------------*
! Some deallocation                                                    *
!----------------------------------------------------------------------*

call mma_deallocate(S_tmp)
call mma_deallocate(S_orig)
if (nSym > 1) then
  call mma_deallocate(P)
  call mma_deallocate(PInv)
  call mma_deallocate(S_blo)
end if

!----------------------------------------------------------------------*
! Retrieve the density matrix as produced by LoProp                    *
!----------------------------------------------------------------------*

call mma_allocate(DNAO,NBAST,NBAST,label='DNAO')
DNAO(:,:) = Zero

write(Label,'(A,I1)') 'LoProp Dens ',0
call qpg_dArray(Label,Exists,nDens)
if ((.not. Exists) .or. (nDens == 0)) then
  call SysAbendMsg('get_density_matrix','Could not locate:',Label)
end if
call mma_allocate(Tmp,nDens,label='Tmp')
Tmp(:) = Zero

call Get_dArray(Label,Tmp,nDens)

#ifdef _DEBUGPRINT_
call TriPrt('Density Matrix',' ',Tmp,NBAST)
#endif

! Square the density matrix

Tmp(:) = Half*Tmp(:)

call Square(Tmp,DNAO,1,NBAST,NBAST)

call mma_deallocate(Tmp)

#ifdef _DEBUGPRINT_
call RecPrt('Loprop D Matrix squared = ',' ',DNAO,NBAST,NBAST)

E = Zero
do I=1,NBAST
  do J=1,NBAST
    E = E+DNAO(I,J)*S(I,J)
  end do
end do
write(u6,*)
write(u6,*) 'Number of electrons as sum of D and S elements = ',E
#endif

!----------------------------------------------------------------------*
! DS matrix                                                            *
!----------------------------------------------------------------------*

call mma_allocate(DS,NBAST,NBAST,label='DS')
call DGEMM_('N','N',NBAST,NBAST,NBAST,One,DNAO,NBAST,S,NBAST,Zero,DS,NBAST)

#ifdef _DEBUGPRINT_
call RecPrt('DS-NAO Matrix = ',' ',DS,NBAST,NBAST)
E = Zero
do I=1,NBAST
  E = E+DS(I,I)
end do
write(u6,*)
write(u6,*) 'Number of electrons as sum of the DS diagonal = ',E
#endif

!----------------------------------------------------------------------*
! Some more deallocation and variables initialization                  *
!----------------------------------------------------------------------*

call mma_deallocate(S)
call mma_deallocate(DNAO)

call mma_allocate(DS_orig,NBAST,NBAST,label='DS_orig')
DS_orig(:,:) = DS(:,:)

isThereAtLeastABond = 0

!----------------------------------------------------------------------*
do

  TotCoreElec = Zero
  TotLoneElec = Zero
  TotBondElec = Zero
  TotSingleElec = Zero
  TotTriplBondElec = Zero
  iBondNumb = 0
  iSingNumb = 0
  iTriplBondNumb = 0

  !--------------------------------------------------------------------*
  ! How many electrons we have                                         *
  !--------------------------------------------------------------------*

  TotEl = Zero
  do I=1,NBAST
    TotEl = TotEl+DS(I,I)
  end do

# ifdef _DEBUGPRINT_
  write(u6,*)
  write(u6,*) 'Number of electrons = ',TotEl
# endif

  !--------------------------------------------------------------------*
  ! Depletion of core orbitals. First we build and diagonalize         *
  ! a one center matrix of pre-NBO from the DS matrix and then         *
  ! we eliminate those contributions with eigenvalues > thr_CO from    *
  ! the DNAO density matrix. DS = DS - eival * eivect * eivect T       *
  !--------------------------------------------------------------------*

  do IAtom=1,tRealNUC

    nBasAtoms = NBFpA(IAtom)

    ! extraction of sub matrix from DNAO
    ! SubDNAO -> sub matrix
    ! SubVec  -> will contain eigen vectors
    ! SubVal  -> will contain eigen values, real part
    ! SubIVal -> will contain eigen values, imaginary part

    call mma_allocate(SubDNAO,nBasAtoms,nBasAtoms,label='SubDNAO')
    call mma_allocate(SubDNAOindex,2,nBasAtoms,nBasAtoms,label='SubDNAOindex')
    call mma_allocate(SubVec,nBasAtoms,nBasAtoms,label='SubVec')
    call mma_allocate(SubVal,nBasAtoms,label='SubVal')
    call mma_allocate(SubIVal,nBasAtoms,label='SubIVal')

    mSub = 0
    do MY=1,NBAST
      AtomA = center(MY)
      !if (ICNT(MY) <= 0) cycle ! skip pseudo center
      if (AtomA /= IAtom) cycle ! we want just one atom a time
      mSub = mSub+1
      nSub = 0
      do NY=1,NBAST
        AtomB = center(NY)
        !if (ICNT(NY) <= 0) cycle ! skip pseudo center
        if (AtomB /= IAtom) cycle ! they have to be the same atom
        nSub = nSub+1

        SubDNAO(nSub,mSub) = DS(NY,MY)
        SubDNAOindex(:,nSub,mSub) = [NY,MY]
      end do
    end do

#   ifdef _DEBUGPRINT_
    write(u6,*)
    call RecPrt('SubDNAO Matrix = ',' ',SubDNAO,nBasAtoms,nBasAtoms)
    write(u6,*)
    call iVcPrt('SubDNAO index Matrix = ',' ',SubDNAOindex,2*nBasAtoms*nBasAtoms)
    write(u6,*) 'SubDNAO diagonal elements'
    do I=1,nBasAtoms
      write(u6,*) SubDNAO(I,I)
    end do

    !write(u6,*)
    !write(u6,*) 'SubDS=',DDot_(nBasAtoms*nBasAtoms,SubDNAO,1,SubDNAO,1),DDot_(nBasAtoms*nBasAtoms,SubDNAO,1,One,0)
#   endif

    ! Diagonalization

    call xEigen(1,nBasAtoms,nBasAtoms,SubDNAO,SubVal,SubIVal,SubVec,iErr)

    if (iErr /= 0) then
      call Error()
      return
    end if

#   ifdef _DEBUGPRINT_
    write(u6,*)
    write(u6,*) 'One atom submatrix diagonalization'
    call RecPrt('Eigen vectors Matrix = ',' ',SubVec,nBasAtoms,nBasAtoms)
    write(u6,*)
    call RecPrt('Eigen values, real = ',' ',SubVal,nBasAtoms,1)
    write(u6,*)
    call RecPrt('Eigen values, imaginary = ',' ',SubIVal,nBasAtoms,1)

#   endif

    call Seek_n_Destroy(nBasAtoms,SubVal,SubVec,nBast,thr_CO,thr_Dummy,TotCoreElec,SubDNAOindex,DS,0,iDummy,iDummy,iDummy(1), &
                        IAtom,IAtom,Dummy,iDummy,IAtom)

    !------------------------------------------------------------------*
    ! Depletion of lone pair orbitals. We use the already diagonalized *
    ! one center matrix of pre-NBO from the DS matrix and then we      *
    ! eliminate those contributions with thr_LP <= eigenval <= thr_CO  *
    ! from the DNAO density matrix. DS =DS - eival * eivect * eivect T *
    !------------------------------------------------------------------*

    call Seek_n_Destroy(nBasAtoms,SubVal,SubVec,nBast,thr_LP,thr_CO,TotLoneElec,SubDNAOindex,DS,0,iDummy,iDummy,iDummy(1),IAtom, &
                        IAtom,Dummy,iDummy,IAtom)

    call mma_deallocate(SubDNAO)
    call mma_deallocate(SubDNAOindex)
    call mma_deallocate(SubVec)
    call mma_deallocate(SubVal)
    call mma_deallocate(SubIVal)
  end do

# ifdef _DEBUGPRINT_
  E = Zero
  do I=1,NBAST
    E = E+DS(I,I)
  end do
  write(u6,*)
  write(u6,*) 'Number of electrons as sum of the DS diagonal = ',E
# endif

  !--------------------------------------------------------------------*
  ! Now for the real thing: NBO generation                             *
  !--------------------------------------------------------------------*

  !--------------------------------------------------------------------*
  do

    !------------------------------------------------------------------*
    ! Copy of DS matrix for the depletion                              *
    !------------------------------------------------------------------*

    call mma_allocate(DS_tmp,NBAST,NBAST,label='DS_tmp')
    DS_tmp(:,:) = DS(:,:)

    !------------------------------------------------------------------*
    ! 2 atoms loop                                                     *
    !------------------------------------------------------------------*

    do IAtom=1,tRealNUC-1
      do JAtom=IAtom+1,tRealNUC

        ! If the two atoms are far from each other we do not consider them.
        ! Distance checked against covalent radius. The threshold is used
        ! to tune the acceptance coefficient

        x = Allc(1,IAtom)-Allc(1,JAtom)
        y = Allc(2,IAtom)-Allc(2,JAtom)
        z = Allc(3,IAtom)-Allc(3,JAtom)
        rij2 = x**2+y**2+z**2
        rij = sqrt(rij2)

        ! To avoid warnings for Atomic Nr > 86

        iANr = ANr(IAtom)
        if (iANr > 86) then
          Covrad1 = 2.70_wp
        else
          Covrad1 = Covrad(iANr)
        end if

        jANr = ANr(JAtom)
        if (jANr > 86) then
          Covrad2 = 2.70_wp
        else
          Covrad2 = Covrad(jANr)
        end if

        covij = Covrad1+Covrad2

        thr_Diff = thr_Orig-thr_BO
        coeff = 1.25_wp+thr_Diff

        if (rij > (coeff*covij)) cycle

        isThereAtLeastABond = 1

#       ifdef _DEBUGPRINT_
        write(u6,*)
        write(u6,*) 'Good rij = ',rij
        write(u6,*) 'covalent = ',covij
#       endif

        nBasAtomsA = NBFpA(IAtom)
        nBasAtomsB = NBFpA(JAtom)

        nBasAtoms = nBasAtomsA+nBasAtomsB

        ! extraction of sub matrix from DNAO
        ! SubDNAO -> sub matrix
        ! SubVec  -> will contain eigen vectors
        ! SubVal  -> will contain eigen values, real part
        ! SubIVal -> will contain eigen values, imaginary part

        call mma_allocate(SubDNAO,nBasAtoms,nBasAtoms,label='SubDNAO')
        call mma_allocate(SubDNAOindex,2,nBasAtoms,nBasAtoms,label='SubDNAOindex')
        call mma_allocate(SubVec,nBasAtoms,nBasAtoms,label='SubVec')
        call mma_allocate(SubVal,nBasAtoms,label='SubVal')
        call mma_allocate(SubIVal,nBasAtoms,label='SubIVal')

        mSub = 0
        do MY=1,NBAST
          AtomA = center(MY)
          !if (ICNT(MY) <= 0) cycle   ! skip pseudo center
          if ((AtomA /= IAtom) .and. (AtomA /= JAtom)) cycle
          mSub = mSub+1
          nSub = 0
          do NY=1,NBAST
            AtomB = center(NY)
            !if (ICNT(NY) <= 0) cycle ! skip pseudo center
            if ((AtomB /= IAtom) .and. (AtomB /= JAtom)) cycle
            nSub = nSub+1

            SubDNAO(nSub,mSub) = DS(NY,MY)
            SubDNAOindex(:,nSub,mSub) = [NY,MY]
          end do
        end do

#       ifdef _DEBUGPRINT_
        write(u6,*)
        call RecPrt('SubDNAO Matrix = ',' ',SubDNAO,nBasAtoms,nBasAtoms)
#       endif

        ! Diagonalization

        call xEigen(1,nBasAtoms,nBasAtoms,SubDNAO,SubVal,SubIVal,SubVec,iErr)

        if (iErr /= 0) then
          call Error()
          return
        end if

#       ifdef _DEBUGPRINT_
        write(u6,*)
        write(u6,*) 'Two atoms submatrix diagonalization'
        call RecPrt('Eigen vectors Matrix = ',' ',SubVec,nBasAtoms,nBasAtoms)
        write(u6,*)
        call RecPrt('Eigen values, real = ',' ',SubVal,nBasAtoms,1)
        write(u6,*)
        call RecPrt('Eigen values, imaginary = ',' ',SubIVal,nBasAtoms,1)
#       endif

        call Seek_n_Destroy(nBasAtoms,SubVal,SubVec,nBast,thr_BO,thr_Dummy1+thr_Diff,TotBondElec,SubDNAOindex,DS_tmp,1,BondAtomA, &
                            BondAtomB,iBondNumb,IAtom,JAtom,Bonds,iDummy,IAtom)

        call mma_deallocate(SubDNAO)
        call mma_deallocate(SubDNAOindex)
        call mma_deallocate(SubVec)
        call mma_deallocate(SubVal)
        call mma_deallocate(SubIVal)
      end do

    end do

    !------------------------------------------------------------------*
    ! Copy of DS matrix                                                *
    !------------------------------------------------------------------*

    DS(:,:) = DS_tmp(:,:)
    call mma_deallocate(DS_tmp)

#   ifdef _DEBUGPRINT_
    call RecPrt('DS-NAO depleted Matrix = ',' ',DS,NBAST,NBAST)
    E = Zero
    do K=1,NBAST
      E = E+DS(K,K)
    end do
    write(u6,*)
    write(u6,*) 'Number of electrons as sum of the DS diagonal = ',E
#   endif

    !------------------------------------------------------------------*
    ! Search for non assigned electrons                                *
    !------------------------------------------------------------------*

    ElecNonAssgn = TotEl-TotCoreElec-TotLoneElec-TotBondElec

#   ifdef _DEBUGPRINT_
    write(u6,*)
    write(u6,*) 'Number of non assigned electrons = ',ElecNonAssgn
#   endif

    !------------------------------------------------------------------*
    ! First we decrease the bond threshold and do the NBO again        *
    !------------------------------------------------------------------*

    if (ElecNonAssgn < 2*thr_NA) exit
    thr_Decr = thr_Decr+thr_DecrStep
    thr_BO = thr_BO-thr_Decr
    if (thr_BO < thr_MIN) exit
  end do

  !--------------------------------------------------------------------*
  ! Second: we search for three centre bonds. This part of the code    *
  ! is quite experimental. If the result is inconsistent, it will be   *
  ! ignored.                                                           *
  !--------------------------------------------------------------------*

  if ((ElecNonAssgn >= 2*thr_NA) .and. (tNUC > 2)) then

    !------------------------------------------------------------------*
    ! Copy of DS matrix for the depletion                              *
    !------------------------------------------------------------------*

    call mma_allocate(DS_tmp,NBAST,NBAST,label='DS_tmp')
    DS_tmp(:,:) = DS(:,:)

    !------------------------------------------------------------------*
    ! Creation of the three centre bond vector and other memory stuff  *
    ! + 100 added to stay on the safe side to avoid memory problems    *
    !------------------------------------------------------------------*

    iElToAsgn = int(ElecNonAssgn+Half)+100

    if (ix_Triple < iElToAsgn) then
      if (SearchedTriple) then
        call mma_deallocate(Tripl)
        call mma_deallocate(TriplAtomA)
        call mma_deallocate(TriplAtomB)
        call mma_deallocate(TriplAtomC)
      end if
      SearchedTriple = .true.
      ix_Triple = iElToAsgn
      call mma_allocate(Tripl,iElToAsgn,label='Tripl')
      call mma_allocate(TriplAtomA,iElToAsgn,label='TriplAtomA')
      call mma_allocate(TriplAtomB,iElToAsgn,label='TriplAtomB')
      call mma_allocate(TriplAtomC,iElToAsgn,label='TriplAtomC')
    end if

    Tripl(:) = 0
    TriplAtomA(:) = 0
    TriplAtomB(:) = 0
    TriplAtomC(:) = 0

    ! Reset the bond threshold to the original value
    thr_BO = thr_Orig

    !------------------------------------------------------------------*
    ! 3 atoms loop                                                     *
    !------------------------------------------------------------------*
    do IAtom=1,tRealNUC-2
      do JAtom=IAtom+1,tRealNUC-1
        do KAtom=JAtom+1,tRealNUC

          nBasAtomsA = NBFpA(IAtom)
          nBasAtomsB = NBFpA(JAtom)
          nBasAtomsC = NBFpA(KAtom)

          nBasAtoms = nBasAtomsA+nBasAtomsB+nBasAtomsC

          ! extraction of sub matrix from DNAO
          ! SubDNAO -> sub matrix
          ! SubVec  -> will contain eigen vectors
          ! SubVal  -> will contain eigen values, real part
          ! SubIVal -> will contain eigen values, imaginary part

          call mma_allocate(SubDNAO,nBasAtoms,nBasAtoms,label='SubDNAO')
          call mma_allocate(SubDNAOindex,2,nBasAtoms,nBasAtoms,label='SubDNAOindex')
          call mma_allocate(SubVec,nBasAtoms,nBasAtoms,label='SubVec')
          call mma_allocate(SubVal,nBasAtoms,label='SubVal')
          call mma_allocate(SubIVal,nBasAtoms,label='SubIVal')

          mSub = 0
          do MY=1,NBAST
            AtomA = center(MY)
            !if (ICNT(MY) <= 0) cycle   ! skip pseudo center
            if ((AtomA /= IAtom) .and. (AtomA /= JAtom) .and. (AtomA /= KAtom)) cycle
            mSub = mSub+1
            nSub = 0
            do NY=1,NBAST
              AtomB = center(NY)
              !if (ICNT(NY) <= 0) cycle ! skip pseudo center
              if ((AtomB /= IAtom) .and. (AtomB /= JAtom) .and. (AtomB /= KAtom)) cycle
              nSub = nSub+1

              SubDNAO(nSub,mSub) = DS(NY,MY)
              SubDNAOindex(:,nSub,mSub) = [NY,MY]
            end do
          end do

#         ifdef _DEBUGPRINT_
          write(u6,*)
          call RecPrt('SubDNAO Matrix = ',' ',SubDNAO,nBasAtoms,nBasAtoms)
#         endif

          ! Diagonalization

          call xEigen(1,nBasAtoms,nBasAtoms,SubDNAO,SubVal,SubIVal,SubVec,iErr)

          if (iErr /= 0) then
            call Error()
            return
          end if

#         ifdef _DEBUGPRINT_
          write(u6,*)
          call RecPrt('Eigen vectors Matrix = ',' ',SubVec,nBasAtoms,nBasAtoms)
          write(u6,*)
          call RecPrt('Eigen values, real = ',' ',SubVal,nBasAtoms,1)
          write(u6,*)
          call RecPrt('Eigen values, imaginary = ',' ',SubIVal,nBasAtoms,1)
#         endif

          call Seek_n_Destroy(nBasAtoms,SubVal,SubVec,nBast,thr_LP,thr_Dummy1,TotTriplBondElec,SubDNAOindex,DS_tmp,3,TriplAtomA, &
                              TriplAtomB,iTriplBondNumb,IAtom,JAtom,Tripl,TriplAtomC,KAtom)

          call mma_deallocate(SubDNAO)
          call mma_deallocate(SubDNAOindex)
          call mma_deallocate(SubVec)
          call mma_deallocate(SubVal)
          call mma_deallocate(SubIVal)
        end do
      end do
    end do

    if (TotTriplBondElec > ElecNonAssgn) then

      iTriplBondNumb = 0

    else

      ElecNonAssgn = ElecNonAssgn-TotTriplBondElec
      TotBondElec = TotBondElec+TotTriplBondElec

      !----------------------------------------------------------------*
      ! Copy of DS matrix                                              *
      !----------------------------------------------------------------*

      DS(:,:) = DS_tmp(:,:)

#     ifdef _DEBUGPRINT_
      call RecPrt('DS-NAO depleted Matrix = ',' ',DS,NBAST,NBAST)
      E = Zero
      do K=1,NBAST
        E = E+DS(K,K)
      end do
      write(u6,*)
      write(u6,*) 'Number of electrons as sum of the DS diagonal = ',E
#     endif

    end if
    call mma_deallocate(DS_tmp)

  end if

  !--------------------------------------------------------------------*
  ! Third: if no bond has been found, but it should, we do             *
  ! EVERYTHING from the beginning, with a lower threshold for lone     *
  ! pairs.                                                             *
  !--------------------------------------------------------------------*

  if ((isThereAtLeastABond /= 0) .and. (TotBondElec < 0.1_wp) .and. (thr_LP == thr_LP_Orig)) then
    thr_LP = thr_LP+0.1_wp
    DS(:,:) = DS_orig(:,:)
  else
    exit
  end if
end do

call mma_deallocate(DS_orig)

!----------------------------------------------------------------------*
! Fourth: we look for single occupied orbitals on each atom.           *
! This part of the code is quite experimental, so if it will give      *
! strange results, they will simply be ignored.                        *
!----------------------------------------------------------------------*

if (ElecNonAssgn >= thr_NA) then

  ! upper limit, overestimated to stay on the safe side
  iElToAsgn = int(ElecNonAssgn+0.6_wp)+10
  !iElToAsgn = int(ElecNonAssgn+0.6_wp)+1

  if (ix_Single < iElToAsgn) then
    if (SearchedSingle) then
      call mma_deallocate(SingEl)
      call mma_deallocate(SingElAtom)
    end if
    ix_Single = iElToAsgn
    SearchedSingle = .true.
    call mma_allocate(SingEl,iElToAsgn,label='SingEl')
    call mma_allocate(SingElAtom,iElToAsgn,label='SingElAtom')
  end if

  SingEl(:) = Zero
  SingElAtom(:) = 0

  do IAtom=1,tRealNUC

    nBasAtoms = NBFpA(IAtom)

    ! extraction of sub matrix from DNAO
    ! SubDNAO -> sub matrix
    ! SubVec  -> will contain eigen vectors
    ! SubVal  -> will contain eigen values, real part
    ! SubIVal -> will contain eigen values, imaginary part

    call mma_allocate(SubDNAO,nBasAtoms,nBasAtoms,label='SubDNAO')
    call mma_allocate(SubDNAOindex,2,nBasAtoms,nBasAtoms,label='SubDNAOindex')
    call mma_allocate(SubVec,nBasAtoms,nBasAtoms,label='SubVec')
    call mma_allocate(SubVal,nBasAtoms,label='SubVal')
    call mma_allocate(SubIVal,nBasAtoms,label='SubIVal')

    mSub = 0
    do MY=1,NBAST
      AtomA = center(MY)
      !if (ICNT(MY) <= 0) cycle ! skip pseudo center
      if (AtomA /= IAtom) cycle ! we want just one atom a time
      mSub = mSub+1
      nSub = 0
      do NY=1,NBAST
        AtomB = center(NY)
        !if (ICNT(NY) <= 0) cycle ! skip pseudo center
        if (AtomB /= IAtom) cycle ! they have to be the same atom
        nSub = nSub+1

        SubDNAO(nSub,mSub) = DS(NY,MY)
        SubDNAOindex(:,nSub,mSub) = [NY,MY]
      end do
    end do

#   ifdef _DEBUGPRINT_
    write(u6,*)
    call RecPrt('SubDNAO Matrix = ',' ',SubDNAO,nBasAtoms,nBasAtoms)
#   endif

    ! Diagonalization

    call xEigen(1,nBasAtoms,nBasAtoms,SubDNAO,SubVal,SubIVal,SubVec,iErr)

    if (iErr /= 0) then
      call Error()
      return
    end if

#   ifdef _DEBUGPRINT_
    write(u6,*)
    call RecPrt('Eigen vectors Matrix = ',' ',SubVec,nBasAtoms,nBasAtoms)
    write(u6,*)
    call RecPrt('Eigen values, real = ',' ',SubVal,nBasAtoms,1)
    write(u6,*)
    call RecPrt('Eigen values, imaginary = ',' ',SubIVal,nBasAtoms,1)
#   endif

    call Seek_n_Destroy(nBasAtoms,SubVal,SubVec,nBast,thr_SO,thr_Dummy2,TotSingleElec,SubDNAOindex,DS,2,SingElAtom,iDummy, &
                        iSingNumb,IAtom,IAtom,SingEl,iDummy,IAtom)

    call mma_deallocate(SubDNAO)
    call mma_deallocate(SubDNAOindex)
    call mma_deallocate(SubVec)
    call mma_deallocate(SubVal)
    call mma_deallocate(SubIVal)
  end do

  if (TotSingleElec > ElecNonAssgn) then
    iSingNumb = 0
  else
    ElecNonAssgn = ElecNonAssgn-TotSingleElec
  end if

end if

! End of searching for non-assigned electrons

!----------------------------------------------------------------------*
! Print out of the NBO analysis                                        *
!----------------------------------------------------------------------*

LabelSym = 'Atom A:Gen.  Atom B:Gen.  Bond Order'
LabelNoSym = 'Atom A       Atom B       Bond Order'
Label3ASym = 'Atom A:Gen.  Atom B:Gen.  Atom C:Gen.  Bond Order'
Label3ANoSym = 'Atom A       Atom B       Atom C       Bond Order'

write(u6,*)
call CollapseOutput(1,'   Natural Bond Order analysis')
write(u6,'(3X,A)') '   ---------------------------'

write(u6,'(6X,A)') 'Based on LoProp computed density'
write(u6,'(6X,85A)') (cSign,i=1,85)

if (iBondNumb > 1) then
  ! Two columns format printing
  iHalf = int(iBondNumb/2)
  iOdd = 0
  if (iHalf*2 /= iBondNumb) iOdd = 1

  if (nSym > 1) then
    write(u6,'(6X,A)') LabelSym//' | '//LabelSym
  else
    write(u6,'(6X,A)') LabelNoSym//' | '//LabelNoSym
  end if
  do I=1,iHalf
    Atom_A = LblCnt4(BondAtomA(I))
    Atom_B = LblCnt4(BondAtomB(I))
    Atom_C = LblCnt4(BondAtomA(I+iHalf))
    Atom_D = LblCnt4(BondAtomB(I+iHalf))
    write(u6,'(8X,A,3X,A,3X,F7.3,2X,A,3X,A,3X,A,3X,F7.3)') Atom_A,Atom_B,Bonds(I),'|',Atom_C,Atom_D,Bonds(I+iHalf)
  end do
  if (iOdd == 1) then
    Atom_A = LblCnt4(BondAtomA(iBondNumb))
    Atom_B = LblCnt4(BondAtomB(iBondNumb))
    write(u6,'(8X,A,3X,A,3X,F7.3,2X,A)') Atom_A,Atom_B,Bonds(iBondNumb),'|'
  end if

else
  ! One column format printing
  if (nSym > 1) then
    write(u6,'(6X,A)') LabelSym
  else
    write(u6,'(6X,A)') LabelNoSym
  end if
  do I=1,iBondNumb
    Atom_A = LblCnt4(BondAtomA(I))
    !& TLbl(BondAtomA(I))//' -  '//LblCnt(BondAtomA(I))(Index(LblCnt(BondAtomA(I)),':')+1:)
    !Atom_A = TLbl(BondAtomA(I))
    Atom_B = LblCnt4(BondAtomB(I))
    !& TLbl(BondAtomB(I))//' -  '//LblCnt(BondAtomB(I))(Index(LblCnt(BondAtomB(I)),':')+1:)
    !Atom_B = TLbl(BondAtomB(I))
    write(u6,'(8X,A,3X,A,3X,F7.3)') Atom_A,Atom_B,Bonds(I)
  end do
end if

write(u6,'(6X,85A)') (cSign,i=1,85)

if (iTriplBondNumb > 0) then
  if (nSym > 1) then
    write(u6,'(6X,A)') Label3ASym
  else
    write(u6,'(6X,A)') Label3ANoSym
  end if
  do I=1,iTriplBondNumb
    Atom_A = LblCnt4(TriplAtomA(I))
    Atom_B = LblCnt4(TriplAtomB(I))
    Atom_C = LblCnt4(TriplAtomC(I))
    write(u6,'(8X,A,3X,A,3X,A,3X,F7.3)') Atom_A,Atom_B,Atom_C,Tripl(I)
  end do

  write(u6,'(6X,85A)') (cSign,i=1,85)

end if

if (TotCoreElec > Zero) then
  write(u6,'(6X,A,F10.3,A)') 'NBO located ',TotCoreElec,' core electrons.'
end if

if (TotLoneElec > Zero) then
  write(u6,'(6X,A,F10.3,A)') 'NBO located ',TotLoneElec,' lone pair electrons.'
end if

if (TotBondElec > Zero) then
  write(u6,'(6X,A,F10.3,A,I4,A)') 'NBO located ',TotBondElec,' electrons involved in ',iBondNumb+iTriplBondNumb,' bonds.'
end if

if (iSingNumb > 0) then
  do I=1,iSingNumb
    Atom_A = LblCnt4(SingElAtom(I))
    write(u6,'(6X,A,F10.3,A,A)') 'NBO located ',SingEl(I),' non-bonded electrons on atom ',Atom_A

  end do
end if

if (ElecNonAssgn > 6.0e-4_wp) then
  write(u6,'(6X,A,F8.3,A)') 'The remaining ',ElecNonAssgn,' electrons are to be considered as diffuse'
  !write(u6,'(29X,A)') 'diffuse on more than one bond.'
elseif (ElecNonAssgn < (Zero-0.01_wp)) then
  write(u6,'(6X,A)') 'NBO analysis, and just that ONLY, did not converge to a'
  write(u6,'(6X,A)') 'proper answer, sorry. Calculation will continue as normal.'
end if

call CollapseOutput(0,'   Natural Bond Order analysis')
write(u6,*)

!----------------------------------------------------------------------*
! Deallocation                                                         *
!----------------------------------------------------------------------*

call mma_deallocate(LblCnt4)

call mma_deallocate(DS)
call mma_deallocate(center)
call mma_deallocate(NBFpA)
call mma_deallocate(BondAtomB)
call mma_deallocate(BondAtomA)
call mma_deallocate(Bonds)
call mma_deallocate(Allc)
call mma_deallocate(ANr)

if (SearchedSingle) then
  call mma_deallocate(SingEl)
  call mma_deallocate(SingElAtom)
end if

if (SearchedTriple) then
  call mma_deallocate(Tripl)
  call mma_deallocate(TriplAtomA)
  call mma_deallocate(TriplAtomB)
  call mma_deallocate(TriplAtomC)
end if

!----------------------------------------------------------------------*
! That's All, Folks                                                    *
!----------------------------------------------------------------------*

return

contains

subroutine Error()
  write(u6,*) 'Something went wrong when diagonalizing.'
  write(u6,*) 'NBO analysis cannot be finished, sorry.'
  call mma_deallocate(LblCnt4)
end subroutine Error

end subroutine NAT_BOND_ORDER
