/*###############################################################################
# Linux Management Providers (LMP), SSH provider package
# Copyright (C) 2010 Feng Zhanlei, REDFLAG <fengzhanlei@redflag-linux.com>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by Guillaume BOTTEX <guillaumebottex@uxsystem.net>, UXSystem, 2010
###############################################################################*/

#include "OpenDRIM_SSHProtocolEndpointAccess.h"

const string systemCreationClassName = "OpenDRIM_ComputerSystem";
const string creationClassName = "OpenDRIM_SSHProtocolEndpoint";
string systemName;

int SSH_OpenDRIM_SSHProtocolEndpoint_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	CF_assert(CF_getSystemName(systemName, errorMessage));
	_L_;
	return OK;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_unload(string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_SSHProtocolEndpoint>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	vector<string> endpoints;
	CF_getActiveSSHSessionItem(endpoints, errorMessage);
	
	for (unsigned int i = 0; i < endpoints.size(); i++)
	{
		//Initialize an instance
		OpenDRIM_SSHProtocolEndpoint instance;
		instance.setSystemCreationClassName(systemCreationClassName);
		instance.setSystemName(systemName);
		instance.setCreationClassName(creationClassName);
		
		instance.setName(endpoints[i]);
		
		if (discriminant == "ei")
			CF_assert(SSH_OpenDRIM_SSHProtocolEndpoint_populate(instance, errorMessage));
		
		result.push_back(instance);
	}
	_L_;
	return OK;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_SSHProtocolEndpoint& instance, const char** properties, string& errorMessage) {
	_E_;
	vector<string>::size_type index;
	vector<string> if_endpoints;

	CF_getActiveSSHSessionItem(if_endpoints, errorMessage);

	if (instance.SystemCreationClassName == systemCreationClassName &&
			instance.CreationClassName == creationClassName &&
			instance.SystemName == systemName &&
			CF_foundInList(instance.Name, if_endpoints, index))
	{
		CF_assert(SSH_OpenDRIM_SSHProtocolEndpoint_populate(instance, errorMessage));
	}
	else
	{
		errorMessage = "Invalid path";
		return NOT_FOUND;
	}
	_L_;
	return OK;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_SSHProtocolEndpoint& newInstance, const OpenDRIM_SSHProtocolEndpoint& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_SSHProtocolEndpoint& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_SSHProtocolEndpoint& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_SSHProtocolEndpoint& instance, unsigned int& returnValue, const OpenDRIM_SSHProtocolEndpoint_RequestStateChange_In& in, OpenDRIM_SSHProtocolEndpoint_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_BroadcastReset(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_SSHProtocolEndpoint& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int SSH_OpenDRIM_SSHProtocolEndpoint_populate(OpenDRIM_SSHProtocolEndpoint& instance, string& errorMessage) {
	_E_;
	/*
	 * Properties to fill from profile
	 * + Mandatory:
	 * [X] SystemCreationClassName   [KEY]
	 * [X] CreationClassName         [KEY]
	 * [X] SystemName                [KEY]
	 * [X] Name                      [KEY]
	 * [X] NameFormat
	 * [X] ProtocolIFType
	 * [X] OtherTypeDescription
	 * [X] ElementName
	 * [X] EnabledSSHVersions
	 * [X] SSHVersion
	 * [X] EnabledEncryptionAlgorithms
	 * [X] EncryptionAlgorithm
	 * [X] Idle Timeout
	 * [X] KeepAlive
	 * [X] ForwardX11
	 * [X] Compression
	 * + Conditinal:
	 * [X] OtherEnabledSSHVersion
	 * [X] OtherSSHVersion
	 * [X] OtherEnabledEncryptionAlgorithm
	 * [X] OtherEncryptionAlgorithm
	 */

	instance.setNameFormat("network-interface-name");
	instance.setProtocolIFType(IFTYPE_OTHER);
	instance.setOtherTypeDescription("SSH");
	
	if (instance.ProtocolIFType == IFTYPE_OTHER)
		instance.setElementName("SSH Session: " + instance.Name);

	vector<unsigned short> enabledSSHVersions;

	string enabled_ssh_versions;
	CF_getConfigItem("Protocol", enabled_ssh_versions, errorMessage);

	if(enabled_ssh_versions == "1")
	{
		enabledSSHVersions.push_back(SSHV_SSHv1);
		instance.setSSHVersion(SSHV_SSHv1);
	}
	else if (enabled_ssh_versions == "2")
	{
		enabledSSHVersions.push_back(SSHV_SSHv2);
		instance.setSSHVersion(SSHV_SSHv2);
	}
	else if (enabled_ssh_versions == "2,1" ||enabled_ssh_versions == "1,2"||enabled_ssh_versions == "")
	{
		enabledSSHVersions.push_back(SSHV_SSHv1);
		enabledSSHVersions.push_back(SSHV_SSHv2);
		instance.setSSHVersion(OTHER);
		instance.setOtherSSHVersion("SSHv1, SSHv2");
	}
	else
	{
		enabledSSHVersions.push_back(SSHV_UNKNOWN);
		instance.setSSHVersion(SSHV_UNKNOWN);
	}

	instance.setEnabledSSHVersions(enabledSSHVersions);

	/* maybe needed to modified */
	vector<unsigned short> enabledEncryptionAlgorithms;

	enabledEncryptionAlgorithms.push_back(OTHER);
	enabledEncryptionAlgorithms.push_back(DES3);
	enabledEncryptionAlgorithms.push_back(RC4);
	instance.setEnabledEncryptionAlgorithms(enabledEncryptionAlgorithms);

	/* CIM Shcema 2.25.0 Final release version
	vector<string> otherEnabledEncryptionAlgorithm;
	otherEnabledEncryptionAlgorithm.push_back("AES");
	otherEnabledEncryptionAlgorithm.push_back("BLOWFISH");
	otherEnabledEncryptionAlgorithm.push_back("CAST");
	instance.setOtherEnabledEncryptionAlgorithm(otherEnabledEncryptionAlgorithm);
	*/
	
	// CIM Shcema 2.25.0 Experimental release version
	instance.setOtherEnabledEncryptionAlgorithm("AES, BLOWFISH, CAST");
	
	string encryptionAlgorithm;
	CF_getConfigItem("Ciphers", encryptionAlgorithm, errorMessage);

	if (encryptionAlgorithm.size() != 0)
	{
		instance.setEncryptionAlgorithm(OTHER);
		/* CIM Shcema 2.25.0 Final release version
		vector<string> otherEncryptionAlgorithm;
		otherEncryptionAlgorithm.push_back(encryptionAlgorithm);
		instance.setOtherEncryptionAlgorithm(otherEncryptionAlgorithm);
		*/
		
		// CIM Shcema 2.25.0 Experimental release version
		instance.setOtherEncryptionAlgorithm(encryptionAlgorithm);
	}
	else
	{
		instance.setEncryptionAlgorithm(OTHER);
		/* CIM Shcema 2.25.0 Final release version
		vector<string> otherEncryptionAlgorithm;
		
		otherEncryptionAlgorithm.push_back("aes128-cbc");
		otherEncryptionAlgorithm.push_back("3des-cbc");
		otherEncryptionAlgorithm.push_back("blowfish-cbc");
		otherEncryptionAlgorithm.push_back("cast128-cbc");
		otherEncryptionAlgorithm.push_back("arcfour128");
		otherEncryptionAlgorithm.push_back("arcfour256");
		otherEncryptionAlgorithm.push_back("arcfour");
		otherEncryptionAlgorithm.push_back("aes192-cbc");
		otherEncryptionAlgorithm.push_back("aes256-cbc");
		otherEncryptionAlgorithm.push_back("aes128-ctr");
		otherEncryptionAlgorithm.push_back("aes192-ctr");
		otherEncryptionAlgorithm.push_back("aes256-ctr");
		*/
		
		// CIM Shcema 2.25.0 Experimental release version
		string otherEncryptionAlgorithm;
		
		otherEncryptionAlgorithm  = "aes128-cbc, ";
		otherEncryptionAlgorithm += "3des-cbc, ";
		otherEncryptionAlgorithm += "blowfish-cbc, ";
		otherEncryptionAlgorithm += "cast128-cbc, ";
		otherEncryptionAlgorithm += "arcfour128, ";
		otherEncryptionAlgorithm += "arcfour256, ";
		otherEncryptionAlgorithm += "arcfour, ";
		otherEncryptionAlgorithm += "aes192-cbc, ";
		otherEncryptionAlgorithm += "aes256-cbc, ";
		otherEncryptionAlgorithm += "aes128-ctr, ";
		otherEncryptionAlgorithm += "aes192-ctr, ";
		otherEncryptionAlgorithm += "aes256-ctr";
		
		instance.setOtherEncryptionAlgorithm(otherEncryptionAlgorithm);
	}

	instance.setIdleTimeout(0);

	string config_str;
	CF_getConfigItem("KeepAlive",config_str, errorMessage);

	instance.setKeepAlive(!CF_startsWithNoCase(config_str, "no"));

	CF_getConfigItem("X11Forwarding", config_str, errorMessage);
	
	instance.setForwardX11(CF_startsWithNoCase(config_str, "yes"));

	CF_getConfigItem("Compression", config_str, errorMessage);

	instance.setCompression(!CF_startsWithNoCase(config_str, "no"));

	_L_;
	return OK;
}

