/*
 * Copyright 1997-2025 Optimatika
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.ojalgo.optimisation.convex;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import org.ojalgo.TestUtils;
import org.ojalgo.array.ArrayR256;
import org.ojalgo.function.constant.BigMath;
import org.ojalgo.optimisation.Expression;
import org.ojalgo.optimisation.ExpressionsBasedModel;
import org.ojalgo.optimisation.ModelFileTest;
import org.ojalgo.optimisation.Optimisation;
import org.ojalgo.optimisation.Optimisation.Result;
import org.ojalgo.type.context.NumberContext;

/**
 * A collection of datasets found here: ftp://ftp.numerical.rl.ac.uk/pub/cutest//marosmeszaros/marmes.html
 * <p>
 * Tests with more than 1000 variables and/or constraints are tagged "slow" (can't be solved with community
 * version of CPLEX)
 * <p>
 * Tests that are otherwise difficult for ojAlgo are tagged "unstable"
 *
 * @author apete
 */
public class CuteMarosMeszarosCase extends OptimisationConvexTests implements ModelFileTest {

    public static final class ModelInfo {

        /**
         * number of rows in A
         */
        public int M;
        /**
         * number of variables
         */
        public int N;
        /**
         * number of nonzeros in A
         */
        public int NZ;
        /**
         * solution value obtained by the default settings of BPMPD solver
         */
        public BigDecimal OPT;
        /**
         * number of quadratic variables
         */
        public int QN;
        /**
         * number of off-diagonal entries in the lower triangular part of Q
         */
        public int QNZ;

        /**
         * The fraction of variables that are quadratic.
         *
         * @return [0,1]
         */
        public double getRatioQP() {
            return (double) QN / (double) N;
        }

        /**
         * All variables are quadratic
         */
        public boolean isPureQP() {
            return QN == N;
        }

        /**
         * The quadratic problem is called separable if Q has no off-diagonal nonzeros.
         */
        public boolean isSeparable() {
            return QNZ == 0;
        }

        /**
         * Number of variables and constraints are less than 1000 – can then use the community edition of
         * CPLEX.
         */
        public boolean isSmall() {
            return M <= 1_000 && N <= 1_000;
        }

        @Override
        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("ModelInfo [M=");
            builder.append(M);
            builder.append(", N=");
            builder.append(N);
            builder.append(", NZ=");
            builder.append(NZ);
            builder.append(", QN=");
            builder.append(QN);
            builder.append(", QNZ=");
            builder.append(QNZ);
            builder.append(", OPT=");
            builder.append(OPT);
            builder.append("]");
            return builder.toString();
        }

    }

    /**
     * The optimal objective function value is given with 8 digits in the file 00README.QP, but they don't
     * seem to be exact.
     */
    private static final NumberContext ACCURACY = NumberContext.of(6, 8);

    private static final Map<String, ModelInfo> MODEL_INFO;

    static {

        Map<String, ModelInfo> modelInfo = new HashMap<>();

        String line;

        try (BufferedReader reader = new BufferedReader(new InputStreamReader(TestUtils.getResource("optimisation", "marosmeszaros", "00README.CSV")))) {

            while ((line = reader.readLine()) != null) {

                String[] parts = line.split("\\s+");

                String key = parts[0].toUpperCase();

                ModelInfo value = new ModelInfo();

                value.M = Integer.parseInt(parts[1]);
                value.N = Integer.parseInt(parts[2]);
                value.NZ = Integer.parseInt(parts[3]);
                value.QN = Integer.parseInt(parts[4]);
                value.QNZ = Integer.parseInt(parts[5]);
                value.OPT = new BigDecimal(parts[6]);

                modelInfo.put(key, value);
            }

        } catch (IOException cause) {
            throw new RuntimeException(cause);
        }

        // Overide with values known to be better

        modelInfo.get("HS268").OPT = BigMath.ZERO;

        MODEL_INFO = Collections.unmodifiableMap(modelInfo);
    }

    public static Map<String, ModelInfo> getModelInfo() {
        return MODEL_INFO;
    }

    public static ModelInfo getModelInfo(final String model) {
        String key = model.toUpperCase();
        key = key.replace("_", "");
        int dotIndex = key.indexOf(".");
        if (dotIndex > 0) {
            key = key.substring(0, dotIndex);
        }
        return MODEL_INFO.get(key);
    }

    public static ExpressionsBasedModel makeModel(final String name) {
        return ModelFileTest.makeModel("marosmeszaros", name, false);
    }

    private static void doTest(final String name, final NumberContext accuracy) {

        String expMinValString = CuteMarosMeszarosCase.getModelInfo(name).OPT.toPlainString();
        ExpressionsBasedModel model = ModelFileTest.makeModel("marosmeszaros", name, false);

        if (DEBUG) {
            // model.options.debug(Optimisation.Solver.class);
            // model.options.debug(IntegerSolver.class);
            model.options.debug(ConvexSolver.class);
            model.options.convex().extendedPrecision(true);
            // model.options.debug(LinearSolver.class);
            // model.options.progress(IntegerSolver.class);
            // model.options.validate = false;
            // model.options.mip_defer = 0.25;
            // model.options.mip_gap = 1.0E-5;
        }

        ModelFileTest.assertValues(model, expMinValString, null, accuracy != null ? accuracy : ACCURACY);

        //  BasicLogger.debug(CuteMarosMeszarosCase.getModelInfo(name));

    }

    static void doTest(final String name) {
        CuteMarosMeszarosCase.doTest(name, ACCURACY);
    }

    /**
     * <p>
     * The QP solver ends up (de)activating inequality constraints "forever"...
     */
    @Test
    @Tag("unstable")
    @Tag("slow")
    public void testCVXQP1_M() {
        CuteMarosMeszarosCase.doTest("CVXQP1_M.SIF");
    }

    /**
     * <p>
     * The QP solver ends up (de)activating inequality constraints "forever"...
     */
    @Test
    @Tag("unstable")
    @Tag("slow")
    public void testCVXQP2_M() {
        CuteMarosMeszarosCase.doTest("CVXQP2_M.SIF");
    }

    /**
     * <p>
     * The QP solver ends up (de)activating inequality constraints "forever"...
     */
    @Test
    @Tag("unstable")
    @Tag("slow")
    public void testCVXQP3_M() {
        CuteMarosMeszarosCase.doTest("CVXQP3_M.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testDUALC1() {
        CuteMarosMeszarosCase.doTest("DUALC1.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testDUALC2() {
        CuteMarosMeszarosCase.doTest("DUALC2.SIF", ACCURACY.withPrecision(7));
    }

    /**
     * <p>
     */
    @Test
    public void testDUALC5() {
        CuteMarosMeszarosCase.doTest("DUALC5.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testDUALC8() {
        CuteMarosMeszarosCase.doTest("DUALC8.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testGENHS28() {
        CuteMarosMeszarosCase.doTest("GENHS28.SIF");
    }

    /**
     * <p>
     * The QP solver ends up (de)activating inequality constraints "forever"...
     */
    @Test
    @Tag("unstable")
    @Tag("slow")
    public void testGOULDQP2() {
        CuteMarosMeszarosCase.doTest("GOULDQP2.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testHS21() {
        CuteMarosMeszarosCase.doTest("HS21.SIF");
    }

    /**
     * <p>
     * The given objective function value is 5.7310705e-07 but CPLEX gets 1.1702830307e-05 (and ojAlgo
     * 1.9521E-23). The CPLEX solution is:
     *
     * <pre>
    C------1                      0.995735
    C------2                      1.995283
    C------3                     -0.999028
    C------4                      2.989736
    C------5                     -3.982628
     * </pre>
     *
     * Guessing that { 1.0, 2.0, -1.0, 3.0, -4.0 } is the exact/actual optimal solution. That gives the
     * objective value 0.0 (exactly).
     * <p>
     * Find it somewhat surprising that:
     * <ol>
     * <li>The given value is so inexact. It's a small model. You can validate it with pen and paper.
     * <li>CPLEX returns a very inexact solution, it's only correct to 2 digits precision, resulting in a
     * value worse than the given.
     * </ol>
     */
    @Test
    public void testHS268() {

        CuteMarosMeszarosCase.doTest("HS268.SIF", ACCURACY.withScale(4));

        ExpressionsBasedModel model = CuteMarosMeszarosCase.makeModel("HS268.SIF");

        Result proposed = new Result(Optimisation.State.OPTIMAL, ArrayR256.wrap(BigMath.ONE, BigMath.TWO, BigMath.NEG, BigMath.THREE, BigMath.FOUR.negate()));
        Result cplex = Result.of(Double.NaN, Optimisation.State.OPTIMAL, 0.995735, 1.995283, -0.999028, 2.989736, -3.982628);

        Expression obj = model.objective();

        // Assert that the proposed solution results in a better objective function value.
        BigDecimal propVal = obj.evaluate(proposed);
        BigDecimal cplexVal = obj.evaluate(cplex);

        TestUtils.assertLessThan(5.7310705e-07, propVal.doubleValue()); // Given
        TestUtils.assertLessThan(1.1702830307e-05, propVal.doubleValue()); // CPLEX
        TestUtils.assertTrue(propVal.compareTo(cplexVal) < 0); // From CPLEX solution

        TestUtils.assertEquals(1.1702830307e-05, cplexVal.doubleValue(), ACCURACY.withScale(4));

        // Assert proposed solution valid with very high precision, and that it gives objective function value 0.0
        NumberContext VERY_HIGH_PRECISION = NumberContext.of(24);
        TestUtils.assertTrue(model.validate(cplex, VERY_HIGH_PRECISION));
        TestUtils.assertEquals(BigMath.ZERO, propVal, VERY_HIGH_PRECISION);
    }

    /**
     * <p>
     */
    @Test
    public void testHS35() {
        CuteMarosMeszarosCase.doTest("HS35.SIF", ACCURACY.withPrecision(6));
    }

    /**
     * <p>
     */
    @Test
    public void testHS35MOD() {
        CuteMarosMeszarosCase.doTest("HS35MOD.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testHS51() {
        CuteMarosMeszarosCase.doTest("HS51.SIF", ACCURACY.withScale(8));
    }

    /**
     * <p>
     */
    @Test
    public void testHS52() {
        CuteMarosMeszarosCase.doTest("HS52.SIF", ACCURACY.withScale(7));
    }

    /**
     * <p>
     */
    @Test
    public void testHS53() {
        CuteMarosMeszarosCase.doTest("HS53.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testHS76() {
        CuteMarosMeszarosCase.doTest("HS76.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testKSIP() {
        CuteMarosMeszarosCase.doTest("KSIP.SIF");
    }

    /**
     * <p>
     * Just takes way too long – 275s
     */
    @Test
    @Tag("slow")
    public void testMOSARQP2() {
        CuteMarosMeszarosCase.doTest("MOSARQP2.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testPRIMAL1() {
        CuteMarosMeszarosCase.doTest("PRIMAL1.SIF", ACCURACY.withScale(6));
    }

    /**
     * <p>
     */
    @Test
    public void testPRIMALC2() {
        CuteMarosMeszarosCase.doTest("PRIMALC2.SIF", ACCURACY.withScale(6));
    }

    /**
     * <p>
     */
    @Test
    public void testPRIMALC8() {
        CuteMarosMeszarosCase.doTest("PRIMALC8.SIF", ACCURACY.withScale(6));
    }

    /**
     * <p>
     */
    @Test
    public void testQADLITTL() {
        CuteMarosMeszarosCase.doTest("QADLITTL.SIF");
    }

    /**
     * <p>
     * Wrong solution!
     */
    @Test
    @Tag("unstable")
    public void testQBANDM() {

        // CPLEX: OPTIMAL 16352.342063920036 @ { 0, 0, 2.7308565E-7, 0.34199997729322, 5.31699999869973, 29.84239364133395, 1.88999978123391, 22.37334536993129, 9.11828541355016, 0.99999875239495, 1.44599979765239, 0, 0, 7.86099802542357, 4.43600077467359, 1.59599918660345, 1.9807168E-7, 0, 1.19199996993713, 0.00000231458979, 3.37899996920831, 0, 0, 1.4177412E-7, 0.97299995952348, 0.41399827460137, 0.11849502002161, 3.574754E-8, 2.7197548E-7, 2.5414653E-7, 2.6680405E-7, 2.043066E-8, 0.71233555441194, 0.00000190421969, 1.970747E-8, 4.846987E-8, 2.358804E-8, 0.00000366490918, 2.291753E-8, 4.212149E-8, 1.739177E-8, 2.133776E-8, 0, 5.074477E-8, 6.939099E-8, 0.00000270175154, 0.00000317949089, 3.64807E-8, 3.399359E-8, 0.00000335397862, 4.030601E-8, 0.00000358959994, 0.00000354819168, 0.00000315188808, 3.158481E-8, 0.00000267205213, 1.48148053385165, 0.00000422636935, 0.00000575874131, 1.2592453E-7, 2.043848E-8, 1.9869679E-7, 7.4453706E-7, 8.480411E-8, 9.6547148E-7, 3.1721742E-7, 8.436234E-8, 0.48099989171204, 9.81250436490421, 0.90472777978913, 7.98168801044593, 2.89168881284467, 8.04817915185546, 1.84542473675475, 2.1455755300493, 4.46204E-8, 2.4029755E-7, 2.57370333536571, 3.7700002E-7, 3.08667971914987, 5.32164835193133, 1.35145462011842, 8.22468455939326, 3.11661182893992, 0.76075409335455, 1.03121615348996, 1.99314324064819, 6.4047998E-7, 1.32916453483867, 1.311018E-8, 0.84516914852092, 0.00000387595326, 2.23888985126182, 6.40030274457484, 1.20165874079533, 3.25465155228594, 10.42789341389705, 2.3098659325387, 5.365379E-7, 18.74778955930257, 14.20165082037393, 16.18010477911494, 9.0222486319391, 1.60411065861029, 37.33352617621192, 13.45048072109713, 3.90886604495328, 18.2760364671452, 1.15208871858372, 7.26055956386172, 0.0000106043455, 0.38531042052454, 0.00000863608428, 11.40703368717047, 0.00000381920793, 0.00000206467132, 17.23589183076318, 0.00010407044963, 0.00000531781386, 4.6559069E-7, 0.0000042958297, 7.91522185706216, 0.0, 13.37322474914993, 1.42582770151345, 40.54674944382537, 4.65840845976203, 0.00000410268247, 31.61839303698477, 6.38799351454512, 0.00000502278561, 0.00002661049332, 4.6365334E-7, 0.00000236995042, 1.01064823398536, 1.02294662324322, 0.00000671668527, 0.28681803169544, 0.00004905715506, 6.23392494816385, 0.00000474373398, 0.00000584330447, 3.1297258E-7, 0.00002110332111, 0.00000921174185, 1.52829220518094, 0.00005239308541, 20.35198167223498, 1.00131456266524, 42.9026126582033, 1.28814409805976, 0.7691495865033, 0.145, 0.93, 6.15335962128262, 168.1115167362439, 2.02336179974667, 1.67188338205026, 10.12952025191488, 1.84734125920467, 6.23763889263889, 11.11859371654775, 2.59799998688982, 3.872295E-8, 8.1339655E-7, 9.83931043927201, 0.21712256605814, 4.7614487E-7, 0.81221860985961, 0, 1.83033730192832, 4.047652E-8, 0.73799989870458, 2.270678E-8, 3.006287E-8, 0.00000721136876, 3.58111816615608, 2.9050410874546, 0.00000407989641, 3.42299592010359, 0, 0.00000981562897, 1.28099021516478, 3.079169E-8, 0.5290642, 0.578, 0.64, 0.43824598515343, 1.588, 5.495986E-8, 1.46275591476667, 3.42824399453415, 0, 1.44897608509837, 1.00413071259333, 4.9697772E-7, 0.95011999390322, 1.61080127652372, 4.53629536331307, 0.38449647675308, 0.1090972028257, 1.99999938968782, 9.521568E-8, 5.2132993E-7, 0.446, 0.00000290871863, 0.20499709128137, 0.51161565082805, 0.00002912364947, 5.8968855E-7, 0.05500444645845, 0.00035018938678, 0.65535261680824, 0.18600219211716, 0.0000034679763, 3.6825297E-7, 0.13487921600808, 0.37311244106922, 0.06567179834926, 0.04674359058684, 4.03518493864065, 0.00007147077744, 7.26056707011656, 0.00000750625484, 0.00000511986118, 0.0001087336351, 0.00000466318546, 0.00000181231597, 0.00000409760255, 17.23599646679572, 0.74868041712022, 1.42583520723119, 0.00000750571775, 0.00000903609042, 14.79905995638113, 0.00000648545488, 0.00000146266927, 0.00000112708241, 6.388, 0.00000190734363, 6.38799809265637, 1.35547475923795, 2.58, 1.430805, 1.279, 0.93376728875823, 9.48922E-9, 7.343265E-8, 1.0659081E-7, 1.66790572801926, 2.3392096E-7, 0.35, 0.687, 1.6, 0.0016542747037, 5.2142585E-7, 7.1030856E-7, 0.55834085979409, 0.00000331189641, 3.891118E-8, 4.589313E-8, 2.8085013E-7, 0.00000152647183, 4.0434159E-7, 0.00000197457643, 0, 0, 0, 0, 0, 0, 0, 1.635927701954998155662117299889340, 2.78941992897301, 4.1100925E-7, 1.26754846860069, 3.28472085086386, 1.55135475734984, 100.7077, 195.008578103926, 0.05035385, 0.10785887023483, 49.73, 452.1380263859978, 150.7880317997303, 0.15078803179973, 90.48186726656482, 9.60824071908567, 19.21648143817135, 38.47692293797738, 168.6593558766703, 168.6593558766703, 843.2967793833516, 0.12005500091947, 0.63377423430034, 0.02418037804107, 3.72342837980422, 18.61714189902108, 5.65247918277012, 14.31349363473625, 21.65543993341398, 0.00290380349868, 17.10407508510231, 23.37213683025545, 0.03847103380737, 15.38841352294826, 2.05564352574642, 5.4664412E-7, 11.80549563509325, 11.80549563509049, 0, 3.281, 1.437, 30.15760635994605, 1.30027E-9, 1.147, 1.88265112415704, 0, 19.46, 0.763, 0.283, 1.67628206600573, 0.394, 0.153, 9.27471793399427, 0.44615305744574, 3.69399986565091, 24.01100013434909, 4.954497E-8, 2.68214024838854, 7.44685675960843, 3.84329628763427, 2.19999962299998, 0.50148536854616, 1.369, 2.77996222598069, 36.01649931036961, 4.26190379156868, 4.40596488913404, 0.09643525519852, 0.00003511086596, 6.07044185677063, 4.99042360213434, 0.350982, 12.32057639786566, 0.51640303987107, 2.03172278831359, 7.69420676147413, 4.11128705149283, 0.00000182212883, 6.99255816400048, 0.00005379909094, 0.38234708415243, 2.593338199457, 3.68460855116478, 0.00000228019326, 3.28709883740085, 3.98483520498245, 2.95262420658927, 1.8249641E-7, 3.44958258434486, 1.5993108E-7, 1.2000124905977, 2.18494206207847, 0.00000233821431, 5.496769E-8, 0.2680737330316, 2.96199968278258, 0.88499991563766, 1.0828796E-7, 1.14165219021519, 1.71700074856303, 1.544, 0.06745212838014, 0.56306636462648, 1.99823155764743, 1.79621078318551, 4.4214995E-7, 0.9875145106674, 0.92304618928153, 1.47872184281388, 0.00000569146086, 0.00000224498055, 0.59840378408312, 2.4944585E-7, 0.11686782216635, 0.18859119341722, 0.27666040887231, 0.04481971154731, 0.00000293558688, 0.00000870453552, 1.31395403720419, 0.45494036162758, 1.35434497129564, 0.00000536200002, 8.71177009594251, 2.26184574163884, 2.51218921858915, 0.00000124760505, 2.023476E-7, 0, 1.84800021876609, 2.1876609E-7, 0.518, 0.228, 0.138, 0.63781853627385, 4.92807673960403, 0.6627680192719, 1.68658275347793, 0.00002256347056, 4.95106422255263, 0.00000449811578, 0.42754383679326, 1.8048588E-7, 1.50517860635426, 0.11665497191439, 1.5753783E-7, 1.21266198977652, 3.8100425E-7, 2.95558412475556, 0.00000267361964, 0.65655493856853, 0.00000117526411, 7.9725878E-7, 1.44653879032205, 0.12350962616875, 0.1986355, 0.29, 0.374, 13.18921775898520084566596194503171, 1.749642802466284539350613501527963, 3.949803E-8, 0.43824598515343, 8.01361E-8, 7.85740619966256, 1.2573052E-7, 1.99999996578436, 3.421564E-8, 5.61234085923599, 0.00003662289112, 0.00001147748158, 0.53881552244319, 4.946566E-8, 3.11397641995383, 4.524331E-8, 0.156, 1.282, 1.057616623488773747841105354058722, 1.903175257731958762886597938144330, 0.421, 0.2653974548137218738472888233124309, 0.11525389487116, 3.33919000379971, 24.64038727524204702627939142461964, 2.680870353530570617846365775089286, 0.00002005681933, 0.12191477741285, 3.18708522258715, 0.48, 3.388, 0.223, 0.345, 2.793, 10.46099996050197, 4.9353151034869, 0.12270887584296, 0.20000019869679, 0.17286294104397, 4.09415068334512, 0.023164, 0.74268205630276 }

        CuteMarosMeszarosCase.doTest("QBANDM.SIF", ACCURACY.withScale(4));
    }

    /**
     * <p>
     * There are redundant constraints
     * <p>
     * The fixed variables (from presolve) completely zeros the Q-matrix – no longer suitable for ConvexSolver
     * <p>
     * The returned solution comes from LP in initialisation
     * <p>
     * 2022-11-30: Started failing due to changes in parameter scaling. QP problems no longer scaled, but PL
     * are. Problem when QP delegates to LP for initial solution. Hope problem will disappear with new
     * RevisedSimplex.
     */
    @Test
    @Tag("unstable")
    public void testQBORE3D() {
        CuteMarosMeszarosCase.doTest("QBORE3D.SIF", ACCURACY.withScale(6));
    }

    /**
     * <p>
     * Wrong solution!
     */
    @Test
    @Tag("unstable")
    public void testQE226() {

        // CPLEX: OPTIMAL 212.65343305400975 @ { 0.19839999997548, 0.09665930803277, 0.04249999997664, 0.22378911662472, 0.09055801091746, 0.10072139376764, 0, 6.08399995905671, 0.07749999972344, 0, 0.01296245738468, 0, 0.00017099446472, 0, 8.517E-11, 0, 0.04869999981495, 0.39869999982998, 0, 0.394298195933, 2.1984E-10, 10.21400368740736, 0.37645023073007, 4.16201129399917, 1.96419999989225, 0.48378732730074, 2.90799999990717, 1.5086002121292, 0.10189999999222, 0.60449999954682, 0.63168206136057, 0, 11.94034348652553, 0.71046176361395, 0.0525088885978, 9.973E-11, 0.057051546878, 0.2476676391691, 0.66649999963027, 2.1984E-10, 9.11E-12, 0, 0.04953352779705, 7.21E-12, 2.749325E-7, 4.5755E-10, 0.08558106930029, 2.2007E-10, 0.16042105080192, 2.365E-10, 0.18918455513579, 1.36279E-9, 4.423E-10, 0.11713917617051, 1.66415478576733, 7.6436E-10, 0.63439999963732, 0.17371739826086, 1.66415478580811, 0.1140250010492, 0.6833812717794, 0.03269999981499, 4.32577E-9, 0.01217499860187, 2.164E-11, 0.01293619916568, 0.16392755289815, 0.43136858382122, 0.16535031964814, 0.0627503199425, 0.29103619819346, 1.08095700848071, 0.11790966643904, 0.09646238091951, 32.57792753336028, 17.18592414477628, 0.03563117182759, 0.9306239566266, 3.8254, 0.11281495403048, 0.59653213701837, 13.5557060556775, 22.69715474262225, 66.7387517975336, 1.01650478401888, 0.16988415065522, 0.01306661542415, 4.7376E-10, 3.3207E-10, 9.1219E-10, 0.15276794201098, 9.7483E-10, 0.02838897458268, 0.00525555761325, 0.1992209072509, 9.35999998974632, 13.56056623948398, 36.46242765821633, 5.60192872946066, 109.9958063266048, 4.67999999881539, 14.20050143075701, 72.31159733460701, 2.01E-10, 1.5715E-10, 3.6233E-10, 4.9403E-10, 8.8006E-10, 0.23139999231869, 1.56789E-9, 0, 3.58859554747101, 0.3469999955967, 1.00339E-9, 0.15792051377731, 2.122E-11, 7.5341E-10, 5.92493E-9, 1.17741E-9, 8.1556E-10, 5.224E-10, 2.8596E-9, 1.39676E-9, 2.14656E-9, 0.07818174892328, 0.22314819404891, 9.3619E-10, 5.8643E-10, 0.04674538822395, 0.54005063450148, 2.4684E-10, 2.7426E-10, 2.09204E-9, 6.92336E-9, 0.38639786890861, 4.5709E-10, 5.0486E-10, 4.8481E-10, 8.7202E-10, 3.74323E-9, 1.135766E-8, 0.6523822802224, 0.01657224339446, 1.09229E-9, 6.8307E-10, 5.1635E-10, 4.149E-10, 4.3323E-10, 2.6891E-10, 2.0529E-9, 0.41706346096401, 0.08847997165195, 1.073957E-8, 0.73429550529593, 0.02616918594459, 0.52128895722759, 2.7827E-10, 0.66218901647268, 0.03764258284082, 9.0185E-10, 1.1569E-10, 2.8076E-10, 7.577E-11, 0.59206207231578, 1.29888658925779, 1.14142451665259, 0.16922456619406, 2.7232E-10, 2.817E-10, 2.8563E-10, 2.4643E-10, 2.46944E-9, 5.46315E-9, 1.42379352904594, 3.21889E-9, 9.33802E-9, 0.30439698463528, 3.86349E-9, 1.297504E-8, 8.63922E-9, 2.33198E-9, 0.07205323353787, 2.59355E-9, 1.211E-10, 3.0571E-10, 0.00507195384017, 0.03545253027551, 0.36287345095059, 1.50946E-9, 0.88943840779968, 1.009713E-8, 0.05112459433601, 1.30569353938369, 0.09692186053205, 0, 0, 0, 0, 0, 0, 0.14763181437675, 6.7098E-10, 5.7036E-10, 5.35426E-9, 0.04860283517647, 2.75364E-9, 2.99079E-9, 8.64226E-9, 8.78797E-9, 0.09486279910556, 0.2518343422537, 5.13189E-9, 1.031324E-8, 0.14107592785697, 6.537E-10, 0.109, 0.40151695164825, 0.30052427058139, 0.07686701520862, 0.07832961715436, 0.13948304833822, 0, 0.48744822890522, 1.697E-11, 8.976E-11, 0.01990159281103, 0.02133252604259, 0.43048280563693, 0.04213626908479, 0, 0.78569551252371, 0.69, 4.205E-11, 0.31969873803531, 0.29228529836066, 0.62036360317326, 0, 4.379E-11, 0.30050674893643, 0.48360167404306, 0.52014318798227, 0.49944912538753, 2.244E-11, 1.13854167447314, 1.695E-11, 8.957E-11, 0.51381260123254, 2.293E-11, 0.0438975619393, 0.67032265682035, 0.70905060638452, 0.7020742051143, 0.55467032560718, 0, 0.71010086212385, 0.78578094718609, 0.69557470180613, 3.454E-11, 4.204E-11, 0.29227932189647, 0.27183113176096, 1.12545832551836, 0, 0, 2.194E-11, 0.03328670926342, 0.01325268799733, 0.64866722644223, 0.24470440786732, 0.21962526610593, 0.39880914407174, 0.37502174445241, 0.38147701733889, 0.36027011947534, 0.68975003489087, 0.63984962135222, 0.38010493352897, 0.38863702755105, 0.35465874452906, 0.36708596499102, 7.343E-11, 0 }

        CuteMarosMeszarosCase.doTest("QE226.SIF");
    }

    /**
     * <p>
     * The QP solver ends up (de)activating inequality constraints "forever", but does eventually finish with
     * a correct solution.
     * <p>
     * 2022-11-30: Started failing due to changes in parameter scaling. QP problems no longer scaled, but PL
     * are. Problem when QP delegates to LP for initial solution. Hope problem will disappear with new
     * RevisedSimplex.
     */
    @Test
    @Tag("unstable")
    public void testQFORPLAN() {

        // CPLEX: OPTIMAL 7.456631484484689E9 @ { 8101.028950056928, 8101.031690168545, 8211.489646484042, 8211.491497696492, 8141.862628770412, 8141.86387885075, 8141.862831548196, 8141.863676072971, 8141.862968218055, 8141.863539403093, 8141.863060995846, 8141.863446625325, 8141.863123773624, 8141.86338384754, 0, 16283.72650832583, 0, 16283.72650832583, 0, 16283.72650832583, 0, 16283.72650832583, 0, 16283.72650832583, 0, 16283.72650832583, 0, 16283.72650832583, 0, 16283.72650832583, 1.34673187420286, 7008.676449322709, 3.92043328670152, 8926.314778511121, 40.27812101446182, 8889.95709078338, 47.84189054165785, 112.7544491645279, 112.7625469491778, 112.8678221544444, 113.0658689502208, 113.1476716469222, 113.1898460765708, 113.2126523419969, 113.2252518628247, 113.2346330383629, 113.2445187657717, 113.2501463258821, 113.254777238634, 113.2583202673773, 113.2610391220928, 113.2630655262416, 113.26444371583, 113.2653941037125, 113.2699301418847, 113.2774783778651, 113.2912112418911, 429.0215751435542, 435.4776804740434, 2230.245986201453, 5.5558438E-7, 0, 2640, 0, 0, 2799.999998160941, 1.9972453E-7, 0, 0, 0.00000156321092, 3.3658343E-7, 1105.112046342575, 4.0560849E-7, 1270.229547908185, 264.6584054432548, 1627.911465483551, 1.9918777E-7, 1035.142141928088, 136.9463917065832, 2.4464217E-7, 2.062484E-7, 2.3969275E-7, 2.0247238E-7, 1841.868973793353, 1000.005600281863, 155.1665719574183, 339.987204250223, 468.9582442629022, 406.6106226688489, 285.4473093855379, 0.07888333366829, 0.02004475008179, 84.53474746844013, 2.82929992715474, 5.70333657150405, 6.18127028410656, 2.7069172755885, 3.3410015489154, 2.16793545446725, 5.37524822975088, 12.74134349804781, 6.70496116492182, 5.6462515740569, 1.11856798747482, 0.1477144784251, 0.40393889867757, 0.63474772567752, 3.0547164118202, 1.06154743844908, 0.27179740258994, 3.07011699283693, 2.13638108110149, 1.10532152155775, 2.06181914861245, 2.1307292983962, 1.47855463724084, 3.79291017768545, 20.49937437225978, 7.94189020991326, 5.21973060024479, 7.57081585966699, 3.80851428400654, 2.64723655687927, 0.85383158534033, 1.06115723475578, 0.18606819138457, 0.45307643499084, 0.68677817340567, 4.10604030799792, 1.05406303202906, 0.28231484138601, 4.3128670474372, 2.86266084542882, 1.36126273529626, 24.38674219028603, 153.7980981407242, 5.13052641758755, 2.31962876350939, 62.56498668146153, 3.20091478819521, 11.79276907952807, 6.70431403147308, 2.16252488173317, 46.63860640905142, 1.15482054473444, 1.23550335151466, 0.5004426203219, 0.50042754735136, 0.84351514545995, 0.27978386311001, 81.85496686393779, 6.75234575120044, 16.57083314430969, 2.64443279158927, 120.3633224754297, 0.62579898325209, 175.2231639765833, 2.96561494616705, 1.45354985492136, 1.06920772190567, 2.9556845E-7, 2.5314775E-7, 1.8817302E-7, 2.1362041E-7, 1.8708678E-7, 7.79881E-9, 7.58589E-9, 1.021726E-8, 1.357744E-8, 1.09616E-8, 2.016125E-8, 1.708478E-8, 2.690743E-8, 1.533231E-8, 4.601427E-8, 1.946469E-8, 2.850483E-8, 4.696815E-8, 2.625951E-8, 7.38316E-9, 7.59645E-9, 1.604033E-8, 1.018089E-8, 1.356943E-8, 8.487218E-8, 1.098038E-8, 1.935945E-8, 7.380038E-8, 1.665336E-8, 2.592297E-8, 2.885542E-8, 1.597376E-8, 4.772619E-8, 3.442998E-8, 2.031667E-8, 3.280082E-8, 2.798887E-8, 5.301249E-8, 3.528213E-8, 5.73399E-8, 7.38574E-9, 7.61024E-9, 2.332844E-8, 1.021184E-8, 1.35564E-8, 9.792835E-8, 1.099149E-8, 1.973865E-8, 1.2432273E-7, 1.700318E-8, 2.762307E-8, 2.98429E-8, 1.57695E-8, 4.65385E-8, 2.680706E-8, 2.150601E-8, 3.333616E-8, 3.074437E-8, 5.482081E-8, 4.288268E-8, 5.81601E-8, 6.56063E-9, 1.374995E-8, 1.109781E-8, 6.320474E-8, 1.934898E-8, 1.750499E-8, 2.305792E-8, 2.269414E-8, 2.644867E-8, 2.684168E-8, 1.953466E-8, 2.265647E-8, 2.22726E-8, 2.728874E-8, 800.018008958079, 400.0128001184465, 4.95505423595404, 7.81620504451656, 2.17951022212194, 2.56329990241895, 1.20698582216636, 1.70856328247882, 1.20012586368536, 1.09911126933416, 0.86122768048499, 0.5314139526607, 3.74862074788309, 1.47955056989531, 2.83588874587007, 2.04573129464174, 1.05844790471335, 1.68911172668994, 1.66426343173065, 1.35930276722551, 1.6145198602068, 2.45943733653885, 1.97552119233456, 1.063034100824, 1.16018598036004, 1.02347980925147, 0.74091015953625, 0.47442349969807, 0.51082804819302, 2.35360298936382, 1.00693465016443, 3.11788842809612, 3.89411678659608, 2.85780198264916, 1.31105769400445, 7.66667023422962, 10.1777953013769, 3.95911834168262, 1.28087073487355, 2.5745946500462, 1.4293441401513, 1.22942009373346, 1.09591433695159, 0.81603705275319, 1.00598779422749, 0.53489826016287, 0.55011741516737, 167.1295912274128, 2.02544128984236, 7.50919832297814, 28.82192450622951, 14.04229518172064, 4.36869096239663, 2.49758330072299, 2.63869267030415, 1.37453992957666, 1.24247991018477, 0.93561626789991, 0.60394731509419, 0.52283137687977, 3.89700188582891, 64.51403275783223, 1.30599E-8, 2.483518E-8, 1.706709E-8, 2.525561E-8, 1.382112E-8, 3.422084E-8, 1.531664E-8, 2.0479E-8, 2.541268E-8, 1.695514E-8, 4.836462E-8, 3.077969E-8, 1.307624E-8, 2.410485E-8, 4.919965E-8, 1.670666E-8, 2.458102E-8, 2.925997E-8, 1.434717E-8, 3.652477E-8, 3.211011E-8, 1.586293E-8, 2.276418E-8, 2.123188E-8, 2.71873E-8, 2.027149E-8, 2.936059E-8, 4.975158E-8, 4.669546E-8, 9.838156E-8, 1.308389E-8, 2.462185E-8, 6.226897E-8, 1.695619E-8, 2.578195E-8, 3.314663E-8, 1.41702E-8, 3.676296E-8, 2.867984E-8, 1.66939E-8, 2.290544E-8, 2.2215E-8, 2.876922E-8, 2.298812E-8, 2.92128E-8, 7.222349E-8, 5.326545E-8, 9.340819E-8, 2.432189E-8, 2.599291E-8, 2.218206E-8, 2.216001E-8, 2.212217E-8, 2.249258E-8, 1.528393E-8, 1.733489E-8, 1.609896E-8, 1.888995E-8, 2.563006E-8, 3.522376E-8, 1458.150001581819, 649.9944002067176, 127.6884715981013, 58.97973872149157, 173.6545463606988, 274.7882993196201, 118.6732936359694, 0.95273846747783, 185.2321959112869, 0.32456159180956, 85.59213432425163, 2.33635980658171, 6.15083175855068, 1.49318897559316, 14.21437549697702, 1.79900847605109, 9.3485727282815, 2.94680459592649, 101.803794464614, 160.9654685373556, 1.2563245E-7, 1.2492871E-7, 8.874729E-8, 4.92088E-9, 4.34575E-9, 1.156699E-8, 5.665317E-8, 1.884165E-8, 7.606721E-8, 2.744146E-8, 2.794497E-8, 4.174841E-8, 3.127657E-8, 2.651281E-8, 2.491344E-8, 4.481559E-8, 4.748283E-8, 129.7622564815452, 309.9888000476266, 0.80786162809947, 1.92415675223133, 20.05095733286638, 60.09222907204875, 21.93746758696658, 14.54861038644049, 43.9319438098138, 0.01607142541346, 25.00602184467687, 1.93148793404709, 3.68611741153359, 1.33692120864288, 5.51021344729122, 2.01871212935243, 12.88600038151957, 7.45278340745377, 2.46201381034944, 37.5328768052611, 1018.598120289203, 5.6854048E-7, 279.9984001242254, 401.3922791244639, 0, 360.0240000046298, 2.564607E-8 }

        CuteMarosMeszarosCase.doTest("QFORPLAN.SIF", ACCURACY.withScale(5));
    }

    /**
     * <p>
     * The QP solver ends up (de)activating inequality constraints "forever"...
     */
    @Test
    @Tag("unstable")
    @Tag("slow")
    public void testQGROW22() {

        // CPLEX: OPTIMAL -1.4962895331592175E8 @ { 0.01092336267343, 0.00934776575925, 0.01134372035533, 0.00779072165058, 0.00942379121408, 0.01120687841938, 0.0255521933768, 0.01494509369601, 0.31338112799051, 0.01707264738142, 0.00703912903117, 0.0081667995104, 0.00425405358974, 0.0417170697778, 0.00060369085576, 0.00640829420688, 0.01915154483078, 0.10991239247624, 0.01347259470693, 0.09023058307028, 0.00005132187401, 0.00005297637956, 0.00008941928587, 0.00085145264227, 0.00017788540432, 0.00022822244478, 0.00086949377339, 0.00042121056857, 0.00448223516162, 0.00597042465546, 0.00054078721867, 0.22213485980252, 0.00021699773081, 0.00051139385087, 0.00040738459254, 0.00073520043822, 0.00357380654864, 0.00050727758986, 0.00528008233792, 0.00335145709845, 0.08974658895284, 0.00072938893031, 0.0049968730027, 0.0542393391677, 0.19435416453888, 0.13410837502627, 0.06713304882311, 0.15357030592258, 0.04480124438396, 0.14368342467871, 0.14129938330144, 3.141287756189, 0.18955433016136, 0.06582911951277, 0.07747701495065, 0.02886035063834, 0.40187735115605, 0.00047307059146, 0.00331805762322, 0.19464413469854, 2.57508756032247, 0.21430849295372, 0.92830574123495, 0.00003192769655, 0.00002478494535, 0.00010257945859, 0.00003237473598, 0.00002813161071, 0.00009952625181, 0.00000540522499, 0.00002583674743, 0.00000782267959, 0.0000175983499, 0.00001805543836, 2.45250743837204, 0.00003435050235, 0.00001849190541, 0.00000594684327, 0.00005566572388, 0.00002765204058, 0.00000334476148, 0.00014387486446, 0.00002736759564, 2.28267397452346, 0.00002962984281, 0.00003134755945, 10562.66544833373, 519243.8062778839, 1038901.43550721, 5752.665134145675, 40130.98583078455, 6717.046597716356, 12449.80849864241, 11467.49471032375, 62907.99999224375, 16765.07860490911, 31144.64436738373, 15128.31753724335, 6158.191504963077, 146068.5472657679, 348.4103792090308, 3199.525527427574, 21292.5385230229, 33158.99998385556, 11230.4014218273, 375504.2007256302, 0.00010018641529, 433310.5490447825, 999999.999713244, 0.00203307619848, 0.00038347673214, 0.00118766069624, 0.00029333528825, 0.00138828558225, 0.00136788099217, 0.00098611044481, 0.00069197442057, 5221.263004469381, 0.00102206456853, 0.00043465706001, 0.0003597495571, 0.0009221972188, 0.00093353591663, 0.00159144820705, 0.00261079304272, 0.00174186894225, 0.00001412187008, 0.00181809551558, 0.00121612638574, 653665.2832687752, 18082.78408323602, 1066182.621732442, 7837.47051880612, 51776.70248891167, 70742.4280038393, 16732.17758951801, 18321.20305876049, 62907.9999923228, 26174.70732289295, 29524.58900882822, 18462.864541471, 15119.79149922736, 137250.8806552899, 1107.589530438532, 6891.425264796057, 40059.12847807898, 26646.11823291986, 17069.32546816942, 437872.3182153137, 414806.6355057261, 0.0004680731754, 999999.9998665288, 7198.129372827419, 12377.93531430329, 21372.9123700114, 1070.817037683919, 9074.832802937901, 6947.563323652181, 2917.458556796775, 3070.546885078258, 482.2801156409352, 4151.698473013924, 4868.928033132638, 3246.204199282731, 5395.305014847857, 6387.703410108293, 687.4763502410832, 3048.884442785748, 8699.64553341178, 5203.265961771034, 3529.024568859642, 28687.18690097806, 236361.9165826377, 32988.15396217339, 1146667.402844306, 6599.058651533936, 54413.86858840625, 53568.5438039797, 17357.06723422118, 16967.64660891179, 62907.99999231516, 19921.14222130462, 27659.22416233231, 16346.45303493716, 11585.80230766297, 153791.7649240676, 523.4436458909191, 4245.642275246695, 29263.75622993637, 23019.02916627948, 13087.98706487318, 452611.183953455, 0.00010551458392, 0.00014546870334, 999999.9997692154, 146474.5226624669, 37587.56542682562, 124611.7322731999, 1872.535566191744, 21684.89309528458, 28459.51416925487, 7546.32118437945, 5251.580375239683, 8663.584454155696, 8664.58136440188, 6815.203148148367, 4763.341049097535, 8622.375711419354, 28233.01863884753, 787.9234649762691, 3516.621787985086, 14317.81876809525, 7089.829336189686, 4526.274313324956, 105035.4583782708, 915193.8672916897, 24838.87280520084, 946115.9321055717, 5739.04813863524, 33571.69934125655, 60651.62587166719, 9934.682480293723, 10899.03547610146, 62907.99999231492, 21401.59658916514, 20062.11788474938, 11302.21404604161, 7037.963307391098, 90035.8050538665, 364.7667739471295, 3220.69259445226, 27895.01467827311, 20536.34421253339, 10139.80007703041, 328280.6295335511, 742638.4603467854, 0.00047009592797, 999999.9991455463, 11562.71616607051, 56322.81983105875, 32725.15560908664, 2009.265266751215, 16336.70748222886, 10753.30007761377, 6439.096938523518, 5185.855113969431, 757.5893671875843, 6499.606870243349, 8772.927626583287, 6202.88023344366, 9372.224109569501, 12500.3372390429, 840.5888298315176, 3886.589221752576, 12363.98402601788, 9043.088191292174, 4809.054436140633, 52901.38439560813, 235428.062100781, 552429.2543079924, 217414.177302711, 5645.226132365121, 33229.17847807738, 50049.65422252149, 16365.00690922927, 13681.69183245048, 62907.99999230981, 20739.37179234117, 23137.69954225342, 13075.3164029725, 6391.704059528623, 138383.8786359335, 288.6136985002091, 2947.499138558033, 25886.98765108428, 23681.45163124316, 8466.522135020829, 340381.0071150414, 0.00010560482462, 499999.9996180722, 0.00050244894643, 152302.2614648934, 18266.09335442304, 232589.5862748991, 2117.637824493343, 30571.1624246355, 30626.31114549382, 8015.844786030388, 6099.296631345611, 7515.665745040791, 8485.342043347386, 8093.162720986969, 5544.661884710918, 9557.395531717517, 33694.90184960966, 808.0133189998212, 4047.911157702357, 15120.53356767983, 6527.220667728719, 5904.57032865753, 134203.4987402481, 917663.1978103092, 27549.45260936715, 857708.2752615751, 5275.358872839866, 31699.28782130027, 61278.32197510892, 10115.77830267977, 10624.35080148894, 62907.99999230947, 21157.95052877839, 18339.79278956054, 9986.544544627901, 4793.983174515068, 84388.08358904904, 235.1075613822318, 2754.880646203115, 26818.09621737391, 19256.08824486588, 9060.24477802509, 303695.3526092539, 745436.6385389036, 0.00047155094029, 999999.9991474173, 16296.15648779915, 39409.69723910809, 55050.82486759713, 2055.06873989867, 26146.36382808101, 13513.95152151382, 7140.843221386409, 6121.211872585553, 1098.239345575608, 6470.658091602539, 9802.869571897045, 6531.570935650981, 8904.236949573116, 18915.0415031505, 753.7856520858552, 4172.031921302693, 12894.38524595389, 8343.343372478927, 5916.288334254208, 78758.37350216958, 266994.9988427714, 50219.76902311146, 1143642.586564015, 5067.306929143624, 43667.33174232038, 50155.28536671893, 12021.05715791619, 13103.03829157647, 62907.99999230594, 17873.84106343659, 22348.08250631803, 10997.42830873918, 4080.743093785086, 143795.0271505923, 195.4861179238963, 2732.933988341949, 24464.7043547712, 20336.39874378483, 9920.399543746786, 411676.209233597, 0.00010566922036, 0.00014567975166, 999999.9997694595, 178541.3311591274, 79438.10821171373, 156872.3214071196, 2222.722365909102, 28709.27466094624, 30676.69609488488, 8839.435273517527, 6200.833801313524, 9711.805220433798, 9213.807761772456, 8728.384774040409, 5749.874789469585, 7134.062975854788, 39247.25041625678, 566.8966329255774, 3509.376940650958, 15348.63151920966, 8168.928116071036, 5273.371905797085, 124747.8664539515, 889183.4748560329, 48273.61909672635, 960924.9457699466, 4977.411131456462, 33908.53849823109, 59229.4511398697, 8487.50581919708, 10134.20670096381, 62907.99999230572, 20990.93101411227, 18808.46769563103, 8535.288739630407, 3069.157717864077, 93461.79958641916, 155.5354222568879, 2240.729699919674, 25979.61851375929, 20299.86607198268, 9056.165509478893, 334022.903162885, 746365.3506797069, 0.00047219818405, 999999.9998666871, 22411.23973885353, 117927.2822680577, 79760.89126873353, 1946.63493612488, 23334.54777306318, 13570.96886717717, 6897.718139371846, 5537.552077562877, 2071.123314743922, 6843.175850835107, 9384.785064931491, 5464.984922816965, 5035.58324814681, 25192.70642160306, 407.3092022921126, 2912.586544991738, 12058.11526687123, 9293.157208390074, 4855.886496898348, 75913.57634680676, 140179.8341580028, 458059.542113949, 1025297.693178427, 4135.033339170367, 30900.84125634462, 35073.85767957727, 7203.192539375038, 8655.236819427297, 62907.99999230386, 17573.8948171981, 20724.85029035348, 7300.437284190871, 2310.972992448449, 122495.1223729563, 120.0533667913408, 1669.596218391197, 20381.26232417667, 23603.06212044874, 7466.646583628335, 339486.7623479767, 0.00010571567173, 499999.99961907, 999999.9997695253, 103156.1058078944, 669.1434378624357, 66890.06225345422, 1601.446064426117, 15164.47203471569, 28056.90433858215, 4130.961446839788, 3778.550621632188, 6810.866666617932, 7224.934257537602, 3992.298120678033, 2352.293419000819, 2746.222955816127, 15241.4994711135, 211.9378549719587, 1725.901312852695, 11266.05600387758, 3204.930468382939, 3144.59450935877, 49312.9748757221, 962167.6116734821, 898.6450038853826, 967777.0915529643, 2481.883866833175, 23478.30687007277, 51531.46110077104, 4130.413117205142, 4827.16245149239, 62907.99999230354, 15363.98701483826, 10744.76745881746, 3240.917070912627, 1635.989708279539, 81546.71042599264, 85.53493088126956, 951.0326896570838, 15512.95828238881, 13365.07404258456, 3956.037874506626, 323926.3114925628, 746894.6814559387, 0.00050622346181, 999999.9991500432, 0.00036873231709, 0.00053101392797, 0.00113973293437, 0.00031475183602, 0.00058985580206, 0.00046239114116, 0.00015690104479, 0.00022121500514, 0.00002249397819, 0.00056931382887, 0.00058701635515, 0.0004031696229, 0.0004068968108, 0.00034940826656, 0.0043602174836, 0.0021893656348, 0.00135154637245, 0.00180478342064, 0.00315484573304, 0.00174401844577, 300445.5046732776, 91497.71128442053, 265114.7546760643, 7626.998989015089, 45296.32573927825, 66120.87055630606, 24322.98142653654, 20331.72681079306, 62907.99998976471, 23621.07642653597, 26994.43629502615, 19770.61896685819, 17580.79183402192, 132660.6885779411, 1068.365791361485, 6230.988787738454, 36577.88433713194, 22914.64119204672, 14231.0998279877...

        CuteMarosMeszarosCase.doTest("QGROW22.SIF", ACCURACY.withScale(6));
    }

    /**
     * <p>
     * The QP solver ends up (de)activating inequality constraints "forever"...
     * <p>
     * Initial solution very close to 0.0 and therefore ends up with "Freak solution!"
     * <p>
     * Q matrix is not SPD, Cholesky fails and the matrix is patched. Both Q and C are very sparse. Patching
     * probabbly changes the structure Q significantly.
     * <p>
     * Objective function (Q matrix) adjusted by -1
     * <p>
     * With adjustments turned off, and no inequalities active, the full KKT system is unsolvable!
     */
    @Test
    @Tag("unstable")
    public void testQGROW7() {

        // CPLEX: OPTIMAL -4.2798713862415135E7 @ { 0.0160678086992, 0.00540511207202, 0.0255612422291, 0.02011032774044, 0.01172486891935, 0.01442934821509, 0.04597939210731, 0.04048122350345, 0.98985573211076, 0.0443236087681, 0.01504591809461, 0.01962644026264, 0.00735240101566, 0.09299137179792, 0.0002267537129, 0.0019865332159, 0.03236585520176, 0.02150981595949, 0.01749479008516, 0.20119822776542, 0.00001534431094, 0.00001787432018, 0.00003516993267, 0.00002010026826, 0.00002557615674, 0.00003918919318, 0.00002025811313, 0.00001494620355, 0.00001266136986, 0.00000704892394, 0.00001058294691, 0.71092048966596, 0.00001175813174, 0.00001570979249, 0.00001473047463, 0.00000394659136, 0.00002053261834, 0.00002273184948, 0.00000364452797, 0.00001959131791, 0.00002316330478, 0.00001214061549, 0.00002324111414, 293407.886370509, 88911.44066983282, 1126478.644017619, 7782.401352072035, 58935.84400141655, 50305.76379745777, 20453.45865023398, 20019.44403342968, 62907.9999998116, 23973.58428122677, 33127.21176821922, 20327.39128612429, 12862.4740389652, 158867.1459566136, 731.2894333404639, 6090.817140877632, 36853.1940386066, 29617.31060757102, 16944.69182394585, 454088.6599528854, 145436.9621207764, 0.00001144147244, 999999.9999783719, 34078.16967067493, 71654.20650983986, 81000.55949648975, 1073.977082084232, 16212.00574733186, 14864.16770825047, 5723.182542831339, 4164.070796904228, 3702.052624309026, 6210.460092035213, 5089.89154579996, 3237.759622293091, 3797.825609850296, 19059.22202395755, 300.2876883533582, 2184.24698160291, 9979.385817092232, 5852.295177558273, 3091.615127193328, 60684.83802394568, 116913.8200200692, 513839.8435537839, 1010663.971576223, 5780.934323257901, 36775.21985711016, 32158.26311841169, 9554.452331044147, 11010.87323189131, 62907.99999981168, 18765.3413092735, 27828.62158122184, 12386.51430042435, 4251.177120997063, 136218.0563778279, 232.064104670629, 2457.874385384077, 23193.62959089545, 28507.08842795989, 10375.47369709271, 356645.8360878981, 0.00000259061303, 499999.9999902759, 999999.9999943228, 99306.14710584197, 780.4802615177505, 52784.00998348656, 1341.564374172005, 13354.57553957497, 28331.44010919421, 3829.973766018742, 3613.388697101487, 6367.920081509713, 6865.270102669235, 3516.606042246835, 2147.247058575894, 2548.356889726156, 12777.17794766353, 199.2432329123793, 1610.026197430044, 10360.16661355898, 2168.650911562752, 2859.932634905967, 39189.71623352172, 931899.0961102699, 914.1366365636297, 982489.3925249376, 2904.102219726574, 25768.93738036154, 48810.55762994168, 4607.026150741087, 5273.793851146557, 62907.99999981171, 15514.67880932511, 11981.88018572593, 4048.229915328746, 2083.192864029808, 86729.04320394117, 110.6614130940075, 1173.766751988793, 16446.66822421812, 14838.4956569021, 4651.865796467362, 337135.4242616647, 722883.8200050341, 0.00001195869748, 999999.9999967279, 0.00000878316154, 0.00001197227103, 0.00002715171858, 0.00000748146059, 0.00001397775649, 0.00001101197491, 0.00000369699003, 0.00000528047011, 5.367859E-7, 0.00001361576161, 0.00001395698541, 0.00000950853677, 0.00000900939314, 0.00000825994961, 0.00001910426118, 0.00004565060917, 0.00003221320807, 0.00004339908463, 0.00007145619684, 0.00004123284086, 174450.8592496273, 57679.74111725319, 1115657.25850949, 7004.400655707387, 52960.91379094658, 50810.16070449572, 15757.4256138412, 17627.09946725865, 62907.99999974819, 21515.32276976777, 29353.60765315151, 16816.11849290629, 9770.647415459323, 141544.621087154, 554.3996977189199, 4913.176973522552, 32701.9153598568, 26053.54837414548, 15221.13646187513, 405393.1243774313, 0.00000366856422, 0.00000517227908, 999999.9999953067, 97959.86958530694, 45751.78375343201, 71953.79296626331, 1167.127775315283, 12601.72952183883, 23970.35876339684, 3929.286902976307, 3304.084273335831, 10732.61146929366, 6756.97397128886, 4029.409414581573, 2147.910994164142, 2254.256872682216, 15216.92553781273, 170.0020534117123, 1438.091306289277, 9751.612505154209, 4670.611778432105, 2477.648988875221, 47615.10095981009, 978072.9946297683, 77185.29860549807, 963989.0727562256, 3465.326245582558, 27314.44397452418, 58308.53371635134, 5728.8251059342, 6416.662745867156, 62907.9999997481, 17624.70599657768, 13096.67537064033, 4933.168957190171, 2828.258243460317, 92817.39840590394, 151.2311260421249, 1449.915991731854, 19433.61992975965, 17644.46724982956, 5548.996752366262, 347118.612915701, 749999.9999912212, 108866.6891972558, 999999.9999881474, 0.00000367071516, 0.00000485465097, 0.00001143401107, 0.00000313875968, 0.00000584241026, 0.00000460188605, 0.0000015477081, 0.00000219154176, 2.2395467E-7, 0.00000570347526, 0.00000589022978, 0.00000399287656, 0.00000374162626, 0.0000034679498, 0.0000226339107, 0.00001848415107, 0.00001352628981, 2702.415668852725, 0.0000298607142, 0.00001745234113, 64279.83362369857, 598682.2821791811, 1040066.58580558, 6245.985619353918, 41615.18056239079, 11644.32330703796, 13901.78501719943, 12343.27199938739, 62907.99999867937, 18987.99500963988, 33492.20831322629, 16359.35159163764, 6697.081678680302, 157290.6844652724, 373.551531723064, 3417.770465484346, 23853.1440610008, 33158.99997953509, 11560.73548164061, 401641.2588044541, 35042.40998165328, 499999.9999974957, 999999.9999767065, 0.00001546134006, 0.00001946656926, 0.00004650678408, 0.00001253035272, 0.00002434200913, 0.00001939508409, 0.00000651194163, 0.00000917543798, 9.4715531E-7, 0.00002354863256, 0.00002314348178, 0.00001592761416, 0.00001570567294, 0.00001440981269, 0.00005492850621, 0.0000746711186, 0.00005535808909, 0.00001444362118, 0.00011187335397, 0.00006972133341 }

        CuteMarosMeszarosCase.doTest("QGROW7.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testQPCBLEND() {
        CuteMarosMeszarosCase.doTest("QPCBLEND.SIF");
    }

    /**
     * <p>
     * <p>
     * 2022-11-30: Started failing due to changes in parameter scaling. QP problems no longer scaled, but PL
     * are. Problem when QP delegates to LP for initial solution. Hope problem will disappear with new
     * RevisedSimplex.
     */
    @Test
    @Tag("unstable")
    public void testQPCBOEI1() {

        // CPLEX: OPTIMAL 1.1503914011360059E7 @ { 9.99999995031002, 7.791392E-8, 3.21488E-9, 1.39748E-9, 11.99999939624776, 6.2718683E-7, 3.1983E-10, 21.26764076791552, 5.72900505980431, 13.00335419314499, 44.48841944643153, 64.51158056025938, 3.04304E-9, 1.48528E-9, 1.49553E-9, 9.9217E-10, 125.9324568334024, 123.817805135286, 121.7734267502161, 119.7950479294157, 116.6812633545326, 2.09401353823544, 7.21710510877051, 13.68888137222515, 10.92403642475223, 22.075963583979, 6.4521E-10, 92.79824341363998, 100.2017565885183, 11.33279776911655, 12.66720224559475, 35.05882362646948, 33.94117638575626, 243.0000000022953, 62.08208127731879, 69.7334788582842, 67.20394978557323, 67.98049009743693, 0.08658930157485, 1.00083E-9, 0.08636768255954, 1.58848E-9, 33.35221349383538, 47.06954988276664, 19.5782366362395, 16.31910794584536, 14.68089206126283, 2.7889E-10, 91.99497935848321, 24.00502064220582, 3.3719E-9, 166.5846673456689, 165.4153326573304, 33.00000000350173, 6.8323E-10, 23.57087466167962, 23.33174874897776, 23.0973765981399, 105.2222513492115, 104.8348381656218, 103.8566489778462, 103.2261560323419, 102.0103980187727, 103.3071288958122, 1.146369E-8, 1.229113E-8, 11.56050618469905, 96.98207237293077, 474.7189227611942, 472.8921779344246, 469.2443691813489, 167.9999999980633, 167.999999994769, 448.6310028629559, 454.5135272684068, 7.1169E-10, 9.00000000418145, 17.99999998118777, 1.524908E-8, 9.98508E-9, 7.28927055647889, 28.71072943259439, 2.33306E-9, 86.7022286344281, 86.2977713559075, 1.320707E-8, 0.01607836200062, 0, 0.01594541020659, 0, 82.00000000706552, 45.00000000257781, 0, 9.739E-10, 5.52006598721639, 5.47993403109231, 34.03618216901993, 32.11155791338731, 31.85225992618677, 155.0000000019229, 13.75163977449646, 11.62198155554617, 11.62637868433572, 0, 32.00000001254435, 21.00000000441145, 0, 1.21754E-9, 127.766772154534, 127.2332278474634, 60.00000000163732, 240.6264610653763, 239.069482103351, 235.655560730421, 234.209002273403, 1.358643E-8, 155.4394941761482, 169.99999963946, 1.40736E-9, 1.94394E-9, 21.06438512034152, 20.93561490730234, 1.641977E-8, 2.230729E-8, 51.1526198146586, 50.84738015762895, 65.8172771661544, 65.42908038813154, 65.04543600122651, 64.46761104130177, 64.29156709494313, 63.92110864129774, 0, 63.19293024864919, 62.83498942680888, 1.04574E-9, 1.24137E-9, 1.6026E-9, 3.9441E-10, 4.44409E-9, 0.27123524324273, 14.72876475324674, 4.7392E-10, 197.0000000050402, 70.99999999546945, 2.364755E-8, 1.966522E-8, 49.63020458353161, 49.36979537493308, 44.00000000458784, 74.00000000317814, 68.36332670718517, 68.43519415719116, 71.33836861020592, 61.86311052825101, 91.27349940765815, 94.66316471531186, 92.64134902992996, 92.4219868486454, 472.7426589596506, 471.2628247262728, 470.3352664597286, 468.8163728074495, 466.8428770471419, 127.5025768791777, 124.8093798019415, 124.6880433205863, 112.2721711110071, 111.7389226599546, 108.9889062308845, 63.19107538803937, 62.89490853821336, 62.6017532482663, 61.31226282675581, 9.00000000706823, 0E+5, 1.01198E-9, 0, 8.6021E-10, 2.8385E-10, 23.00000000080425, 1.29359E-9, 2.23337E-9, 9.0000000068062, 0, 6.90343722155338, 6.78213877383671, 7.31442401032963, 2.49302E-9, 13.99999999128717, 1.302101E-8, 4.1831E-10, 21.00000000040285, 35.00000000047873, 6.4311E-10, 7.7824E-10, 11.00000000349982, 8.0704E-10, 1.06866E-9, 0.02537457101604, 9.7825E-10, 1.07658E-9, 1.06045E-9, 1.87304E-9, 1.12727E-9, 9.3998E-10, 1.54999E-9, 1.16993E-9, 1.80768E-9, 2.15863E-9, 1.39456E-9, 2.57304E-9, 3.783E-10, 5.6332E-10, 6.7716E-10, 18.00000000039969, 11.00000000045074, 6.9207E-10, 5.5845E-10, 15.00000000037741, 22.00000000034261, 1.32652E-8, 7.6322E-10, -80.84702101054647, 1.03038E-9, -56.09144844623896, 1.07396E-9, -46.99999999276265, 1.90519E-9, -28.50142173983918, 1.59999E-9, -33.89533984571521, 2.10739E-9, -25.51130731292728, 4.2752E-10, 0.38846158057352, 0.10933265052562, 3.7966E-10, 9.13153E-9, 2.29987447472519, 4.4129E-10, 0.19616766745114, 0.32995166526547, 0.62785413626693, 2.19785373733235, 8.1842E-10, 0.14247058640015, 0.67260080575938, 0.1849286082003, 2.3494E-10, 0.10980653915355, 0.18240594789672, 0.25529944669054, 1.64120378443702, 1.83542E-9, 0.89929370444652, 1.35355578303236, 3.54661857757948, 0.3243264559229, 0.68994970188446, 1.90081E-9, 0.10647404140456, 8.333354E-8, 1.89972E-9, 3.58693E-9, 4.31727E-9, 1.2224278E-7, 3.486184E-8, 1.15120714937876, 1.69176629274322, 1.02373397487963, 2.02341E-9, 2.4377808E-7, 7.473815E-8, 0.07762066654298, 0.22081051495665, 0.20438401402981, 2.11441E-9, 0.31743966543039, 2.28211E-8, 7.2023E-10, 1.2999528E-7, 2.86311E-9, 0.81507125948922, 2.74926E-9, 0.00146400790463, 0.11504742382753, 0.86633658042622, 2.86166E-9, 0.6812440948913, 1.86227967718939, 1.26139165214149, 2.99069E-9, 0.79290640365282, 1.88288702391594, 1.49322988614349, 2.58135E-9, 0.26874853514631, 0.51690771241919, 1.08185519057482, 8.58215E-9, 2.572769E-8, 2.69994E-9, 0.46182607345639, 0.4903635631384, 2.23492457274258, 5.70111E-8, 0.48834899200081, 2.11517E-9, 1.42375622994194, 0.58411896603296, 0.01764534186082, 0.40482582557194, 3.78588E-9, 0.43633299339074, 0.00000223169599, 0.63805511794031, 0.17213087408795, 0.49158526712876, 2.40737E-9, 0.84005956435175, 5.143389E-8, 0.60012551610287, 0, 1, 9.279E-11, 8.637E-11, 1.00000, 0.999999999976, 0.99999999978419, 0.99999999957706, 0.99159987472002, 0.99999999793257, 0.99999999843747, 0.2339932732146, 0.40245134508481, 8.1507E-10, 0.91890069671158, 7.46683E-9, 4.9794E-10, 1.51131857982447, 2.11378007171142, 0.50788644781501, 0.2423297779294, 9.8311E-10, 0.68384832273226, 7.80869E-9, 3.9485E-10, 7.32651E-9, 1.40597977729936, 2.86914990070333, 4.31097176968539, 0.11663719862359, 0.62206477442333, 0.0000012139292, 1.07467332069314, 2.07968563042786, 1.92090649232803, 4.0072914230868, 1.79692016290368, 1.99321660158903, 2.65200906935908, 0.38291288388406, 0.21968553465715, 1.40354463608176, 0.04578645704816, 2.10008E-9, 5.3285E-10, 4.2918E-10, 0.42311995908384, 0.18631044413709, 1.00000000016129, 0.94410079467102, 0.99999999951667, 0.72632184213273, 2.78155E-9, 0.38353971754261, 1.17225252594894, 0.37459753206634, 0.90207796634996, 6.77268E-9, 0.90153673585688, 1.10813926311388, 1.62815036054942 }

        CuteMarosMeszarosCase.doTest("QPCBOEI1.SIF", ACCURACY.withScale(7));
    }

    /**
     * <p>
     */
    @Test
    public void testQPCSTAIR() {
        CuteMarosMeszarosCase.doTest("QPCSTAIR.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testQPTEST() {
        CuteMarosMeszarosCase.doTest("QPTEST.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testQRECIPE() {
        CuteMarosMeszarosCase.doTest("QRECIPE.SIF", ACCURACY.withScale(7));
    }

    /**
     * <p>
     * TIMEOUT
     * <p>
     * Phase2 of LP initialisation never finishes
     */
    @Test
    @Tag("unstable")
    public void testQSCAGR7() {

        // CPLEX: OPTIMAL 2.686594860149913E7 @ { 45.43999998915281, 112.5600000108472, 6.24299E-9, 157.999999993757, 112.5600000108472, 6.24299E-9, 1422.887905822863, 1183.251746814557, 1429.998088715901, 1263.862357675308, 1554.45837981199, 0.00001321161176, 1599.999900971736, 5.24642E-9, 112.5600000061345, 6.24299E-9, 375.2, 349.1203811607291, 2399.997706459081, 1199.998370855172, 9.245707E-8, 187.5999999075429, 187.5999965952228, 0.00000340477715, 187.5999999075429, 187.5999965952228, 1908.838588164798, 2566.669983792553, 312.5741701742359, 511.9173895555093, 488.8078169735854, 0.00001363341353, 1599.999868313373, 1.0610934E-7, 187.5999998018866, 187.5999965952228, 375.2000000061345, 373.6976996427193, 2399.886710667309, 999.035780688778, 1.2263666E-7, 187.5999998814288, 187.5999964451992, 0.00000355841517, 187.5999998814288, 187.5999964451992, 1933.328627251192, 2566.669983106868, 133.8106661388772, 666.190921619195, 614.8750159997721, 0.00001397640653, 1599.999801884112, 1.4470022E-7, 187.5999997367286, 187.5999964451992, 450.2399998070222, 274.4794277040647, 2185.259510028002, 952.3467232646766, 1.7154207E-7, 225.119999731969, 225.1199961786484, 0.00000372486713, 225.119999731969, 225.1199961786484, 1933.329226714584, 2566.669982330036, 0.04025564290152, 799.9609473153006, 1471.831814345166, 0.00001433422762, 1599.999587997289, 2.1954715E-7, 225.1199995128776, 225.1199961786484, 502.767999601734, 114.8540759589892, 1735.067816994186, 896.8516926841379, 2.4522518E-7, 251.3839995560909, 251.3839958862647, 0.00000391460228, 251.3839995560909, 251.3839958862647, 3091.710201050967, 2566.669981489339, 0.02813270562477, 799.9725129950505, 2107.720772509092, 0.00001473187623, 441.6191717590989, 3.5414227E-7, 251.3839992019486, 251.3839958862647, 577.0575992341714, 107.8502795141661, 1232.333576640473, 741.565029071577, 5.3913825E-7, 288.5287990784022, 288.5287954700455, 0.00000414704029, 288.5287990784022, 288.5287954700455, 3533.329113884595, 2566.669980553052, 0.02177606983357, 799.9787356835768, 2981.321440385775, 0.00001516395621, 0.00039380897883, 0.00000166082256, 288.5287974175797, 288.5287954700455, 655.3243186658685, 103.6284182901286, 655.3021086913802, 445.134350278062, 7.3016825E-7, 327.662158602813, 327.6621565953777, 0.00000273760427, 327.662158602813, 327.6621565953777, 3533.329416672271, 2566.669979209763, 0.01854699916729, 799.98186338351, 3910.786388441234, 0.00001585809271, 0.00019373528669, 327.662158602813, 0, 327.6621565953777, 747.2558204836877, 0.00000462099703, 0.00004642564855, 0.00003064326919 }

        CuteMarosMeszarosCase.doTest("QSCAGR7.SIF", ACCURACY.withScale(7));
    }

    /**
     * <p>
     * ojAlgo fails to solve the problems – returns state FAILED
     * <p>
     * There are redundant constraints
     */
    @Test
    @Tag("unstable")
    public void testQSCORPIO() {
        CuteMarosMeszarosCase.doTest("QSCORPIO.SIF");
    }

    /**
     * <p>
     * Wrong solution!
     */
    @Test
    @Tag("unstable")
    public void testQSCTAP1() {

        // CPLEX: OPTIMAL 1415.861117430658 @ { 0.99999999906806, 6.1988E-10, 3.1206E-10, 3.94994E-9, 0.99999999924071, 4.9895E-10, 2.6034E-10, 3.52657E-9, 0.99999999933781, 4.2118E-10, 2.4101E-10, 3.53791E-9, 0.66666666795504, 0.33333333106026, 9.847E-10, 4.14304E-9, 0.99999960350635, 3.9585452E-7, 6.3913E-10, 3.94605E-9, 0.50000060150999, 0.49999939574483, 2.74518E-9, 3.55292E-9, 0.9999999987118, 9.4129E-10, 3.4692E-10, 3.45123E-9, 0.9999999994353, 3.2775E-10, 2.3695E-10, 6.17984E-9, 0.99999999959616, 2.2835E-10, 1.7549E-10, 5.47209E-9, 0.99999999811899, 1.33307E-9, 5.4793E-10, 5.97207E-9, 0.99999999814973, 1.20128E-9, 6.49E-10, 1.074297E-8, 0.99999999939116, 3.8243E-10, 2.2641E-10, 8.75206E-9, 0.99999999797429, 1.34912E-9, 6.7659E-10, 1.437808E-8, 0.99999999871677, 7.7828E-10, 5.0494E-10, 1.422958E-8, 0.99999999813019, 1.23728E-9, 6.3253E-10, 1.295097E-8, 0.33333333564405, 0.66666666298631, 1.36964E-9, 1.173462E-8, 0.3333333356098, 0.66666666280794, 1.58225E-9, 1.023898E-8, 3.1219E-10, 2.10753E-9, 0.99999999758028, 9.19475E-9, 7.2347E-10, 0.99999999689126, 2.38527E-9, 5.35017E-9, 0.86478806989088, 0.1352119294934, 6.1572E-10, 6.66308E-9, 0.99999999964608, 1.9239E-10, 1.6153E-10, 1.126578E-8, 0.40563519582714, 0.59436480330115, 8.7171E-10, 5.83489E-9, 0.99999999592141, 2.98815E-9, 1.09044E-9, 1.5396E-8, 0.90000012006138, 0.09999987878743, 1.15119E-9, 2.207155E-8, 0.70241900125823, 0.1894159943603, 0.10816500438147, 2.20208808558282, 0.94181404611254, 0.03257085355234, 0.02561510033512, 1.89589759885098, 0.62364627634816, 0.24640480577989, 0.12994891787195, 1.90122417134015, 0.39356257906889, 0.40179107902644, 0.20464634190467, 2.59742721534394, 0.38276771837942, 0.42420835764248, 0.1930239239781, 2.10969726824231, 6.3817E-10, 1.73993E-9, 0.9999999976219, 1.7124908814179, 5.7613E-10, 1.68149E-9, 0.99999999774237, 5.00000006920478, 0.66666666601536, 0.33333333266804, 1.3166E-9, 1.277957E-8, 0.99999999964932, 1.9209E-10, 1.5859E-10, 1.041724E-8, 4.5374E-10, 7.09267E-9, 0.9999999924536, 1.133363E-8, 0.60375679596111, 0.39624320261875, 1.42014E-9, 1.302328E-8, 0.11887284142794, 0.88112715822007, 3.5199E-10, 6.95453E-9, 0.6315776800415, 0.22977985088739, 0.13864246907111, 2.85658350035829, 0.9193848673532, 0.04547435378865, 0.03514077885814, 2.95517298247334, 0.39180318782263, 0.37356705239493, 0.23462975978245, 2.5976159566031, 0.49583695079652, 0.31098353550248, 0.193179513701, 3.76467428106816, 0.35264199758747, 0.4102531277547, 0.23710487465783, 3.4132050868202, 0.11624184748584, 0.34794497560729, 0.53581317690687, 3.08413608108367, 0.02798580236581, 0.07720083836299, 0.8948133592712, 12.2813591013543, 0.37779521571066, 0.44179662393896, 0.18040816035038, 2.54214082142547, 0.99999999920538, 4.4826E-10, 3.4637E-10, 1.675488E-8, 1.03159E-9, 3.4564E-9, 0.99999999551201, 1.53569044775445, 1.63706E-9, 0.99999999745673, 9.0621E-10, 7.59062E-9, 1.1809E-10, 1.018E-9, 0.99999999886391, 1.37816E-8, 0.64443198034752, 0.22537435160695, 0.13019366804553, 2.43917225090318, 0.89237334033303, 0.06167836209425, 0.04594829757272, 3.17436568691717, 0.3682846485066, 0.41865503092862, 0.21306032056478, 1.48717196189188, 0.54075231047581, 0.29339144137978, 0.16585624814441, 3.07777669710251, 0.35976579928828, 0.41686612721561, 0.22336807349612, 3.05075605250201, 0.11128859952313, 0.34636373864318, 0.54234766183369, 2.80378225421827, 0.03037856492232, 0.08432335392665, 0.88529808115103, 14.5208800654608, 0.10974632294294, 0.57132839586766, 0.3189252811894, 3.25718698409045, 0.90901858422758, 0.05029004830201, 0.04069136747041, 2.33593170148517, 0.34669359720071, 0.4020375681664, 0.25126883463289, 1.70455485567139, 4.1772E-10, 0.50000000248241, 0.49999999709987, 2.18097E-8, 8.236E-11, 2.6619E-10, 0.99999999965145, 4.4459363782478, 0.64122766767331, 0.24874647257967, 0.11002585974701, 1.14146816515753, 0.80712912591925, 0.1225572040755, 0.07031367000526, 1.81514759294991, 0.51734530440993, 0.48265460521321, 9.037685E-8, 4.190667E-7, 0.58209163167627, 0.30907496214747, 0.10883340617626, 1.52495647777444, 0.40911001048356, 0.44001504892663, 0.15087494058981, 1.7725573021115, 0.07800887774581, 0.27046592797755, 0.65152519427663, 2.26726360955571, 0.03065253467781, 0.0871422314805, 0.8822052338417, 12.18408827197171, 0.05320654759639, 0.59250149223382, 0.35429196016979, 2.04398317783263, 0.90310414580573, 0.05698234287933, 0.03991351131494, 1.34270782485104, 0.79079299106124, 0.12532760357666, 0.0838794053621, 2.04041038589473, 3.8072E-10, 1.01021E-9, 0.99999999860908, 1.28940822773864, 1.0301E-10, 2.427E-10, 0.99999999965428, 10.8039323305139, 0.65068852395697, 0.34931144189034, 3.415269E-8, 1.6814564E-7, 0.76680291008354, 0.23319706963395, 2.028251E-8, 1.6021366E-7, 0.99999992866253, 4.906309E-8, 2.227438E-8, 3.4990506E-7, 0.64587869010237, 0.35412129379408, 1.610355E-8, 1.216737E-7, 0.65377502170434, 0.34622495950768, 1.878798E-8, 1.4234033E-7, 0.1017804518325, 0.29294491677498, 0.60527463139251, 1.07249345585905, 0.03531428291882, 0.11145066752908, 0.8532350495521, 1.51991090231141, 0.06052739870261, 0.715090241098, 0.22438236019939, 1.21827836952003, 0.92714919311284, 0.07285078915923, 1.772793E-8, 1.4746467E-7, 0.72781890167799, 0.19159540783974, 0.08058569048226, 0.74760298759323, 4.2635E-10, 2.37462E-9, 0.99999999719903, 5.80087537506386, 1.9477E-10, 1.12393E-9, 0.99999999868131, 11.28922772513682, 0.99999993824658, 4.576974E-8, 1.598368E-8, 1.7510344E-7, 0.99999998428855, 9.90249E-9, 5.80897E-9, 1.538683E-7, 0.99999996075018, 2.823204E-8, 1.101777E-8, 1.4150811E-7, 0.99999998470983, 9.49297E-9, 5.7972E-9, 1.6878264E-7, 0.99999998693837, 8.02902E-9, 5.03261E-9, 1.5748482E-7, 0.24080191104018, 0.7591980749725, 1.398732E-8, 7.348778E-8, 0.0715957627639, 0.92840422468186, 1.255424E-8, 7.043155E-8, 0.34813934926534, 0.65186064345468, 7.27998E-9, 6.250694E-8, 0.99999999326307, 3.76662E-9, 2.97032E-9, 1.731129E-7, 0.79256147982487, 0.20743850954028, 1.063485E-8, 6.163347E-8, 6.8723E-10, 2.85954E-9, 0.99999999645323, 0.92561484546852, 4.022E-10, 4.34568E-9, 0.99999999525213, 1.06815392426836, 0.9999999607528, 3.025443E-8, 8.99276E-9, 8.830614E-8, 0.9999999870994, 8.16133E-9, 4.73926E-9, 1.2399417E-7, 0.9999999752991, 1.752096E-8, 7.17994E-9, 1.0297386E-7, 0.99999998800983, 7.48476E-9, 4.50541E-9, 1.2972243E-7, 0.99999998887288, 6.84998E-9, 4.27714E-9, 1.3284584E-7, 0.99999998631878, 8.97589E-9, 4.70532E-9, 1.1934213E-7, 0.99999998049185, 1.347326E-8, 6.03489E-9, 1.4355319E-7, 0.99999999106916, 5.29691E-9, 3.63393E-9, 1.3720697E-7, 0.999999994472, 3.23602E-9, 2.29198E-9, 8.462887E-8, 0.99999997379042, 1.841533E-8, 7.79425E-9, 1.2841467E-7, 1.32649E-9, 0.99999999105098, 7.62253E-9, 3.258716E-8, 5.94657E-9, 0.91666676122127, 0.08333323283217, 4.28757E-8, 0.99999998590183, 9.45368E-9, 4.64449E-9, 1.0671591E-7, 0.99999999087632, 5.68524E-9, 3.43844E-9, 1.0376695E-7, 0.99999998744383, 8.39745E-9, 4.15872E-9, 9.41494E-8, 0.99999998973683, 6.41059E-9, 3.85258E-9, 1.2047328E-7, 0.9999999907831, 5.74016E-9, 3.47674E-9, 1.0518115E-7, 0.99999998731183, 8.48899E-9, 4.19918E-9, 9.465294E-8, 0.99999998202044, 1.211106E-8, 5.8685E-9, 1.6793077E-7, 0.99999999179128, 4.67027E-9, 3.53845E-9, 1.7075792E-7, 0.99999999019997, 5.57163E-9, 4.2284E-9, 1.9153028E-7, 0.99999997838684, 1.446387E-8, 7.14929E-9, 1.6161622E-7, 0.99999998562514, 9.17154E-9, 5.20332E-9, 1.4530051E-7, 0.98926379485286, 0.00637875030074, 0.0043574548464, 0.58246490169965 }

        CuteMarosMeszarosCase.doTest("QSCTAP1.SIF");
    }

    /**
     * <p>
     * Returns an incorrect solution
     */
    @Test
    @Tag("unstable")
    @Tag("slow")
    public void testQSHARE1B() {

        // CPLEX: OPTIMAL 720078.3184694768 @ { 333.8999, 37.46085934372832, 65.54637077660308, 148.1051336737728, 3.127423E-8, 2.278724E-8, 72.8, 2.78790021524773, 12.129447733567, 9.42087E-9, 1.075367E-8, 8.99822741634234, 647.6903973349696, 36.15835833438015, 60.34742145671689, 202.8948662770234, 1.0399414E-7, 32.54851317422205, 185.9999997238467, 7303.007189991704, 37960.74956115764, 10777.56328688469, 889506.2876017037, 178.29865336696, 73.18279942033763, 1012.612055460439, 214.9999998539775, 4893.997966734743, 603.9999996592173, 3404.668729439141, 222.6762174232982, 66.51386737926953, 25.05052049517707, 215.1127790261638, 162.9999998885236, 1040.612334593366, 82.8, 5.53054465555529, 27.06769951189914, 0.00003296522839, 50.20159953983185, 0.00000453696223, 148.4, 22.39922325218321, 0.00004571405555, 106.5315454715923, 0.00000298600365, 747.8786913219764, 46.7503753659731, 58.02368723918124, 628.2668549603185, 0.00001567776432, 2115.599902016661, 177710.395013591, 12370.36762160593, 3634.031168506993, 132.5126746056218, 2.825793E-8, 142.1782776164198, 82.52138122647733, 945.0257685928581, 2.1187846E-7, 2020.029728160476, 79.225570397044, 120.9999976227638, 0.00000378637367, 91.82172175398595, 880.1247123569613, 202.002972788626, 574.9999989590176, 283.4787913219764, 36.59288210664387, 3.978178E-8, 278.9999999128518, 247.9999999509883, 3358.444244228303, 4144.872644532776, 13154.01466263757, 7.5679896E-7, 5.3236409E-7, 6.3075185E-7, 6.2959424E-7, 0.00000191390626, 172.9085638012803, 108, 11306.66180526505, 2864.998359744063, 63009.48056997929, 410.2903973349696, 4983.007192008365, 5342.793046609027, 0, 417.8297281604755, 1.099055E-8, 212.8872209738362, 1085.726250435192, 0.00011365094062, 16.32397901863806, 5.15452295376931, 421.9999962136263, 0.00000320121972, 27.5408746672774, 9.194944E-8, 81.2740681275383, 1.51346742105814, 2.4611204E-7, 2.6516236E-7, 2.27623907167979, 46.60828551281571, 1.098859E-8, 8.18385E-9, 0.00000104929054, 80.93484693930488, 48.80638738033572, 0.00000289587853, 8.7093571E-7, 0.00000216108625, 4360.876791818044, 6.3249335E-7, 0.00000126231945, 0.00000484239389, 5.1782974E-7, 0.00000106904727, 0.00000374479675, 4.6358539E-7, 9.0965833E-7, 0.00000442226712, 4.3472357E-7, 0.00000107273279, 2942.130475769188, 0.00000201666035, 57.48613262073047, 18.94947950482293, 1.1147639E-7, 8.714823E-8, 4.901173E-8, 4873.724217169887, 58.97676003311275, 3.0169189E-7, 2012.536811029245, 0.00025027563806, 0.00010452198864, 0.00006108619004, 0.00004168010916, 160.9258102158669, 0.00031997803515, 0.00012606806216, 0.00007055800988, 0.00004693484326, 84.16511811491715, 0.00043474861336, 0.00015966768165, 0.0000847998297, 0.00005420234183, 58.28067368338867, 0.00062642591276, 0.00020884611835, 0.00010617240502, 0.00007985793519, 44.79367141738038, 1043.963500538207, 0.00025019275083, 0.00013719791807, 0.00008122564132, 0.00000513958515, 0.00000333800441, 3.14520021952517, 9.68321610601027, 0.00001901896943, 7.392905E-7, 0.00000250173044, 0.00000372789415, 0.00001114192984, 0.00002502488134, 106.4089312174023, 0.0000020115121, 0.00000386950582, 0.00000834391386, 0.0000178250261, 0.00008095712152, 0.00000167759211, 0.00000328435853, 0.00000677052936, 0.0000143970528, 0.00006002851376, 0.00000143628924, 0.00000281688163, 0.00000577353337, 0.00001293525298, 0.00008972038664, 0.00000125424335, 0.00000244780433, 0.00000505029702, 0.00001244526391, 2009.19069535767, 0.00000162043721, 110.4532361859185, 70.85474716552515, 1.2891021E-7, 0.00117782188595, 996.2720664770459, 0.00052315663849, 0.00013223463645, 0.00012278280684, 0.00028480302047, 0.00087407878118, 0.00045704032983, 0.00017479027485, 0.00010974786327, 0.00016754141831, 0.00048385893562, 0.0005936501136, 0.00026900093596, 0.00014085410479, 0.00011911601798, 0.00034159234344, 0.0009421442194, 0.00046234826411, 978.4832882394979, 0.00009247704752, 0.00026848572434, 45.26507826562338, 0.00118111269967, 0.00047654033451, 2.741815E-8 }

        CuteMarosMeszarosCase.doTest("QSHARE1B.SIF", ACCURACY.withPrecision(3));
    }

    /**
     * <p>
     * The QP solver ends up (de)activating inequality constraints "forever"... Does eventually finish with a
     * solution, but tagged "slow" since it simply takes too long.
     */
    @Test
    @Tag("slow")
    public void testQSTAIR() {
        CuteMarosMeszarosCase.doTest("QSTAIR.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testS268() {
        CuteMarosMeszarosCase.doTest("S268.SIF", ACCURACY.withScale(4));
    }

    /**
     * <p>
     */
    @Test
    public void testTAME() {
        CuteMarosMeszarosCase.doTest("TAME.SIF");
    }

    /**
     * <p>
     */
    @Test
    public void testZECEVIC2() {
        CuteMarosMeszarosCase.doTest("ZECEVIC2.SIF");
    }

}
