/* NVTV Chrontel TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_ch2_7009.c,v 1.10 2004/03/01 21:08:10 dthierbach Exp $
 *
 * Contents:
 *
 * Routines to access the Chrontel chip registers via the I2C bus.
 *
 * Model2: CH7009, CH7010 (and maybe CH7011, CH7012)
 *
 */

#include "local.h" /* before everything else */
#include "xf86_ansic.h" 

#include "xf86i2c.h"
#include "bitmask.h"
#include "tv_i2c.h"
#include "tv_ch2_7009.h"

/* -------- Chrontel -------- */

/* Constant chrontel model 2 register values
 *
 * -  01-6   vof   = 0 (composite and S-Video)
 * -  02-6   cfrb  = 1 (default, lock to vertical) (FIXME extra field)
 * -  02-7   vbid  = 1 (default, minimum flickering during vertical blanking)
 */

void TVCh2Create (TVEncoderObj *this, TVChip chip_type, void* ctrl)
{
  this->type = chip_type;
  this->ctrl = ctrl;
  RAISE (MSG_ERROR, "TVCh2Create: CH7009 family not yet supported.");
}

void TVCh2SetPort (TVEncoderObj *this, int port)
{
  RAISE (MSG_ERROR, "TVCh2SetPort: CH7009 family not yet supported.");
  /* PORT_PCLK_POLARITY  -> 1c:2 (and init 1e-0) */
  /* PORT_PCLK_MODE      -> 1c:3 (and 1e-1)      */
  /* PORT_SYNC_DIR       -> 1f:5 */
  /* PORT_VSYNC_POLARITY -> 1f:4 */
  /* PORT_HSYNC_POLARITY -> 1f:3 */
  /* PORT_FORMAT_MASK    -> 1f:2-0 */
  /* init 1f-6, 1f-7; 1c-1, 1c-0 */
}

void TVCh2GetPort (TVEncoderObj *this, int *port)
{
  RAISE (MSG_ERROR, "TVCh2GetPort: CH7009 family not yet supported.");
}

void TVCh2InitRegs (TVEncoderObj *this, int port)
{
  RAISE (MSG_ERROR, "TVCh2InitRegs: CH7009 family not yet supported.");
}

/*
 * -  00-7:5 ir  =
 * -  00-4:3 vs  =
 * -  00-2:0 sr  = 
 * -  01-6   vof
 * -  01-5:4
 * -  01-3:2
 * -  01-1:0
 * -  02-7
 * -  02-6
 */

void TVCh2SetRegs (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  // RAISE (MSG_ERROR, "TVCh2SetRegs: CH7009 family not yet supported."); return;
  RAISE (MSG_DEBUG, "tv regs ch2");
#ifdef FAKE_I2C
  FPRINTF ("\ni2c [");
#endif
  TVWriteBus (dev, 0x80|0x00, SetBitField(r->ch.dmr_ir,2:0,7:5)
		            | SetBitField(r->ch.dmr_vs,1:0,4:3)
		            | SetBitField(r->ch.dmr_sr,2:0,2:0));
  TVWriteBus (dev, 0x80|0x01, SetBitFlag(r->ch.flags,CH_FLAG_SCART,6)
	                    | SetBitField(r->ch.ffr_fc,1:0,5:4)
 	                    | SetBitField(r->ch.ffr_ft,1:0,3:2)
		            | SetBitField(r->ch.ffr_fy,1:0,1:0));
  TVWriteBus (dev, 0x80|0x02, SetBit(7) /* vbid */
	                    | SetBitFlag(r->ch.flags,CH_FLAG_CFRB,6)
	                    | SetBitFlag(r->ch.flags,CH_FLAG_CVBW,5)
	                    | SetBitField(r->ch.vbw_cbw,0:0,4:4)
	                    | SetBitField(r->ch.vbw_ysv,1:0,3:2)
			    | SetBitField(r->ch.vbw_ycv,1:0,1:0));
  TVWriteBus (dev, 0x80|0x03, SetBitField(r->ch.sav,8:8,5:5)
		            | SetBitField(r->ch.hpr,8:8,4:4)
		            | SetBitField(r->ch.vpr,8:8,3:3)
	                    | SetBitField(r->ch.te, 2:0,2:0));
  TVWriteBus (dev, 0x80|0x04, Set8Bits(r->ch.sav));
  TVWriteBus (dev, 0x80|0x05, Set8Bits(r->ch.hpr));
  TVWriteBus (dev, 0x80|0x06, Set8Bits(r->ch.vpr));
  TVWriteBus (dev, 0x80|0x07, Set8Bits(r->ch.blr));
  TVWriteBus (dev, 0x80|0x08, SetBitField(r->ch.ce,2:0,2:0));
  /* FIXME defaults 0x80 mem=2, ibi=0, pllcpi=0 */
  /*       observed 0xE0 mem=3, ibi=1, pllcpi=0 */
  TVWriteBus (dev, 0x80|0x09, 0xe0 /* mem=2, ibi=0, pllcpi=0 */
	                    | SetBitField(r->ch.pll_n,9:8,4:3)
		            | SetBitField(r->ch.pll_m,8:8,2:2)
	                    | SetBitField(r->ch.pllcap,0:0,0:0));
  TVWriteBus (dev, 0x80|0x0a, Set8Bits(r->ch.pll_m));
  TVWriteBus (dev, 0x80|0x0b, Set8Bits(r->ch.pll_n));
  TVWriteBus (dev, 0x80|0x0c, SetBitField(r->ch.fsci,31:24,7:0));
  TVWriteBus (dev, 0x80|0x0d, SetBitField(r->ch.fsci,23:16,7:0));
  TVWriteBus (dev, 0x80|0x0e, SetBitField(r->ch.fsci,15: 8,7:0));
  TVWriteBus (dev, 0x80|0x0f, SetBitField(r->ch.fsci, 7: 0,7:0));
  TVWriteBus (dev, 0x80|0x10, 0x00 /* civc_paln = 0 */
	                    | SetBitField(r->ch.civh,1:0,3:2)
	                    | SetBitFlag(r->ch.flags,CH_FLAG_ACIV,0));
  /* 0x11-0x13 civ -- read only */
  /* 0x14-0x1b resverd */
  TVWriteBus (dev, 0x80|0x1c, 0x08 /* FIXME m/s etc. hwconfig */ );
  /* reserved register ... 0xc0 ?? */
  TVWriteBus (dev, 0x80|0x1d, 0xc0 | 0x08 /* xcmd=8 */ );
  /* FIXME 0x1e hwconfig */
  /* FIXME 0x1f port */
  /* 0x20 reserved */
  TVWriteBus (dev, 0x80|0x21, 0x00 /* FIXME dacg1 from input format */
	                    | SetBitField(r->ch.dacg,0:0,1:1));
  /* 0x23-0x30 reserved */
  /* 0x31-0x37 dvi -- not used */
  /* 0x38-0x47 reserved */
  /* 0x48      test pattern */
  /* FIXME 0x49 power management */
  /* 0x4a-0x4b version */
  /* 0x4c-0x4f reserved */
  TVWriteBus (dev, 0x80|0x15, 0x00); /* MV off, APS0 */

#ifdef FAKE_I2C
  FPRINTF ("]\n");
#endif
}
 
void TVCh2GetRegs (TVEncoderObj *this, TVEncoderRegs *r)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte regs[0x50];
  int i;

  RAISE (MSG_ERROR, "TVCh2GetRegs: CH7009 family not yet supported."); return;
  for (i = 0x00; i < 0x50; i++) TVReadBus (dev, 0x80|i, &regs[i]);
  r->ch.dmr_ir    = SetBitField(regs[0x00],7:5,2:0);
  r->ch.dmr_vs    = SetBitField(regs[0x00],4:3,1:0);
  r->ch.dmr_sr    = SetBitField(regs[0x00],2:0,2:0);
  r->ch.ffr_fc    = SetBitField(regs[0x01],5:4,1:0);
  r->ch.ffr_ft    = SetBitField(regs[0x01],3:2,1:0);
  r->ch.ffr_fy    = SetBitField(regs[0x01],1:0,1:0);
  r->ch.vbw_flff  = 0;
  r->ch.vbw_cbw   = SetBitField(regs[0x02],5:4,1:0);
  r->ch.vbw_ypeak = 0;
  r->ch.vbw_ysv   = SetBitField(regs[0x02],3:2,1:0);
  r->ch.vbw_ycv   = SetBitField(regs[0x02],1:0,1:0);
  r->ch.te        = SetBitField(regs[0x03],2:0,2:0);
  r->ch.sav       = Set8Bits(regs[0x04])
                  | SetBitField(regs[0x03],5:5,8:8);
  r->ch.hpr       = Set8Bits(regs[0x05]) 
                  | SetBitField(regs[0x03],4:4,8:8);
  r->ch.vpr       = Set8Bits(regs[0x06]) 
                  | SetBitField(regs[0x03],3:3,8:8);
  r->ch.blr       = Set8Bits(regs[0x07]);
  r->ch.ce        = SetBitField(regs[0x08],2:0,2:0);
  r->ch.pllcap    = SetBitField(regs[0x09],0:0,0:0);
  r->ch.pll_m     = Set8Bits(regs[0x0a]) 
                  | SetBitField(regs[0x09],2:2,8:8);
  r->ch.pll_n     = Set8Bits(regs[0x0b]) 
                  | SetBitField(regs[0x09],4:3,9:8);
  r->ch.fsci      = SetBitField(regs[0x0c],7:0,31:24)
                  | SetBitField(regs[0x0d],7:0,23:16)
                  | SetBitField(regs[0x0e],7:0,15: 8)
                  | SetBitField(regs[0x0f],7:0, 7: 0);
  r->ch.civh      = SetBitField(regs[0x10],3:2,1:0);
  r->ch.dacg      = SetBitField(regs[0x21],1:1,0:0);

  r->ch.flags     = CH_FLAG_BOTH 
 	          | GetBitFlag(regs[0x01],6,CH_FLAG_SCART) 
	          | GetBitFlag(regs[0x02],6,CH_FLAG_CFRB)
                  | GetBitFlag(regs[0x02],5,CH_FLAG_CVBW)
	          | GetBitFlag(regs[0x10],0,CH_FLAG_ACIV);
  /* FIXME CH_FLAG_POUTP */

  r->ch.macro = 0;
  r->ch.mode = 0;
}

void TVCh2SetState (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  RAISE (MSG_ERROR, "TVCh2SetState: CH7009 family not yet supported."); 
  return;
  /* FIXME 0x21 dac control, 0x48 test pattern, 0x49 power management */
}

TVConnect TVCh2GetConnect (TVEncoderObj *this)
{
  RAISE (MSG_ERROR, "TVCh2GetConnect: CH7009 family not yet supported.");
  return CONNECT_NONE;
}

long TVCh2GetStatus (TVEncoderObj *this, int index)
{ 
  RAISE (MSG_ERROR, "TVCh2GetStatus: CH7009 family not yet supported.");
  return 0;
}

/* 2004-02-29                Version ID table                        */
/* Collected from Chrontel datasheets                                */
/* Note: all chips have the same did (0x17).                         */
/* vid       chip(s)                                                 */
/* 0x03      7012                                                    */
/* 0x82      7301A                                                   */
/* 0x83      7009A, 7010A, 7010B, 7011 (this may be a copy and paste */
/*             mistake...)                                           */
/* 0x84      ? not found in publicly available Chrontel datasheets   */
/* 0x85      7009B                                                   */
/* 0x95      7301C                                                   */
/* ??        7303    Chrontel has changed it's policy.               */
/* ??        7304    You must now be a licensed partner to           */
/* ??        7305    access the complete datasheets for these chips. */
/*           The newer chips (7015, 7017, 7019, 7205) also requires  */
/*           licensed partnership to get the full documentation.     */

char *TVDetectChrontel2 (I2CDevPtr dev, TVChip *encoder)
{
  I2CByte result;
  char *chip;
  static char version [50];

#ifdef FAKE_PROBE_CHRONTEL
  *encoder = TV_CHRONTEL_MODEL2;
  snprintf (version, 50, "Chrontel-2 Fake (%1s:%02X)", I2C_ID(dev));
  return version;
#else
#ifdef FAKE_I2C
  return NULL;
#endif
#endif  
  tvBusOk = TRUE;  
  TVReadBus (dev, 0x80|0x4a, &result);
  RAISE (MSG_DEBUG, "ch version %02x (%s)", result, tvBusOk ? "ok" : "err");
  if (!tvBusOk) return NULL;
  *encoder = TV_CHRONTEL_MODEL2;
  switch (result) 
  {
    case 0x83: chip = "7009A"; break; /* 7009A vid=0x83 did=0x17 */ 
    case 0x84: chip = "7011A"; break; /* 7011A vid=0x84 did=0x17 */
    // case 0x84: chip = "7009A-T"; break; /* verified for a chip marked "7009A-T" */
    case 0x85: chip = "7009B"; break; /* not verified */
    case 0x03: chip = "7012";  break; /* not verified */
    case 0x82: chip = "7301A"; break; /* not verified */
    case 0x95: chip = "7301C"; break; /* not verified */
    default:   chip = NULL; break;
  }
  if (chip) {
    snprintf (version, 50, "Chrontel %s (%1s:%02X)", chip, I2C_ID(dev));
  } else {
    snprintf (version, 50, "Chrontel-2 (id = %02X) (%1s:%02X)", 
	      result, I2C_ID(dev));
  }
  return version;
}

 /* Clock range is a safe guess based on available 7009 modes */

/* FIXME */

TVEncoderObj tvCh2Template = {
  type: TV_CHRONTEL_MODEL2, ctrl: NULL, minClock: 20000, maxClock: 80000, 
  Create:     TVCh2Create,
  InitRegs:   TVCh2InitRegs, 
  SetRegs:    TVCh2SetRegs, 
  GetRegs:    TVCh2GetRegs, 
  SetPort:    TVCh2SetPort,
  GetPort:    TVCh2GetPort,
  SetState:   TVCh2SetState,
  GetConnect: TVCh2GetConnect, 
  GetStatus:  TVCh2GetStatus
};

/* ---------------------------------------- */

/*

APS0: 15
APS1-3: 15 18 19/1f 1a 1b 28 29 2a/0f 2b/3f 2c 2d 2e/3f 2f/3f 
        40 41 42/7f 43 44/7f 45

Mode: 16/7f 17 38/7f 39/0f 3a 3b 3c/7f 3d/7f 3e 3f

APS_Mode 24/07 25 26 27

ch9_APS0: 00000004
  0000:  15 00 ff  ff
ch9_NTSC_APS1: 0000003A
  15 5b ff  18 0b ff  19 06 1f  1a 05 ff  1b 0d ff 
  28 00 ff  29 0d ff  2a 00 0f  2b 00 3f  2c 00 ff 
  2d 00 ff  2e 15 3f  2f 15 3f  40 0f ff  41 0f ff 
  42 0f 7f  43 f0 ff  44 00 7f  45 00 ff  ff
ch9_NTSC_APS2: 0000003A
  15 5f ff  18 0b ff  19 06 1f  1a 05 ff  1b 0d ff 
  28 ff ff  29 47 ff  2a 0d 0f  2b 11 3f  2c 9e ff 
  2d 2d ff  2e 11 3f  2f 11 3f  40 0f ff  41 0f ff 
  42 0f 7f  43 f0 ff  44 00 7f  45 00 ff  ff
ch9_NTSC_APS3: 0000003A
  15 5f ff  18 0b ff  19 06 1f  1a 05 ff  1b 0d ff 
  28 ff ff  29 87 ff  2a 0b 0f  2b 15 3f  2c 98 ff 
  2d 29 ff  2e 15 3f  2f 15 3f  40 0f ff  41 0f ff 
  42 0f 7f  43 f0 ff  44 00 7f  45 00 ff  ff

ch9_NTSC_Mode11: 0000001F
  16 25 7f  17 3b ff  38 5a 7f  39 0f 0f  3a 05 ff 
  3b 05 ff  3c 41 7f  3d 41 7f  3e 05 ff  3f 05 ff  ff
ch9_NTSC_Mode11_APS1: 00000001
  ff
ch9_NTSC_Mode11_APS2and3: 0000000D
  24 00 07  25 b4 ff  26 b4 ff  27 df ff  ff

ch9_NTSC_Mode17: 0000001F
  16 25 7f  17 38 ff  38 56 7f  39 00 0f  3a fa ff 
  3b fa ff  3c 3e 7f  3d 3e 7f  3e fa ff  3f fa ff  ff
ch9_NTSC_Mode17_APS2and3: 0000000D
  24 00 07  25 ad ff  26 ad ff  27 d6 ff  ff

ch9_NTSC_Mode29: 0000001F
  16 25 7f  17 48 ff  38 6d 7f  39 0f 0f  3a 3e ff 
  3b 3e ff  3c 4f 7f  3d 4f 7f  3e 3e ff  3f 3e ff  ff
ch9_NTSC_Mode29_APS2and3: 0000000D
  24 01 07  25 dd ff  26 dd ff  27 11 ff  ff

ch9_NTSC_Mode35: 0000001F
  16 25 7f  17 42 ff  38 64 7f  39 0f 0f  3a 23 ff 
  3b 23 ff  3c 48 7f  3d 48 7f  3e 23 ff  3f 23 ff  ff
ch9_NTSC_Mode35_APS2and3: 0000000D
  24 00 07  25 c9 ff  26 c9 ff  27 f9 ff  ff

ch9_PAL_APS1: 0000003A
  15 1b ff  18 6c ff  19 03 1f  1a 33 ff  1b 38 ff 
  28 00 ff  29 40 ff  2a 08 0f  2b 22 3f  2c 9b ff 
  2d 64 ff  2e 22 3f  2f 22 3f  40 fe ff  41 7e ff 
  42 3f 7f  43 e0 ff  44 15 7f  45 40 ff  ff
ch9_PAL_Mode8: 0000002B
  16 25 7f  17 38 ff  24 00 07  25 bd ff  26 bd ff 
  27 dd ff  38 4d 7f  39 0c 0f  3a 3c ff  3b 1b ff 
  3c 2d 7f  3d 31 7f  3e 95 ff  3f b7 ff  ff
ch9_PAL_Mode14: 0000002B
  16 25 7f  17 3b ff  24 00 07  25 c7 ff  26 c7 ff 
  27 e9 ff  38 51 7f  39 0c 0f  3a 4c ff  3b 29 ff 
  3c 30 7f  3d 34 7f  3e 9d ff  3f c0 ff  ff
ch9_PAL_Mode26: 0000002B 
  16 25 7f  17 48 ff  24 01 07  25 f2 ff  26 f2 ff 
  27 1c ff  38 63 7f  39 0c 0f  3a 94 ff  3b 6a ff 
  3c 3a 7f  3d 3f 7f  3e bf ff  3f ea ff  ff
ch9_PAL_Mode32: 0000002B 
  16 25 7f  17 46 ff  24 01 07  25 ed ff  26 ed ff 
  27 15 ff  38 60 7f  39 0c 0f  3a 89 ff  3b 60 ff 
  3c 38 7f  3d 3e 7f  3e ba ff  3f e3 ff  ff

*/

/* Registers 

00-7:5 dm_ir     ch1.dmr_ir
00-4:3 dm_vs     ch1.dmr_vs
00-2:0 dm_sr     ch1.dmr_sr
	         
01-6   ff_vof    CH_FLAG_SCART ?? RGB output, 
01-5:4 ff_cff    ch1.fft_fc (different setup)
01-3:2 ff_yfft   ch1.ffr_ft (different setup)
01-1:0 ff_yffnt  ch1.ffr_fy (different setup)

02-7   vbw_vbid  ## vertical blanking interval (ff) defeat
02-6   vbw_cfrb  CH_FLAG_CFRB
02-5   vbw_cvbwb CH_FLAG_CVBW
02-4   vbw_cbw   ch1.vbw_cbw
02-3:2 vbw_ysv   ch1.vbw_ysv
02-1:0 vbw_ycv   ch1.vbw_ycv

03-5   sav/8     ch1.sav
03-4   hp/8      ch1.hpr
03-3   vp/8      ch1.vpr
03-2:0 te        ch1.te (should be ch2.te)

04     sav       ch1.sav
05     hp        ch1.hpr
06     vp        ch1.vpr
07     bl        ch1.blr
08-2:0 ce        ch1.ce

09-7:6 mem       @@ keep
09-5   ibi       @@ keep
09-4:3 pll_n/8   ch1.pll_n
09-2   pll_m/8   ch1.pll_m
09-1   pllcpi    @@ keep
09-0   pllcap    ch1.pllcap (->FLAG ?)

0a     pllm      ch1.pll_m
0b     plln      ch1.pll_n

0c-0f  fsci      ch1.fsci

10-0   civen     CH_FLAG_ACIV
10-3:2 civc      ch1.civh @@ keep
10-...

1d-3:0 xcmd      ## ch2.xcmd  delay

1e-0   poutp     CH_FLAG_POUTP
1e-...

1f     idf       @@ init
1f-7   ibs       0 @@ input buffer select
1f-6   des       0 @@ decode embedded sync
1f-5   syo       0 @@ sync direction, 0=in, 1=out
1f-4   vsp       0 @@ vertical sync polarity, 0=active low
1f-3   hsp       0 @@ horizontal sync polarity, 0=active low
1f-2:0 idf       0 @@ input data format

21-1   dacg0     ch1.dacg  dac gain/format

49-3:0 dacpd     CH_FLAG_DAC_PD[0-3] (additional modes?)     

civc = civh

Additional:

tstp/2 0x48  test pattern

*/

/* Naming scheme:

ch    for all chrontel
ch_a  for 7007 family
ch_b  for 7009 family

cx
cx_a  for bt family
cx_b  for cx family

ph
ph_a
ph_b

numbering in TVChip:

<company><family><major><minor>

conexant bt868    2101
conexant bt869    2102
conexant cx25870  2201
conexant cx25871  2202
conexant cx25872  2203
conexant cx25873  2204
conexant cx25874  2205
conexant cx25875  2206

chrontel ch7003  1103
chrontel ch7004  1104
chrontel ch7005  1105
chrontel ch7006  1106
chrontel ch7007  1107
chrontel ch7008  1108

chrontel ch7009  1201
chrontel ch7010  1202
chrontel ch7011  1203

philips  saa7102/08   3102
philips  saa7103/09   3103
philips  saa7104/08A  3201
philips  saa7105/09A  3202

philips:

status: iferr. bferr
state: dclk, eidiv (but always set?)
pcle, pcli, pclsy, ifra, ifbp, fili, cvbsen2, 
yfil, xint, yupsc, 

extra filter setting??? (rename adaptive to extra filter)
can do upscaling modes, 640 huge, 1024 ..., 1280 ...

 */
