#!/usr/bin/env julia
#
# @license Apache-2.0
#
# Copyright (c) 2018 The Stdlib Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import Distributions: logpdf, Logistic
import JSON

"""
	gen( x, mu, s, name )

Generate fixture data and write to file.

# Arguments

* `x`: input value
* `mu`: location parameter
* `s`: scale parameter
* `name::AbstractString`: output filename

# Examples

``` julia
julia> x = rand( 1000 ) .* 15;
julia> mu = rand( 1000 ) .* 10.0;
julia> s = rand( 1000 ) .* 5.0;
julia> gen( x, mu, s, "data.json" );
```
"""
function gen( x, mu, s, name )
	z = Array{Float64}( undef, length(x) );
	for i in eachindex(x)
		z[ i ] = logpdf( Logistic( mu[i], s[i] ), x[i] );
	end

	# Store data to be written to file as a collection:
	data = Dict([
		("x", x),
		("mu", mu),
		("s", s),
		("expected", z)
	]);

	# Based on the script directory, create an output filepath:
	filepath = joinpath( dir, name );

	# Write the data to the output filepath as JSON:
	outfile = open( filepath, "w" );
	write( outfile, JSON.json(data) );
	close( outfile );
end

# Get the filename:
file = @__FILE__;

# Extract the directory in which this file resides:
dir = dirname( file );

# Negative mean:
x = rand( 1000 ) .* 10.0 .- 20.0;
mu = rand( 1000 ) .* -10.0;
s = rand( 1000 ) .* 5.0;
gen( x, mu, s, "negative_mean.json" );

# Positive mean:
x = rand( 1000 ) .* 10.0 .- 20.0;
mu = rand( 1000 ) .* 10.0;
s = rand( 1000 ) .* 5.0;
gen( x, mu, s, "positive_mean.json" );

# Large variance:
x = rand( 1000 ) .* 5.0;
mu = rand( 1000 );
s = rand( 1000 ) .* 20.0;
gen( x, mu, s, "large_variance.json" );
