/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/chi/cdf' );
import Chi = require( './../../../../../base/dists/chi/ctor' );
import entropy = require( './../../../../../base/dists/chi/entropy' );
import kurtosis = require( './../../../../../base/dists/chi/kurtosis' );
import logpdf = require( './../../../../../base/dists/chi/logpdf' );
import mean = require( './../../../../../base/dists/chi/mean' );
import mode = require( './../../../../../base/dists/chi/mode' );
import pdf = require( './../../../../../base/dists/chi/pdf' );
import quantile = require( './../../../../../base/dists/chi/quantile' );
import skewness = require( './../../../../../base/dists/chi/skewness' );
import stdev = require( './../../../../../base/dists/chi/stdev' );
import variance = require( './../../../../../base/dists/chi/variance' );

/**
* Interface describing the `chi` namespace.
*/
interface Namespace {
	/**
	* Chi distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param k - degrees of freedom
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 8.0 );
	* // returns ~0.143
	*
	* y = ns.cdf( 0.0, 1.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 1.0 );
	*
	* y = mycdf( 2.0 );
	* // returns ~0.954
	*
	* y = mycdf( 1.2 );
	* // returns ~0.77
	*/
	cdf: typeof cdf;

	/**
	* Chi distribution.
	*/
	Chi: typeof Chi;

	/**
	* Returns the differential entropy of a chi distribution (in nats).
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 9.0 );
	* // returns ~1.052
	*
	* @example
	* var v = ns.entropy( 1.0 );
	* // returns ~0.726
	*
	* @example
	* var v = ns.entropy( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a chi distribution.
	*
	* ## Notes
	*
	* -   If provided `k <= 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 9.0 );
	* // returns ~0.011
	*
	* @example
	* var v = ns.kurtosis( 1.0 );
	* // returns ~0.869
	*
	* @example
	* var v = ns.kurtosis( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Chi distribution natural logarithm of probability density function (logPDF).
	*
	* @param x - input value
	* @param k - degrees of freedom
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 1.0 );
	* // returns ~-2.226
	*
	* var mylogpdf = ns.logpdf.factory( 6.0 );
	*
	* y = mylogpdf( 3.0 );
	* // returns ~-1.088
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a chi distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 9.0 );
	* // returns ~2.918
	*
	* @example
	* var v = ns.mean( 1.0 );
	* // returns ~0.798
	*
	* @example
	* var v = ns.mean( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the mode of a chi distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns mode
	*
	* @example
	* var v = ns.mode( 9.0 );
	* // returns ~2.828
	*
	* @example
	* var v = ns.mode( 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Chi distribution probability density function (PDF).
	*
	* @param x - input value
	* @param k - degrees of freedom
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 1.0 );
	* // returns ~0.108
	*
	* var myPDF = ns.pdf.factory( 6.0 );
	*
	* y = myPDF( 3.0 );
	* // returns ~0.337
	*/
	pdf: typeof pdf;

	/**
	* Chi distribution quantile function.
	*
	* @param p - input value
	* @param k - degrees of freedom
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 3.0 );
	* // returns ~2.154
	*
	* var myquantile = ns.quantile.factory( 2.0 );
	*
	* var y = myquantile( 0.3 );
	* // returns ~0.844
	*
	* y = myquantile( 0.7 );
	* // returns ~1.552
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a chi distribution.
	*
	* ## Notes
	*
	* -   If provided `k <= 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 9.0 );
	* // returns ~0.252
	*
	* @example
	* var v = ns.skewness( 1.0 );
	* // returns ~0.995
	*
	* @example
	* var v = ns.skewness( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a chi distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 9.0 );
	* // returns ~0.697
	*
	* @example
	* var v = ns.stdev( 1.0 );
	* // returns ~0.603
	*
	* @example
	* var v = ns.stdev( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a chi distribution.
	*
	* ## Notes
	*
	* -   If provided `k < 0`, the function returns `NaN`.
	*
	* @param k - degrees of freedom
	* @returns variance
	*
	* @example
	* var v = ns.variance( 9.0 );
	* // returns ~0.485
	*
	* @example
	* var v = ns.variance( 1.0 );
	* // returns ~0.363
	*
	* @example
	* var v = ns.variance( -0.2 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Chi distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
