/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var ENV = require( '@stdlib/process/env' );
var now = require( '@stdlib/time/now' );
var chi2gof = require( '@stdlib/stats/chi2gof' );
var isnan = require( '@stdlib/math/base/assert/is-nan' );
var isInteger = require( '@stdlib/assert/is-integer' ).isPrimitive;
var isUint32Array = require( '@stdlib/assert/is-uint32array' );
var UINT32_MAX = require( '@stdlib/constants/uint32/max' );
var Uint32Array = require( '@stdlib/array/uint32' );
var typedarray2json = require( '@stdlib/array/to-json' );
var minstd = require( './../../../base/minstd' );
var factory = require( './../lib/factory.js' );


// VARIABLES //

var opts = {
	'skip': ( ENV.TEST_MODE === 'coverage' )
};


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof factory, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function throws an error if minimum support `a` is not an integer primitive', function test( t ) {
	var values;
	var i;

	values = [
		3.14,
		'5',
		null,
		true,
		false,
		void 0,
		NaN,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( value, 20 );
		};
	}
});

tape( 'the function throws an error if maximum support `b` is not an integer primitive', function test( t ) {
	var values;
	var i;

	values = [
		30.14,
		'5',
		null,
		true,
		false,
		void 0,
		NaN,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( 20, value );
		};
	}
});

tape( 'the function throws an error if minimum support `a` is greater than to maximum support `b`', function test( t ) {
	var values;
	var i;

	values = [
		[ 0, -1 ],
		[ -20, -40 ],
		[ 20, 10 ]
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( arr ) {
		return function badValue() {
			factory( arr[0], arr[1] );
		};
	}
});

tape( 'the function throws an error if provided an options argument which is not an object (no other arguments)', function test( t ) {
	var values;
	var i;

	values = [
		'abc',
		5,
		null,
		true,
		false,
		void 0,
		NaN,
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( value );
		};
	}
});

tape( 'the function throws an error if provided an options argument which is not an object (other arguments)', function test( t ) {
	var values;
	var i;

	values = [
		'abc',
		5,
		null,
		true,
		false,
		void 0,
		NaN,
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( 0, 10, value );
		};
	}
});

tape( 'if provided a `prng` option which is not a function, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'prng': value
			});
		};
	}
});

tape( 'if provided a `prng` option which does not have a valid `MIN` property, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		function rand() {
			return 1;
		}

		rand.MIN = value;
		rand.MAX = 10;

		return function badValue() {
			factory({
				'prng': rand
			});
		};
	}
});

tape( 'if provided a `prng` option which does not have a valid `MIN` property, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		function rand() {
			return 1;
		}

		rand.MIN = 1;
		rand.MAX = value;

		return function badValue() {
			factory({
				'prng': rand
			});
		};
	}
});

tape( 'if provided a `prng` option which is not a function, the function throws an error (other arguments)', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory( 0, 10, {
				'prng': value
			});
		};
	}
});

tape( 'if provided a `prng` option which does not have a valid `MIN` property, the function throws an error (other arguments)', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		function rand() {
			return 1;
		}

		rand.MIN = value;
		rand.MAX = 10;

		return function badValue() {
			factory( 1, 10, {
				'prng': rand
			});
		};
	}
});

tape( 'if provided a `prng` option which does not have a valid `MAX` property, the function throws an error (other arguments)', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		NaN,
		true,
		false,
		null,
		void 0,
		[],
		{}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		function rand() {
			return 1;
		}

		rand.MIN = 1;
		rand.MAX = value;

		return function badValue() {
			factory( 1, 10, {
				'prng': rand
			});
		};
	}
});

tape( 'if provided a `copy` option which is not a boolean, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'copy': value
			});
		};
	}
});

tape( 'if provided a `seed` which is not a positive integer or a non-empty array-like object, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		3.14,
		0.0,
		-5.0,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'seed': value
			});
		};
	}
});

tape( 'the function throws a range error if provided a `seed` which is an integer greater than the maximum unsigned 32-bit integer', function test( t ) {
	var values;
	var i;

	values = [
		UINT32_MAX + 1,
		UINT32_MAX + 2,
		UINT32_MAX + 3
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws a range error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'seed': value
			});
		};
	}
});

tape( 'if provided a `state` option which is not a Uint32Array, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		5,
		NaN,
		true,
		false,
		null,
		void 0,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws a type error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'state': value
			});
		};
	}
});

tape( 'if provided an invalid `state` option, the function throws an error', function test( t ) {
	var values;
	var i;

	values = [
		new Uint32Array( 0 ),
		new Uint32Array( 10 ),
		new Uint32Array( 100 )
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), RangeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			factory({
				'state': value
			});
		};
	}
});

tape( 'the function returns a pseudorandom number generator (no seed)', function test( t ) {
	var discreteUniform;
	var r;
	var i;

	// When binding distribution parameters...
	discreteUniform = factory( -100, 100 );
	for ( i = 0; i < 100; i++ ) {
		r = discreteUniform();
		t.equal( typeof r, 'number', 'returns a number' );
		t.equal( isInteger( r ), true, 'returns an integer' );
	}

	// Without binding distribution parameters...
	discreteUniform = factory();
	for ( i = 0; i < 100; i++ ) {
		r = discreteUniform( -100, 100 );
		t.equal( typeof r, 'number', 'returns a number' );
		t.equal( isInteger( r ), true, 'returns an integer' );
	}
	t.end();
});

tape( 'the function returns a seeded pseudorandom number generator (integer seed)', function test( t ) {
	var discreteUniform1;
	var discreteUniform2;
	var seed;
	var r1;
	var r2;
	var i;

	seed = now();

	discreteUniform1 = factory( 0, 40, {
		'seed': seed
	});
	discreteUniform2 = factory( 0, 40, {
		'seed': seed
	});

	t.notEqual( discreteUniform1, discreteUniform2, 'separate generators' );

	for ( i = 0; i < 100; i++ ) {
		r1 = discreteUniform1();
		r2 = discreteUniform2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'the function returns a seeded pseudorandom number generator (array seed)', function test( t ) {
	var discreteUniform1;
	var discreteUniform2;
	var seed;
	var r1;
	var r2;
	var i;

	seed = [ now()+1, now()+2, now()+3 ];

	discreteUniform1 = factory( 0, 40, {
		'seed': seed
	});
	discreteUniform2 = factory( 0, 40, {
		'seed': seed
	});

	t.notEqual( discreteUniform1, discreteUniform2, 'separate generators' );

	for ( i = 0; i < 100; i++ ) {
		r1 = discreteUniform1();
		r2 = discreteUniform2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'attached to the returned function is the generator name', function test( t ) {
	var discreteUniform = factory();
	t.equal( discreteUniform.NAME, 'discrete-uniform', 'has property' );
	t.end();
});

tape( 'attached to the returned function is the underlying PRNG', function test( t ) {
	var discreteUniform = factory();
	t.equal( typeof discreteUniform.PRNG, 'function', 'has property' );

	discreteUniform = factory({
		'prng': minstd
	});
	t.equal( discreteUniform.PRNG, minstd, 'has property' );
	t.end();
});

tape( 'attached to the returned function is the generator seed (integer seed)', function test( t ) {
	var discreteUniform = factory({
		'seed': 12345
	});
	t.equal( isUint32Array( discreteUniform.seed ), true, 'has property' );
	t.equal( discreteUniform.seed[ 0 ], 12345, 'equal to provided seed' );

	discreteUniform = factory({
		'seed': 12345,
		'prng': minstd
	});
	t.equal( discreteUniform.seed, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator seed (array seed)', function test( t ) {
	var actual;
	var rand;
	var seed;
	var i;

	seed = [ 1234, 5678 ];
	rand = factory({
		'seed': seed
	});

	actual = rand.seed;
	t.equal( isUint32Array( actual ), true, 'has property' );
	for ( i = 0; i < seed.length; i++ ) {
		t.equal( actual[ i ], seed[ i ], 'returns expected value for word '+i );
	}
	t.end();
});

tape( 'attached to the returned function is the generator seed length', function test( t ) {
	var discreteUniform = factory();
	t.equal( typeof discreteUniform.seedLength, 'number', 'has property' );

	discreteUniform = factory({
		'prng': minstd
	});
	t.equal( discreteUniform.seedLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state', function test( t ) {
	var discreteUniform = factory();
	t.equal( isUint32Array( discreteUniform.state ), true, 'has property' );

	discreteUniform = factory({
		'prng': minstd
	});
	t.equal( discreteUniform.state, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state length', function test( t ) {
	var discreteUniform = factory();
	t.equal( typeof discreteUniform.stateLength, 'number', 'has property' );

	discreteUniform = factory({
		'prng': minstd
	});
	t.equal( discreteUniform.stateLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is the generator state size', function test( t ) {
	var discreteUniform = factory();
	t.equal( typeof discreteUniform.byteLength, 'number', 'has property' );

	discreteUniform = factory({
		'prng': minstd
	});
	t.equal( discreteUniform.byteLength, null, 'equal to `null`' );
	t.end();
});

tape( 'attached to the returned function is a method to serialize the generator as a JSON object', function test( t ) {
	var discreteUniform;
	var o;

	discreteUniform = factory();
	t.equal( typeof discreteUniform.toJSON, 'function', 'has method' );

	o = discreteUniform.toJSON();
	t.equal( o.type, 'PRNG', 'has property' );
	t.equal( o.name, discreteUniform.NAME, 'has property' );
	t.deepEqual( o.state, typedarray2json( discreteUniform.state ), 'has property' );
	t.deepEqual( o.params, [], 'has property' );

	discreteUniform = factory( 1, 2 );
	o = discreteUniform.toJSON();

	t.deepEqual( o.params, [ 1, 2 ], 'has property' );

	discreteUniform = factory({
		'prng': minstd
	});
	t.equal( typeof discreteUniform.toJSON, 'function', 'has method' );
	t.equal( discreteUniform.toJSON(), null, 'returns expected value' );

	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if `a` is `NaN`', function test( t ) {
	var discreteUniform;
	var r;

	discreteUniform = factory();
	r = discreteUniform( NaN, 50 );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if `b` is `NaN`', function test( t ) {
	var discreteUniform;
	var r;

	discreteUniform = factory();
	r = discreteUniform( 0, NaN );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if both `a` and `b` are `NaN`', function test( t ) {
	var discreteUniform;
	var r;

	discreteUniform = factory();
	r = discreteUniform( NaN, NaN );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if `a` is not an integer value', function test( t ) {
	var discreteUniform;
	var r;

	discreteUniform = factory();
	r = discreteUniform( 3.14, 50 );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if `b` is not an integer value', function test( t ) {
	var discreteUniform;
	var r;

	discreteUniform = factory();
	r = discreteUniform( 0, 3.14 );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if both `a` and `b` are not integer values', function test( t ) {
	var discreteUniform;
	var r;

	discreteUniform = factory();
	r = discreteUniform( -3.14, 3.14 );

	t.strictEqual( isnan( r ), true, 'returns NaN' );
	t.end();
});

tape( 'when called without arguments, the function returns a function that returns `NaN` if minimum support `a` is greater than maximum support `b`', function test( t ) {
	var discreteUniform;
	var values;
	var r;
	var i;

	discreteUniform = factory();
	values = [
		[ 20, 10 ],
		[ -30, -40 ],
		[ 0, -1 ]
	];

	for ( i = 0; i < values.length; i++ ) {
		r = discreteUniform( values[ i ][ 0 ], values[ i ][ 1 ] );
		t.strictEqual( isnan( r ), true, 'returns `NaN` when provided '+values[ i ] );
	}
	t.end();
});

tape( 'if provided a maximum support equal to the minimum support, the function returns a function that returns the minimum support', function test( t ) {
	var discreteUniform;
	var r;

	discreteUniform = factory( 12, 12 );

	r = discreteUniform();
	t.strictEqual( r, 12, 'returns minimum support' );

	r = discreteUniform();
	t.strictEqual( r, 12, 'returns minimum support' );

	r = discreteUniform();
	t.strictEqual( r, 12, 'returns minimum support' );

	discreteUniform = factory( -10, -10 );

	r = discreteUniform();
	t.strictEqual( r, -10, 'returns minimum support' );

	r = discreteUniform();
	t.strictEqual( r, -10, 'returns minimum support' );

	r = discreteUniform();
	t.strictEqual( r, -10, 'returns minimum support' );

	t.end();
});

tape( 'if provided a maximum support equal to the minimum support, the function returns a function that returns the minimum support', function test( t ) {
	var discreteUniform;
	var r;

	discreteUniform = factory();

	r = discreteUniform( 12, 12 );
	t.strictEqual( r, 12, 'returns minimum support' );

	r = discreteUniform( 2, 2 );
	t.strictEqual( r, 2, 'returns minimum support' );

	r = discreteUniform( -4, -4 );
	t.strictEqual( r, -4, 'returns minimum support' );

	t.end();
});

tape( 'the function supports specifying the underlying PRNG', function test( t ) {
	var discreteUniform;
	var r;
	var i;

	discreteUniform = factory( 1, 10, {
		'prng': minstd
	});

	for ( i = 0; i < 1e2; i++ ) {
		r = discreteUniform();
		t.equal( typeof r, 'number', 'returns a number' );
	}
	t.end();
});

tape( 'the function supports specifying the underlying PRNG (parameters)', function test( t ) {
	var discreteUniform;
	var r;
	var i;

	discreteUniform = factory({
		'prng': minstd
	});

	for ( i = 0; i < 1e2; i++ ) {
		r = discreteUniform( 1, 10 );
		t.equal( typeof r, 'number', 'returns a number' );
	}
	t.end();
});

tape( 'the function supports providing a seeded underlying PRNG', function test( t ) {
	var discreteUniform1;
	var discreteUniform2;
	var randi;
	var seed;
	var r1;
	var r2;
	var i;

	seed = now();

	randi = minstd.factory({
		'seed': seed
	});
	discreteUniform1 = factory( 1, 10, {
		'prng': randi
	});

	randi = minstd.factory({
		'seed': seed
	});
	discreteUniform2 = factory( 1, 10, {
		'prng': randi
	});

	t.notEqual( discreteUniform1, discreteUniform2, 'separate generators' );

	for ( i = 0; i < 1e2; i++ ) {
		r1 = discreteUniform1();
		r2 = discreteUniform2();
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'the function supports providing a seeded underlying PRNG (parameters)', function test( t ) {
	var discreteUniform1;
	var discreteUniform2;
	var randi;
	var seed;
	var r1;
	var r2;
	var i;

	seed = now();

	randi = minstd.factory({
		'seed': seed
	});
	discreteUniform1 = factory({
		'prng': randi
	});

	randi = minstd.factory({
		'seed': seed
	});
	discreteUniform2 = factory({
		'prng': randi
	});

	t.notEqual( discreteUniform1, discreteUniform2, 'separate generators' );

	for ( i = 0; i < 1e2; i++ ) {
		r1 = discreteUniform1( 1, 10 );
		r2 = discreteUniform2( 1, 10 );
		t.equal( r1, r2, 'both return same number' );
	}
	t.end();
});

tape( 'the function returns a PRNG for generating random numbers from a discrete uniform distribution', opts, function test( t ) {
	var threshold;
	var repeats;
	var count;
	var npass;
	var freq;
	var N;
	var a;
	var b;
	var x;

	threshold = 0.15;

	a = -10;
	b = 10;

	freq = new Array( b - a + 1 );
	repeats = 100;
	N = 1e3;

	count = -1;
	npass = 0;

	gof();

	function gof() {
		var discreteUniform;
		var rejected;
		var pValue;
		var bool;
		var i;
		var j;

		count += 1;
		rejected = 0;
		for ( i = 0; i < repeats; i++ ) {
			discreteUniform = factory( a, b );
			t.ok( true, 'seed: '+discreteUniform.seed );

			// Reset the `freq` array...
			for ( j = 0; j < freq.length; j++ ) {
				freq[ j ] = 0;
			}
			for ( j = 0; j < N; j++ ) {
				x = discreteUniform();
				freq[ x-a ] += 1;
			}
			// Test using chi-square goodness-of-fit test:
			pValue = chi2gof( freq, 'discrete-uniform', a, b, {
				'simulate': true,
				'iterations': 500
			}).pValue;
			t.equal( typeof pValue, 'number', 'returns a p-value: '+pValue );
			if ( pValue < 0.05 ) {
				rejected += 1;
			}
		}
		// Account for small sample sizes and few repeats...
		bool = ( rejected / repeats < threshold );

		// If we succeed the first time, we are done...
		if ( count === 0 && bool ) {
			return done( bool, rejected );
		}
		// Retry mode...
		if ( bool ) {
			npass += 1;
		}
		// Retry twice...
		if ( count < 2 ) {
			return gof();
		}
		// Both retries must succeed for test to pass:
		bool = ( npass >= 2 );
		return done( bool, rejected );
	}

	function done( bool, rejected ) {
		t.ok( bool, 'null hypothesis (i.e., that numbers are drawn from DiscreteUniform('+a+','+b+') is rejected in less than '+(threshold*100)+'% of cases ('+rejected+' of '+repeats+'). Repeats: '+npass+' of '+count+'.' );
		t.end();
	}
});

tape( 'the function supports specifying the generator state', function test( t ) {
	var discreteUniform;
	var state;
	var arr;
	var i;

	discreteUniform = factory( -10, 10 );

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		discreteUniform();
	}
	// Capture the current state:
	state = discreteUniform.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( discreteUniform() );
	}

	// Create another PRNG using the captured state:
	discreteUniform = factory( -10, 10, {
		'state': state
	});

	// Replay previously generated values...
	for ( i = 0; i < 100; i++ ) {
		t.equal( discreteUniform(), arr[ i ], 'returns expected value. i: '+i+'.' );
	}
	t.end();
});

tape( 'the function supports specifying a shared generator state', function test( t ) {
	var discreteUniform;
	var shared;
	var state;
	var rand1;
	var rand2;
	var arr;
	var v1;
	var v2;
	var i;
	var j;

	discreteUniform = factory( -10, 10 );

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		discreteUniform();
	}
	// Capture the current state:
	state = discreteUniform.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( discreteUniform() );
	}

	// Create a copy of the state (to prevent mutation) which will be shared by more than one PRNG:
	shared = new Uint32Array( state );

	// Create PRNGs using the captured state:
	rand1 = factory( -10, 10, {
		'state': shared,
		'copy': false
	});
	rand2 = factory( -10, 10, {
		'state': shared,
		'copy': false
	});

	// Replay previously generated values...
	j = 0;
	for ( i = 0; i < 25; i++ ) {
		v1 = rand1();
		v2 = rand2();
		t.equal( v1, arr[ j ], 'returns expected value. i: '+j+'.' );
		t.equal( v2, arr[ j+1 ], 'returns expected value. i: '+(j+1)+'.' );
		j += 2; // stride
	}

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		v2 = rand2();
	}

	// Reset the (shared) state:
	rand1.state = state;

	// Replay previously generated values...
	j = 0;
	for ( i = 0; i < 25; i++ ) {
		v1 = rand1();
		v2 = rand2();
		t.equal( v1, arr[ j ], 'returns expected value. i: '+j+'.' );
		t.equal( v2, arr[ j+1 ], 'returns expected value. i: '+(j+1)+'.' );
		j += 2; // stride
	}
	t.end();
});

tape( 'the returned function supports setting the generator state', function test( t ) {
	var discreteUniform;
	var state;
	var arr;
	var i;

	discreteUniform = factory( -10, 10 );

	// Move to a future state...
	for ( i = 0; i < 100; i++ ) {
		discreteUniform();
	}
	// Capture the current state:
	state = discreteUniform.state;

	// Move to a future state...
	arr = [];
	for ( i = 0; i < 100; i++ ) {
		arr.push( discreteUniform() );
	}
	// Set the state:
	discreteUniform.state = state;

	// Replay previously generated values...
	for ( i = 0; i < 100; i++ ) {
		t.equal( discreteUniform(), arr[ i ], 'returns expected value. i: '+i+'.' );
	}
	t.end();
});
