// include Fake lib
#r @"packages/FAKE/tools/FakeLib.dll"
open Fake
open System
open System.IO
open System.Text.RegularExpressions

Target "RestorePackages" (fun _ -> 
     "packages.config"
     |> RestorePackage (fun p ->
         { p with
             ToolPath = "../lib/nuget/NuGet.exe" })
 )

let buildDir = "./bin/Debug/"
let buildReleaseDir = "./bin/Release/"
let integrationTestDir = "./test/integration/"
let emacsBinDir = "../emacs/bin/"

Target "BuildDebug" (fun _ ->
  MSBuildDebug buildDir "Build" ["./FSharp.AutoComplete.sln"]
  |> Log "Build-Output: "
)

Target "BuildRelease" (fun _ ->
  MSBuildRelease buildReleaseDir "Build" ["./FSharp.AutoComplete.sln"]
  |> Log "Build-Output: "
)

let integrationTests =
  !! (integrationTestDir + "/**/*Runner.fsx")

let runIntegrationTest (fn: string) : bool =
  let dir = Path.GetDirectoryName fn

  tracefn "Running FSIHelper '%s', '%s', '%s'"  FSIHelper.fsiPath dir fn
  let b, msgs = FSIHelper.executeFSI dir fn []
  if not b then
    for msg in msgs do
      traceError msg.Message

  // Normalize output files so that a simple
  // `git diff` will be clean if the tests passed.
  for fn in !! (dir + "/*.txt") ++ (dir + "/*.json") do
    let lines = File.ReadAllLines fn
    for i in [ 0 .. lines.Length - 1 ] do
      if Path.DirectorySeparatorChar = '/' then
        lines.[i] <- Regex.Replace(lines.[i],
                                   "/.*?FSharp.AutoComplete/test/(.*?(\"|$))",
                                   "<absolute path removed>/test/$1")
        lines.[i] <- Regex.Replace(lines.[i],
                                   "\"/[^\"]*?/([^\"/]*?\.dll\")",
                                    "\"<absolute path removed>/$1")
      else
        if Path.GetExtension fn = ".json" then
          lines.[i] <- Regex.Replace(lines.[i].Replace(@"\\", "/"),
                                     "[a-zA-Z]:/.*?FSharp.AutoComplete/test/(.*?(\"|$))",
                                     "<absolute path removed>/test/$1")
          lines.[i] <- Regex.Replace(lines.[i],
                                     "\"[a-zA-Z]:/[^\"]*?/([^\"/]*?\.dll\")",
                                     "\"<absolute path removed>/$1")
        else
          lines.[i] <- Regex.Replace(lines.[i].Replace('\\','/'),
                                     "[a-zA-Z]:/.*?FSharp.AutoComplete/test/(.*?(\"|$))",
                                     "<absolute path removed>/test/$1")

    // Write manually to ensure \n line endings on all platforms
    using (new StreamWriter(fn))
    <| fun f ->
        for line in lines do
          f.Write(line)
          f.Write('\n')
  b

Target "IntegrationTest" (fun _ ->
  let runOk =
   [ for i in integrationTests do
       yield runIntegrationTest i ]
   |> Seq.forall id
  if not runOk then
    failwith "Integration tests did not run successfully"
  else

    let ok, out, err =
      Git.CommandHelper.runGitCommand
                        "."
                        ("diff --exit-code " + integrationTestDir)
    if not ok then
      trace (toLines out)
      failwithf "Integration tests failed:\n%s" err
)


Target "BuildEmacs" (fun _ ->
  MSBuildDebug emacsBinDir "Build" ["./FSharp.AutoComplete.sln"]
  |> Log "Build-Output: "
)

module Emacs =
  let emacsd = "../emacs/"
  let srcFiles = !! (emacsd + "*.el")

  let testd = emacsd + "test/"
  let utils = !! (testd + "/test-common.el")

  let tmpd = emacsd + "tmp/"
  let bind = emacsd + "bin/"

  let exe =
    match buildServer with
    | AppVeyor -> Path.GetFullPath "../emacs-local/bin/emacs.exe"
    | _ -> @"emacs"

  let compileOpts =
    [ for f in srcFiles do
        yield sprintf """--batch --eval "(package-initialize)" --eval "(add-to-list 'load-path \"%s\")" --eval "(setq byte-compile-error-on-warn t)" -f batch-byte-compile %s"""
                ((Path.GetFullPath emacsd).Replace(@"\", @"\\").TrimEnd('\\'))
                f ]

  let checkDeclareOpts =
    sprintf """--batch --eval "(package-initialize)" --eval "(when (check-declare-directory \"%s\") (kill-emacs 1)))" """
      ((Path.GetFullPath emacsd).Replace(@"\", @"\\").TrimEnd('\\'))

  let makeLoad glob =
    [ for f in glob do yield "-l " + f ]
    |> String.concat " "

Target "EmacsTest" (fun _ ->
  if not (Directory.Exists (Path.GetFullPath Emacs.tmpd)) then
    Directory.CreateDirectory Emacs.tmpd |> ignore
  let home = Environment.GetEnvironmentVariable("HOME")
  Environment.SetEnvironmentVariable("HOME", Path.GetFullPath Emacs.tmpd)

  let loadFiles = Emacs.makeLoad Emacs.utils

  let tests =
    [ yield Emacs.exe, loadFiles + " --batch --eval \"(progn (set-default-coding-systems 'utf-8) (load-packages))\""
      yield Emacs.exe, loadFiles + " --batch -f run-fsharp-unit-tests"
      // AppVeyor doesn't currently run the integration tests
      if buildServer <> AppVeyor then
        yield Emacs.exe, loadFiles + " --batch -f run-fsharp-integration-tests"
      yield! [ for opts in Emacs.compileOpts do yield Emacs.exe, opts ]
      // AppVeyor also fails to run check-declare-directory
      if buildServer <> AppVeyor then
        yield Emacs.exe, Emacs.checkDeclareOpts ]

  ProcessTestRunner.RunConsoleTests
    (fun p -> { p with WorkingDir = Emacs.emacsd })
    tests

  Environment.SetEnvironmentVariable("HOME", home)
)

Target "Clean" (fun _ ->
  CleanDirs [ buildDir; buildReleaseDir; emacsBinDir ]
)

Target "Build" id
Target "Test" id
Target "All" id

"RestorePackages"
  ==> "BuildDebug"
  ==> "Build"
  ==> "IntegrationTest"

"RestorePackages"
  ==> "BuildEmacs"
  ==> "EmacsTest"

"EmacsTest" ==> "Test"
"IntegrationTest" ==> "Test"

"BuildDebug" ==> "All"
"Test" ==> "All"

RunTargetOrDefault "BuildDebug"

