/* 
  Copyright (C) 2008 Kai Hertel

	This file is part of mmpong.

	mmpong is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	mmpong is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __GAME_HEADER__
#define __GAME_HEADER__

#include <stdint.h>
#include <sys/time.h>
#include "dllhelper.h"

#ifdef __cplusplus
extern "C" {
#endif

#define PONG_SUCCESS 		0
#define PONG_PADHIT 		1
#define PONG_SCORE 			2
#define PONG_ARGINVALID 	(-1)
#define PONG_UNSUPPORTED 	(-2)
#define PONG_INTERROR 		(-3)

#define PONG_RANGE_SPREAD 	32767
#define GAME_NUM_EPS 		1.e-6

// Keep in mind: net traffic is of the essence and, to that end, messages are capped at 256 bytes at the moment
#define GAME_MAX_OBJECTS 	4
#define GAME_MAX_TEAMS 		2
#define GAMEBALL_DIR_MAX 	10.0f

struct netmessage; 	// forward declaration


// Everything is scaled to [ 0 ; 32767 ] representing float values in [ 0 ; 1 ]
// (Mainly due to the fact that transmitting float values over a network
//  independent of the architectures involved is rather non-trivial)
enum gamestatus {
	gamestatus_running= 'r',
	gamestatus_onhold= 'h',
	gamestatus_stall= 's'
};

enum gamemode {
	gamemode_linear= 0,
	gamemode_badminton
};

enum gamepadprofile {
	gamepadprofile_flat= 0,
	gamepadprofile_round
};

enum gameobjecttype {
	gameobjecttype_endmarker= 0,
	gameobjecttype_ball,
	gameobjecttype_paddle,
	gameobjecttype_wall
//	gameobjecttype_blackhole, ...
};

// no padding for structures related to the net code
#pragma pack(push, 1)

struct gameball_public {
	uint16_t pos[2]; 	// position
	uint16_t dir[2]; 	// direction & velocity in cartesian coordinates
	// future additions: spin & diameter
};

struct gamepaddle_public {
	uint16_t mean; 	// position mean
	uint16_t var; 	// variance
	uint16_t size; 	// racket size
//	uint16_t dir;   // velocity (obviously not an exact science in this context)
};

struct gamepaddle_attributes_public { 	// represents the team
	uint32_t score; 	// bind the score to the team
	uint32_t peers; 	// players in the team
	uint16_t profile; 	// paddle profile
};

struct gametime_public {
	uint32_t tv_sec;
	uint32_t tv_usec;
};

struct gameobject_public {
	uint16_t type;
	uint16_t dynlen; 	// determines to what extend parts of the object will be transmitted during game state updates
	union {
		struct gameball_public ball;
		struct {
			struct gamepaddle_public pad;
			struct gamepaddle_attributes_public pad_attr;
		} paddle;
		struct {
			// normals are computable and don't need to be sent over the net
			uint16_t start[2];
			uint16_t end[2];
			// possible additions: linear, curved, ... modes
		} wall;
	} desc;
};

// future additions:
// * set a `racket profile` independent of the game model (possibly for each team independently)
// * allow `wall profiles` for each wall
// * variable number of teams (0 - 4, since we use a square model; possibly up to six once we go cubic)
struct gameplay_public {
	uint16_t version; 			// gameplay version ( == sizeof(struct gameplay))
	struct gametime_public stamp; 	// time stamp
	uint16_t mode;
	uint16_t status;
	struct gameball_public ball;
	struct gamepaddle_public pad[GAME_MAX_TEAMS];
	struct gamepaddle_attributes_public pad_attr[GAME_MAX_TEAMS];
//	struct gameobject_public geo[GAME_MAX_OBJECTS];
	// extend for more elaborate game models
};

#pragma pack(pop)

struct gameball {
	float pos[2];
	float dir[2]; 
};

struct gamepaddle {
	float mean;
	float var;
	float size;
//	float dir; 
};

struct gamepaddle_attributes { 	// represents the team
	unsigned score; 	// bind the score to the team
	unsigned peers; 	// players in the team
	enum gamepadprofile profile; 	// paddle profile
};

struct gameobject {
	enum gameobjecttype type;
	short dynlen;
	// the following are to be embedded sooner or later in a union like the one seen in `struct gameobject`
	float normal[GAME_MAX_OBJECTS][2]; 	// normals
	float start[GAME_MAX_OBJECTS][2]; 	// start points
	float end[GAME_MAX_OBJECTS][2]; 	// end points
};

struct gameplay {
	uint16_t version;
	struct timeval stamp;
	enum gamemode mode;
	enum gamestatus status;
	struct gameball ball;
	struct gamepaddle pad[GAME_MAX_TEAMS];
	struct gamepaddle_attributes pad_attr[GAME_MAX_TEAMS];
	struct timeval lasthit; 	// stalled state detection
	struct gameobject geo;
};

EXPORT short gameplay_getversion(void);
EXPORT struct gameplay *gameplay_create(void);
EXPORT int gameplay_init(struct gameplay *, int, const enum gamemode, const enum gamepadprofile);
EXPORT int gameplay_update(struct gameplay *, const struct timeval *);
EXPORT int gameplay_public_to_internal(const struct gameplay_public *, struct gameplay *);
EXPORT int gameplay_internal_to_public(const struct gameplay *, struct gameplay_public *);
EXPORT const char *gameplay_spell(const enum gamemode, const enum gamepadprofile);
EXPORT int gameplay_parse(const char *, enum gamemode *, enum gamepadprofile *);
EXPORT int gameplay_apply_state(const struct netmessage *, struct gameplay *, uint16_t *);

#ifdef __cplusplus
}
#endif

#endif
