/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <QApplication>
#include <QComboBox>
#include <QDebug>
#include <QDialogButtonBox>
#include <QGridLayout>
#include <QGroupBox>
#include <QInputDialog>
#include <QLabel>
#include <QLineEdit>
#include <QListWidget>
#include <QMessageBox>
#include <QPushButton>
#include <QSettings>
#include <QStackedWidget>
#include <QToolButton>
#include <QTreeView>
#include <QVBoxLayout> 

#include "MvQKeyDialog.h"
#include "MvQKeyModel.h"
#include "MvQKeyManager.h"
#include "MvQTreeView.h"

#include "MvKeyProfile.h"

//===================================
//
// MvQKeyProfileDialog
//
//===================================


MvQKeyProfileDialog::MvQKeyProfileDialog(QString title,MvQKeyManager *manager,int currentRow,QWidget *parent) : QWidget(parent), manager_(manager)
{
	QVBoxLayout *layout=new QVBoxLayout;
	setLayout(layout);

	QHBoxLayout *hb = new QHBoxLayout;
	layout->addLayout(hb);

	profList_ = new QListWidget;
	//profList_->setProperty("mvStyle",0);
	//profList_->setAlternatingRowColors(true);
	hb->addWidget(profList_);
	
	QVBoxLayout *vb = new QVBoxLayout;
	hb->addLayout(vb);

	QPushButton *addPb=new QPushButton(tr("&New"));
	addPb->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/profile_add.svg"))));
	addPb->setToolTip(tr("Add new profile"));
 
	QPushButton *saveAsPb=new QPushButton(tr("Du&plicate"));
	saveAsPb->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/profile_duplicate.svg"))));
	saveAsPb->setToolTip(tr("Duplicate profile"));

	renamePb_=new QPushButton(tr("&Rename ..."));
	renamePb_->setToolTip(tr("Rename profile"));

	deletePb_=new QPushButton(tr("&Delete"));
	deletePb_->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/profile_delete.svg"))));
	deletePb_->setToolTip(tr("Delete profile"));

	vb->addWidget(addPb);
	vb->addWidget(saveAsPb);
	vb->addWidget(renamePb_);
	vb->addWidget(deletePb_);
	vb->addStretch(1);

	connect(profList_,SIGNAL(currentRowChanged (int)),
		this,SLOT(slotSelectProfile(int)));

	connect(addPb,SIGNAL(clicked()),
		this,SLOT(slotAddProfile()));

	connect(saveAsPb,SIGNAL(clicked()),
		this,SLOT(slotDuplicateProfile()));

	connect(deletePb_,SIGNAL(clicked()),
		this,SLOT(slotDeleteProfile()));

	connect(renamePb_,SIGNAL(clicked()),
		this,SLOT(slotRenameProfile()));


	init(currentRow);
}

void MvQKeyProfileDialog::init(int row)
{
   	for (vector<MvKeyProfile*>::const_iterator it=manager_->data().begin(); it != manager_->data().end(); it++) 
	{
		MvKeyProfile *prof = *it;
		QListWidgetItem *item=new QListWidgetItem(QString(prof->name().c_str()));

		if((*it)->systemProfile())
		{
			item->setIcon(QPixmap(QString::fromUtf8(":/window/metview_logo")));
		}

		profList_->addItem(item);		
	}

	if(profList_->count() >0)
		profList_->setCurrentRow(row);
}

void MvQKeyProfileDialog::slotUpdate()
{
	disconnect(profList_, SIGNAL(currentRowChanged(int)),this,0);
	int row=profList_->currentRow();
	profList_->clear();
	init(row);
	connect(profList_,SIGNAL(currentRowChanged (int)),
		this,SLOT(slotSelectProfile(int)));
	emit dataChanged();	 
}

void MvQKeyProfileDialog::slotSelectProfile(int row)
{
	if(row < 0 || row >= manager_->data().size())
		return;

	if(manager_->data().at(row)->systemProfile())
	{
		deletePb_->setEnabled(false);
		renamePb_->setEnabled(false);
	}
	else
	{
		deletePb_->setEnabled(true);
		renamePb_->setEnabled(true);	
	}

	emit currentProfileChanged(row);
}

void MvQKeyProfileDialog::slotAddProfile()
{
	bool ok;
     	QString text = QInputDialog::getText(this, tr("Add profile"),
                                          tr("New profile name:"), QLineEdit::Normal,
                                          "", &ok);
	if(ok && !text.isEmpty())
        {
	 	manager_->addProfile(text.toStdString());
		profList_->addItem(text);
		emit dataChanged();
		profList_->setCurrentRow(profList_->count()-1);
	}
}

void MvQKeyProfileDialog::slotDeleteProfile()
{
	int index=profList_->currentRow();
	if(index < 0 || index >= manager_->data().size())
		return;

	if(manager_->data().at(index)->systemProfile())
		return;

	QMessageBox msgBox;
	msgBox.setWindowTitle(tr("Delete profile"));

	QString str=tr("Are you sure that you want to delete profile: <b>");
	str+=QString::fromStdString(manager_->data().at(index)->name()) ;
	str+="</b>?";

 	msgBox.setText(str);
 	msgBox.setIcon(QMessageBox::Warning);
	msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::Cancel);
	msgBox.setDefaultButton(QMessageBox::Cancel);
	int ret=msgBox.exec();

	switch (ret) 
	{
   	case QMessageBox::Yes:
       	{	//keyModel_->profileIsAboutToChange();
		//keyModel_->setKeyProfile(0);

		manager_->deleteProfile(index);

		QListWidgetItem   *item=profList_->takeItem(index);
		delete item;

		emit dataChanged();

		if(index < profList_->count())
			profList_->setCurrentRow(index);
		else
			profList_->setCurrentRow(profList_->count()-1);
		//slotSelectProfile(0);
	}
      	break;
   	case QMessageBox::Cancel:
       	// Cancel was clicked
       	break;
   	default:
       	// should never be reached
      	 break;
	}
}

void MvQKeyProfileDialog::slotDuplicateProfile()
{
	int index=profList_->currentRow();
	if(index < 0 || index >= manager_->data().size() || manager_->data().size() == 0)
		return;

	bool ok;
     	QString text = QInputDialog::getText(this, tr("Duplicate profile"),
                                          tr("New profile name:"), QLineEdit::Normal,
                                          "", &ok);
	if(ok && !text.isEmpty())
        { 	
		MvKeyProfile* newProf=manager_->addProfile(text.toStdString(),manager_->data().at(index));
		newProf->setSystemProfile(false);

		profList_->addItem(text);
		emit dataChanged();
		profList_->setCurrentRow(profList_->count()-1);		
	}
}

void MvQKeyProfileDialog::slotRenameProfile()
{
	int index=profList_->currentRow();
	if(index < 0 || index >= manager_->data().size() || manager_->data().size() == 0)
		return;

	if(manager_->data().at(index)->systemProfile())
		return;

	bool ok;
     	QString text = QInputDialog::getText(this, tr("Rename profile"),
                                          tr("Profile name:"), QLineEdit::Normal,
                                          "", &ok);
	if(ok && !text.isEmpty())
        {
		MvKeyProfile *prof=manager_->data().at(index);
		prof->setName(text.toStdString());
		profList_->item(index)->setData(Qt::DisplayRole,text);
		emit dataChanged();	 	
	}
}


//=================================================
//
// MvQKeyListPag
//
//=================================================

MvQKeyListPage::MvQKeyListPage(QString title,MvQKeyManager *manager,QList<MvKeyProfile *> allKeys,
	QWidget *parent) : QWidget(parent), manager_(manager), allKey_(allKeys)
{
	//setWindowTitle(title);

	QVBoxLayout *layout=new QVBoxLayout;
	setLayout(layout);

	//Profiles

	QHBoxLayout *profHb=new QHBoxLayout;
	layout->addLayout(profHb);

	QLabel *label = new QLabel(tr("Profile:"));
	profCombo_ = new QComboBox;
	label->setBuddy(profCombo_);
	profHb->addWidget(label);
	profHb->addWidget(profCombo_);
	profHb->addStretch(1);

	QHBoxLayout *mainHb=new QHBoxLayout;
	layout->addLayout(mainHb);	

	//-----------------------------
	// Keys in a given profile
	//-----------------------------

	QGroupBox *keyBox= new QGroupBox(tr("Keys of profile"));
	mainHb->addWidget(keyBox);

	QVBoxLayout *vbKey = new QVBoxLayout;
	keyBox->setLayout(vbKey);

	QHBoxLayout *hbPb=new  QHBoxLayout;
	vbKey->addLayout(hbPb);

	addKeyTb_=new QToolButton(this);
	addKeyTb_->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/add.svg"))));
	addKeyTb_->setToolTip(tr("Add new key"));
	if(manager_->predefinedKeysOnly())
	{
	  	addKeyTb_->setEnabled(false);
	}	

	deleteKeyTb_=new QToolButton(this);
	deleteKeyTb_->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/remove.svg"))));
	deleteKeyTb_->setToolTip(tr("Delete key"));

	moveUpKeyTb_=new QToolButton(this);
	moveUpKeyTb_->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/arrow_up.svg"))));
	moveUpKeyTb_->setToolTip(tr("Move up key"));

	moveDownKeyTb_=new QToolButton(this);
	moveDownKeyTb_->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/arrow_down.svg"))));
	moveDownKeyTb_->setToolTip(tr("Move down key"));

	editKeyTb_=new QToolButton(this);
	editKeyTb_->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/configure.svg"))));
	editKeyTb_->setToolTip(tr("Edit key"));
	
	hbPb->addWidget(editKeyTb_);
	hbPb->addWidget(addKeyTb_);
	hbPb->addWidget(deleteKeyTb_);
	hbPb->addWidget(moveUpKeyTb_);
	hbPb->addWidget(moveDownKeyTb_);
	hbPb->addStretch(1);
	
	keyTree_ = new QTreeView;
	keyTree_->setRootIsDecorated(false);
	keyTree_->setObjectName("keyTree");
	//keyTree_->setProperty("mvStyle",0);
	//keyTree_->setAlternatingRowColors(true);
        keyTree_->setAllColumnsShowFocus(true);
	keyTree_->setDragEnabled(true);
 	keyTree_->setAcceptDrops(true);
	keyTree_->setHorizontalScrollMode(QAbstractItemView::ScrollPerPixel);
	vbKey->addWidget(keyTree_);

	keyModel_=new MvQKeyModel;
	keyTree_->setModel(keyModel_);

	//-----------------------------
	// Helper containing all the keys
	//-----------------------------

	QVBoxLayout *vbToProf = new QVBoxLayout;
	mainHb->addLayout(vbToProf);

	addKeyToProfTb_=new QToolButton(this);	
	addKeyToProfTb_->setToolTip(tr("Add key to key profile"));
	//addKeyToProfTb->setIcon(QIcon(QApplication::style()->standardIcon(QStyle::SP_ArrowLeft)));
	addKeyToProfTb_->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/arrow_left.svg"))));
	vbToProf->addWidget(addKeyToProfTb_);
	
	QGroupBox *allKeyBox= new QGroupBox(tr("All available keys"));
	mainHb->addWidget(allKeyBox);

	QVBoxLayout *vbAllKey = new QVBoxLayout;
	allKeyBox->setLayout(vbAllKey);

	QHBoxLayout *comboLayout=new  QHBoxLayout;
	QLabel *comboLabel = new QLabel(tr("Key categories:"));
	allKeyCombo_ = new QComboBox;
	comboLabel->setBuddy(allKeyCombo_);

	comboLayout->addWidget(comboLabel);
	comboLayout->addWidget(allKeyCombo_);
	//comboLayout->addStretch(1);
	vbAllKey->addLayout(comboLayout);

	allKeyTree_ = new QTreeView;
	allKeyTree_->setRootIsDecorated(false);
	allKeyTree_->setObjectName("allKeyTree");
	//allKeyTree_->setProperty("mvStyle",0);
	//allKeyTree_->setAlternatingRowColors(true);
        allKeyTree_->setAllColumnsShowFocus(true);
	allKeyTree_->setDragDropMode(QAbstractItemView::DragOnly);
	allKeyTree_->setHorizontalScrollMode(QAbstractItemView::ScrollPerPixel);
	vbAllKey->addWidget(allKeyTree_);

	allKeyModel_=new MvQKeyModel(MvQKeyModel::AllKeysMode);
	allKeyTree_->setModel(allKeyModel_);

	//-----------------------------
	// Signals and slots
	//-----------------------------
	
	connect(keyTree_,SIGNAL(doubleClicked(const QModelIndex&)),
		this,SLOT(slotEditTree(const QModelIndex&)));

	connect(addKeyTb_,SIGNAL(clicked()),
		this,SLOT(slotAddKey()));

	connect(deleteKeyTb_,SIGNAL(clicked()),
		this,SLOT(slotDeleteKey()));

	connect(moveUpKeyTb_,SIGNAL(clicked()),
		this,SLOT(slotMoveUpKey()));

	connect(moveDownKeyTb_,SIGNAL(clicked()),
		this,SLOT(slotMoveDownKey()));

	connect(editKeyTb_,SIGNAL(clicked()),
		this,SLOT(slotEditKey()));

	connect(addKeyToProfTb_,SIGNAL(clicked()),
		this,SLOT(slotAddKeyToProf()));

	init();	
	setAllKeyList(allKey_);
}

void MvQKeyListPage::init()
{
   	int i=0;
	for (vector<MvKeyProfile*>::const_iterator it=manager_->data().begin(); it != manager_->data().end(); it++) 
	{
		MvKeyProfile *prof = *it;
		profCombo_->addItem(QString(prof->name().c_str() ));
		if(prof->systemProfile())
		{
			profCombo_->setItemIcon(i,QPixmap(QString::fromUtf8(":/window/metview_logo")));
		}
		i++;		
	}


	profCombo_->setCurrentIndex(-1);

	connect(profCombo_,SIGNAL(currentIndexChanged(int)),
		this,SLOT(slotSelectProfile(int)));

	if(profCombo_->count() >0)
		profCombo_->setCurrentIndex(0);
}

void MvQKeyListPage::slotProfilesChanged()
{	
	QString currentProfName=profCombo_->currentText();

	disconnect(profCombo_, SIGNAL(currentIndexChanged(int)),0,0);

	//Clear the profile list
	profCombo_->clear();

	int i=0;
	int index=-1;
	for (vector<MvKeyProfile*>::const_iterator it=manager_->data().begin(); it != manager_->data().end(); it++) 
	{
		profCombo_->addItem(QString((*it)->name().c_str() ));
		if((*it)->name() == currentProfName.toStdString())
		{
			index=i;
		}
		if((*it)->systemProfile())
		{
			profCombo_->setItemIcon(i,QPixmap(QString::fromUtf8(":/window/metview_logo")));
		}
		i++;
	}

	profCombo_->setCurrentIndex(-1);

	connect(profCombo_, SIGNAL(currentIndexChanged(int)),  
		this,SLOT(slotSelectProfile(int)));

	if(index != -1)
	{
		profCombo_->setCurrentIndex(index);
	}
	else
	{
		profCombo_->setCurrentIndex(0);
	}

}

void MvQKeyListPage::slotChangeCurrentProfile(int row)
{
	profCombo_->setCurrentIndex(row);
}

void MvQKeyListPage::slotSelectProfile(int row)
{
	if(row < 0 || row >= manager_->data().size())
		return;

	MvKeyProfile* prof=manager_->data().at(row);

	keyModel_->profileIsAboutToChange();
	keyModel_->setKeyProfile(prof);
	resizeKeyTreeColumns();

	if(prof->systemProfile())
	{
	  	keyTree_->setEnabled(false);
		//keyTree_->setDragEnabled(false);
 		//keyTree_->setAcceptDrops(false);
		addKeyTb_->setEnabled(false);
		deleteKeyTb_->setEnabled(false);
		moveUpKeyTb_->setEnabled(false);
		moveDownKeyTb_->setEnabled(false);
		editKeyTb_->setEnabled(false);
		addKeyToProfTb_->setEnabled(false);
	}
	else
	{
		keyTree_->setEnabled(true);
		//keyTree_->setDragEnabled(true);
 		//keyTree_->setAcceptDrops(true);
		if(!manager_->predefinedKeysOnly())
		{
		  	addKeyTb_->setEnabled(true);
		}
		deleteKeyTb_->setEnabled(true);
		moveUpKeyTb_->setEnabled(true);
		moveDownKeyTb_->setEnabled(true);
		editKeyTb_->setEnabled(true);
		addKeyToProfTb_->setEnabled(true);
	}

}

void MvQKeyListPage::slotAddKey()
{
	/*MvKeyProfile *prof=keyModel_->profile();
	QModelIndex index=keyTree_->currentIndex();

	if(prof==0) //|| !index.isValid())
		return;

	bool ok;
     	QString text = QInputDialog::getText(this, tr("Add key"),
                                          tr("New key name:"), QLineEdit::Normal,
                                          "", &ok);
	if(ok && !text.isEmpty())
	{
		MvKey* key=new MvKey(text.toStdString(),text.toStdString());

         	keyModel_->profileIsAboutToChange();
		prof->addKey(key);
		keyModel_->setKeyProfile(prof);
		resizeKeyTreeColumns();
	}*/

	MvKeyProfile *prof=keyModel_->profile();
	QModelIndex index=keyTree_->currentIndex();

	if(prof==0) //|| !index.isValid())
		return;
	
	MvKey *key = new MvKey;	
	MvQKeyEditDialog keyDialog(key);

	if(keyDialog.exec()  == QDialog::Accepted && !key->name().empty())
	{  
		keyModel_->profileIsAboutToChange();
		prof->addKey(key);
		keyModel_->setKeyProfile(prof);
		resizeKeyTreeColumns();		
	}
	else
	{
	  	delete key;
	}
}

void MvQKeyListPage::slotDeleteKey()
{
	MvKeyProfile *prof=keyModel_->profile();
	QModelIndex index=keyTree_->currentIndex();

	if(prof==0 || !index.isValid() )
		return;

	QString str=tr("Are you sure that you want to delete key: <b>");
	str+=QString::fromStdString(prof->at(index.row())->name()) ;
	str+="</b>?";

 	QMessageBox msgBox;
	msgBox.setWindowTitle(tr("Delete key"));
	msgBox.setText(str);
	msgBox.setIcon(QMessageBox::Warning);
 	msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::Cancel);
	msgBox.setDefaultButton(QMessageBox::Cancel);
	int ret=msgBox.exec();

	switch (ret) 
	{
   	case QMessageBox::Yes:
       	{	keyModel_->profileIsAboutToChange();
		prof->deleteKey(index.row());
		keyModel_->setKeyProfile(prof);
		resizeKeyTreeColumns();
	}
      	break;
   	case QMessageBox::Cancel:
       	// Cancel was clicked
       	break;
   	default:
       	// should never be reached
      	 break;
	}
}

void MvQKeyListPage::slotMoveUpKey()
{
	MvKeyProfile *prof=keyModel_->profile();
	QModelIndex index=keyTree_->currentIndex();

	if(prof==0 || !index.isValid())
		return;

	if(index.row() > 0)
	{
		keyModel_->profileIsAboutToChange();
		prof->reposition(index.row(),index.row()-1);
		keyModel_->setKeyProfile(prof);
		resizeKeyTreeColumns();
		keyTree_->setCurrentIndex(keyModel_->index(index.row()-1,0));
	}
}

void MvQKeyListPage::slotMoveDownKey()
{
	MvKeyProfile *prof=keyModel_->profile();
	QModelIndex index=keyTree_->currentIndex();

	if(prof==0 || !index.isValid())
		return;

	if(index.row() < prof->size()-1)
	{
		keyModel_->profileIsAboutToChange();
		prof->reposition(index.row(),index.row()+1);
		keyModel_->setKeyProfile(prof);
		resizeKeyTreeColumns();
		keyTree_->setCurrentIndex(keyModel_->index(index.row()+1,0));
	}
}

void MvQKeyListPage::slotEditKey()
{
	MvKeyProfile *prof=keyModel_->profile();
	QModelIndex index=keyTree_->currentIndex();

	if(prof==0 || !index.isValid())
		return;

	if(index.row() <= prof->size()-1)
	{
		MvQKeyEditDialog edit(prof->at(index.row()));
		if(edit.exec()  == QDialog::Accepted)
		{
			//Not too efficient
			keyModel_->profileIsAboutToChange();
			keyModel_->setKeyProfile(prof);
			resizeKeyTreeColumns();
		}
	}
}

void MvQKeyListPage::slotEditTree(const QModelIndex& index)
{
	slotEditKey();
}

void MvQKeyListPage::slotAddKeyToProf()
{
	MvKeyProfile *allProf=allKeyModel_->profile();
	QModelIndex index=allKeyTree_->currentIndex();

	if(allProf==0 || !index.isValid() )
		return;

	MvKeyProfile *prof=keyModel_->profile();
	if(prof)
	{
		MvKey* key=allProf->at(index.row())->clone();

       		keyModel_->profileIsAboutToChange();
		prof->addKey(key);
		keyModel_->setKeyProfile(prof);
		resizeKeyTreeColumns();
	}
}	

void MvQKeyListPage::resizeKeyTreeColumns()
{
	for(int i=0;i < keyModel_->columnCount()-1; i++)
	{ 
		keyTree_->resizeColumnToContents(i);
	}
}

void MvQKeyListPage::setAllKeyList(QList<MvKeyProfile*> allKeys)
{
	allKeyTree_->setModel(allKeyModel_);

	allKey_=allKeys;

	foreach(MvKeyProfile* prof,allKey_)
	{
		allKeyCombo_->addItem(QString::fromStdString(prof->name()));
	}

	allKeyCombo_->setCurrentIndex(-1);

	connect(allKeyCombo_,SIGNAL(currentIndexChanged(int)),
		this,SLOT(slotSelectAllKeyGroup(int)));

	allKeyCombo_->setCurrentIndex(0);
} 

void MvQKeyListPage::slotSelectAllKeyGroup(int index)
{
	if(index >= 0 && index < allKey_.count() )
	{
		allKeyModel_->profileIsAboutToChange();
		allKeyModel_->setKeyProfile(allKey_[index]);
	}	
}


MvQKeyImportPage::MvQKeyImportPage(QString title,MvQKeyManager *manager,QWidget *parent) : QWidget(parent), manager_(manager)
{
	MvQKeyManager::KeyType type=MvQKeyManager::GribType;
	
	inManager_=new MvQKeyManager(type);
	inManager_->loadProfiles();

	QLabel *label;
	
	QVBoxLayout *layout=new QVBoxLayout;
	setLayout(layout);

	QHBoxLayout *hb = new QHBoxLayout;
	layout->addLayout(hb);

	label=new QLabel(tr("Profiles in GribExaminer:"));
	hb->addWidget(label);

	QHBoxLayout *hbMain = new QHBoxLayout;
	layout->addLayout(hbMain);

	profList_ = new QListWidget;
	hbMain->addWidget(profList_);
	
	QVBoxLayout *vb = new QVBoxLayout;
	hbMain->addLayout(vb);

	QPushButton *importPb=new QPushButton(tr("&Import"));
	importPb->setIcon(QIcon(QPixmap(QString::fromUtf8(":/keyDialog/profile_import.svg"))));
	importPb->setToolTip(tr("Import selected profiles"));
 
	QPushButton *selectAllPb=new QPushButton(tr("&Select all"));
	selectAllPb->setToolTip(tr("Select all profiles"));

        QPushButton *unselectAllPb=new QPushButton(tr("&Unselect all"));
	unselectAllPb->setToolTip(tr("Unselect all profiles"));

	vb->addWidget(importPb);
	vb->addWidget(selectAllPb);
	vb->addWidget(unselectAllPb);
	vb->addStretch(1);

	connect(importPb,SIGNAL(clicked()),
		this,SLOT(slotImport()));

	connect(selectAllPb,SIGNAL(clicked()),
		this,SLOT(slotSelectAll()));

	connect(unselectAllPb,SIGNAL(clicked()),
		this,SLOT(slotUnselectAll()));

	init();
}

MvQKeyImportPage::~MvQKeyImportPage()
{
	delete inManager_;
}

void MvQKeyImportPage::init()
{
  	//QStringList items;
   	for (vector<MvKeyProfile*>::const_iterator it=inManager_->data().begin(); it != inManager_->data().end(); it++) 
	{
		MvKeyProfile *prof = *it;
		QListWidgetItem *item=new QListWidgetItem(profList_);
		item->setText(QString(prof->name().c_str()));

		if(prof->systemProfile())
		{
			item->setFlags(Qt::NoItemFlags);
			item->setIcon(QPixmap(QString::fromUtf8(":/window/metview_logo")));
		}
		else
		{
			item->setFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled |Qt::ItemIsUserCheckable);
			item->setCheckState(Qt::Unchecked);
		}		
	}

	if(profList_->count() >0)
		profList_->setCurrentRow(0);
}

void MvQKeyImportPage::slotSelectAll()
{
	for(int i=0; i < profList_->count(); i++)
	{
		if(profList_->item(i)->flags() & Qt::ItemIsUserCheckable)
			profList_->item(i)->setCheckState(Qt::Checked);
	}
}

void MvQKeyImportPage::slotUnselectAll()
{
	for(int i=0; i < profList_->count(); i++)
	{
		if(profList_->item(i)->flags() & Qt::ItemIsUserCheckable)
			profList_->item(i)->setCheckState(Qt::Unchecked);
	}
}

void MvQKeyImportPage::slotImport()
{
	bool done=false;
	for(int i=0; i < profList_->count(); i++)
	{
		if(profList_->item(i)->flags() & Qt::ItemIsUserCheckable &&
		   profList_->item(i)->checkState() == Qt::Checked)
		{
			MvKeyProfile *prof=inManager_->data().at(i);
			string name= prof->name();
	
			QString qsName(name.c_str());			
			qsName=manager_->findUniqueProfileName(qsName);
			
			manager_->addProfile(qsName.toStdString(),prof,inManager_->keyType());
			done = true;
		}					
	}

	if(done)
		emit dataChanged();
}

MvQKeyEditDialog::MvQKeyEditDialog(MvKey *key,QWidget *parent) : QDialog(parent), key_(key)
{
	setWindowTitle(tr("Edit key"));

	QVBoxLayout *layout=new QVBoxLayout;
	setLayout(layout);

	QGridLayout *grid = new QGridLayout;
	layout->addLayout(grid);

	int row=0;
	
	grid->addWidget(new QLabel(tr("Name:")),row,0);
	nameLe_ = new QLineEdit(key->name().c_str());
	if(!key->editable())
	{
		nameLe_->setReadOnly(true);
		nameLe_->setToolTip(tr("Read only item!"));
	}
	grid->addWidget(nameLe_,row,1);
	row++;

	grid->addWidget(new QLabel(tr("Header:")),row,0);
	shortLe_ = new QLineEdit(key->shortName().c_str());
	grid->addWidget(shortLe_,row,1);
	row++;
	
	grid->addWidget(new QLabel(tr("Description:")),row,0);
	descLe_ = new QLineEdit(key->description().c_str());
	if(!key->editable())
	{
		descLe_->setReadOnly(true);
		descLe_->setToolTip(tr("Read only item!"));
	}
	grid->addWidget(descLe_,row,1);
	row++;

	//colourCb_=0;

	for(map<string,string>::const_iterator it=key->metaData().begin(); it != key->metaData().end(); it++)
	{
		QWidget *w;

		grid->addWidget(new QLabel(QString::fromStdString(it->first) + ":"),row,0);
		/*if(it->first == "colour" && !colourCb_)
		{
			colourCb_ = new QComboBox;
			colourCb_->setStandardColors();
			colourCb_->setColorDialogEnabled(true);
			colourCb_->setCurrentColor(QColor(QString::fromStdString(it->second)));
			w=colourCb_;
		}
		else
		{*/	
			metaLe_[it->first] = new QLineEdit(QString::fromStdString(it->second));
			if(!key->editable())
			{
				metaLe_[it->first]->setReadOnly(true);
				metaLe_[it->first]->setToolTip(tr("Read only item!"));
			}
			w=metaLe_[it->first];		
		//}

		grid->addWidget(w,row,1);
		row++;	 
	}

	QDialogButtonBox *buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok
                                      | QDialogButtonBox::Cancel);
 
	connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
     	connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
	
	layout->addWidget(buttonBox);
}

MvQKeyEditDialog::~MvQKeyEditDialog()
{

}

void MvQKeyEditDialog::accept()
{
	key_->setName(nameLe_->text().toStdString());
	key_->setShortName(shortLe_->text().toStdString());
	key_->setDescription(descLe_->text().toStdString());
	QMapIterator<string, QLineEdit*> it(metaLe_);
	while (it.hasNext()) 
	{
     		it.next();
     		key_->setMetaData(it.key(),it.value()->text().toStdString());
	}	

	/*if(colourCb_)
	{
		key_->setMetaData("colour",colourCb_->currentColor().name().toStdString());
	}*/	

	QDialog::accept();	
}

void MvQKeyEditDialog::reject()
{
	QDialog::reject();
}


//======================================
//
// MvQKeyDialog
//
//=======================================

MvQKeyDialog::MvQKeyDialog(QString title,MvQKeyManager::KeyType type,
	MvQKeyManager *manager,int  currentRow,QList<MvKeyProfile *> allKeys,
	QWidget *parent) :  
	MvQConfigDialog(title,parent), manager_(manager)
{			
	MvQKeyProfileDialog *profPage=new MvQKeyProfileDialog("Profiles",manager_,currentRow,this);
	QIcon iconProf(QPixmap(QString::fromUtf8(":/keyDialog/profile.svg")));
	addPage(profPage,iconProf,"Profiles");

	MvQKeyListPage *keyPage=new MvQKeyListPage("Keys",manager_,allKeys,this);
	QIcon iconKey(QPixmap(QString::fromUtf8(":/keyDialog/key.svg")));
	addPage(keyPage,iconKey,"Keys");

	connect(profPage,SIGNAL(currentProfileChanged(int)),
		keyPage,SLOT(slotChangeCurrentProfile(int)));

	connect(profPage,SIGNAL(dataChanged()),
		keyPage,SLOT(slotProfilesChanged()));

	keyPage->slotChangeCurrentProfile(currentRow);
		
	if(type == MvQKeyManager::FrameType)
	{	
		MvQKeyImportPage *importPage=new MvQKeyImportPage("Import from GribExaminer",manager_,this);
		QIcon iconImport(QPixmap(QString::fromUtf8(":/keyDialog/profile_import.svg")));
		addPage(importPage,iconImport,"Import");
		
		connect(importPage,SIGNAL(dataChanged()),
			profPage,SLOT(slotUpdate()));
	}

	readSettings();
}


MvQKeyDialog::~MvQKeyDialog()
{
	//if(manager_)
	//	delete manager_;
	writeSettings();
}

void MvQKeyDialog::accept()
{
	//manager_->saveProfiles();

	//hide();

	//emit dataChanged(manager_);

	//delete manager_;

	QDialog::accept();	
}

void MvQKeyDialog::reject()
{
	//if(manager_)
	//	delete manager_;

	QDialog::reject();
}

void MvQKeyDialog::writeSettings()
{
	QSettings settings("ECMWF","MV4-MvQKeyDialog");
	
	settings.beginGroup("dialog");
	settings.setValue("geometry",saveGeometry());
	settings.setValue("currentRow",list_->currentRow());
	//settings.setValue("state",saveState());
	settings.endGroup();
}

void MvQKeyDialog::readSettings()
{
	QSettings settings("ECMWF","MV4-MvQKeyDialog");
	
	settings.beginGroup("dialog");
	restoreGeometry(settings.value("geometry").toByteArray());
	//restoreState(settings.value("state").toByteArray());

	int ival;
	if(settings.value("currentRow").isNull())
	{
	  	ival=1;
	}	
	else
	{  
		ival=settings.value("currentRow").toInt();
	}
	
	list_->setCurrentRow(ival);

	settings.endGroup();
}
