/*-------------------------------------------------------------------------
 * Copyright (c) 2000-2002 Kenneth W. Sodemann (stufflehead@bigfoot.com)
 *-------------------------------------------------------------------------
 * player
 *
 * Synopsis:
 *   The player class.
 *
 * $Id: player.c,v 1.3 2002/05/24 22:11:39 stuffle Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330 
 * Boston, MA  02111-1307  USA
 *
 *-------------------------------------------------------------------------
 */
#define __PLAYER_BODY

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>

#include "player.h"


typedef struct player_object
{
   GString     *name;
   gint         score;
   gint         guesses;
   gint         correct;
   gint         bonus;
   player_type  type;
} player_object;


player_object * 
create_player (const gchar *name, player_type type)
{
   player_object   *new_player = NULL;

   new_player = g_malloc (sizeof (player_object));
   new_player->name = g_string_new (name);
   new_player->score = 0;
   new_player->guesses = 0;
   new_player->correct = 0;
   new_player->bonus = 0;
   new_player->type = type;

   return new_player;
}


void
destroy_player (player_object *player)
{
   g_return_if_fail (player != NULL);
   g_return_if_fail (player->name != NULL);
   g_return_if_fail (player->type == HUMAN || player->type == COMPUTER);

   g_string_free (player->name, TRUE);
   player->name = NULL;
   player->type = 0;
   g_free (player);
}


void
reset_player (player_object *player) 
{
   g_return_if_fail (player != NULL);
   g_return_if_fail (player->name != NULL);
   g_return_if_fail (player->type == HUMAN || player->type == COMPUTER);

   player->score = 0;
   player->guesses = 0;
   player->correct = 0;
   player->bonus = 0;
} 


gint
update_player_stats (player_object *player,
                     gint           score,
                     gboolean       is_bonus)
{
   g_return_val_if_fail (player != NULL, 0);
   g_return_val_if_fail (player->name != NULL, 0);
   g_return_val_if_fail 
      (player->type == HUMAN || player->type == COMPUTER, 0);

   player->score += score;

   if (is_bonus)
      {
      player->bonus += score;
      }
   else
      {
      player->guesses++;
      if (score > 0)
         {
         player->correct++;
         }
      }

   return player->score;
}


player_type
get_type_of_player (const player_object *player) 
{
   g_return_val_if_fail (player != NULL, HUMAN);
   g_return_val_if_fail (player->name != NULL, HUMAN);
   g_return_val_if_fail 
      (player->type == HUMAN || player->type == COMPUTER, HUMAN);

   return player->type;
} 


gint
get_player_guesses (const player_object *player)
{
   g_return_val_if_fail (player != NULL, 0);
   g_return_val_if_fail (player->name != NULL, 0);
   g_return_val_if_fail 
      (player->type == HUMAN || player->type == COMPUTER, 0);

   return player->guesses;
}

gint
get_player_correct_guesses (const player_object *player)
{
   g_return_val_if_fail (player != NULL, 0);
   g_return_val_if_fail (player->name != NULL, 0);
   g_return_val_if_fail 
      (player->type == HUMAN || player->type == COMPUTER, 0);

   return player->correct;
}

gint
get_player_bonus (const player_object *player)
{
   g_return_val_if_fail (player != NULL, 0);
   g_return_val_if_fail (player->name != NULL, 0);
   g_return_val_if_fail 
      (player->type == HUMAN || player->type == COMPUTER, 0);

   return player->bonus;
}


gint
get_player_score (const player_object *player) 
{
   g_return_val_if_fail (player != NULL, 0);
   g_return_val_if_fail (player->name != NULL, 0);
   g_return_val_if_fail 
      (player->type == HUMAN || player->type == COMPUTER, 0);

   return player->score;
}


const gchar * 
get_player_name (const player_object *player) 
{
   g_return_val_if_fail (player != NULL, 0);
   g_return_val_if_fail (player->name != NULL, 0);
   g_return_val_if_fail 
      (player->type == HUMAN || player->type == COMPUTER, 0);

   return player->name->str;
}

#undef __PLAYER_BODY
