#include <cmath>
#include "RenderArea.h"
#include "Graph.h"
#include "MainWindow.h"

RenderArea::RenderArea(Graph* parent) :
    QWidget(parent)
{
    mainWindow = parent->mainWindow;
    logicalRectange = QRectF();
    transform = QTransform();
    connectPoints = true;
}

RenderArea::~RenderArea()
{
}

void RenderArea::resizeEvent(QResizeEvent* event)
{
    plot();
    QWidget::resizeEvent(event);
}

void RenderArea::plot()
{
    image = QImage(size(), QImage::Format_RGB32);
    image.fill(Qt::white); // works only as of Qt 4.8
    //    image->fill(QColor::QColor (255, 255, 255));
    transform.reset();
    transform.scale(image.width() / (logicalRectange.right() - logicalRectange.left()),
                    image.height() / (logicalRectange.top() - logicalRectange.bottom()));
    transform.translate(-logicalRectange.left(), -logicalRectange.bottom());

    QPainter painter(&image);
    QTransform invertedTransform(transform.inverted());
    qreal x, prevX;
    QVariant y1, y2, y3, prevY1, prevY2, prevY3;
    painter.setPen(Qt::lightGray);

    // draw cross
    painter.drawLine(transform.map(QPointF(0, logicalRectange.bottom())), transform.map(QPointF(0, logicalRectange.top())));
    painter.drawLine(transform.map(QPointF(logicalRectange.left(), 0)), transform.map(QPointF(logicalRectange.right(), 0)));

    // draw marks
    qreal factor(1);
    if (static_cast<Graph*>(parent())->scaleStyle == Graph::Radians)
        factor = 3.141592;

    int xp, yp;
    QPointF p;
    QString label;

    // major marks
    QList<qreal>marksList = marks(logicalRectange.left() / factor, logicalRectange.right() / factor);
    foreach (qreal x, marksList) {
        label = tr("%1").arg(x);
        if (factor == 3.141592)
            label += QChar(7464); // Pi
        x *= factor;

        p = transform.map(QPointF(x, 0));
        xp = p.x();
        yp = p.y();
        painter.drawLine(QPoint(xp, yp - 5), QPoint(xp, yp + 5));

        painter.drawText(QPoint(xp + 1, yp + 12), label);
    }

    marksList = marks(logicalRectange.top(), logicalRectange.bottom());
    foreach (qreal y, marksList) {
        label = tr("%1").arg(y);

        p = transform.map(QPointF(0, y));
        xp = p.x();
        yp = p.y();
        painter.drawLine(QPoint(xp - 5, yp), QPoint(xp + 5, yp));

        painter.drawText(QPoint(xp + 12, yp + 1), label);
    }

    // minor marks
    marksList = marks(logicalRectange.left() / factor, logicalRectange.right() / factor, 10);
    foreach (qreal x, marksList) {
        x *= factor;

        p = transform.map(QPointF(x, 0));
        xp = p.x();
        yp = p.y();
        painter.drawLine(QPoint(xp, yp - 2), QPoint(xp, yp + 2));
    }

    marksList = marks(logicalRectange.top(), logicalRectange.bottom(), 10);
    foreach (qreal y, marksList) {
        p = transform.map(QPointF(0, y));
        xp = p.x();
        yp = p.y();
        painter.drawLine(QPoint(xp - 2, yp), QPoint(xp + 2, yp));
    }

    for(int i = 0; i < image.width(); ++i) {
        x = invertedTransform.map(QPointF(i, 0)).x();
        y1 = mainWindow->evaluate(x, static_cast<Graph*>(parent())->y1LineEdit->text());
        y2 = mainWindow->evaluate(x, static_cast<Graph*>(parent())->y2LineEdit->text()).toReal();
        y3 = mainWindow->evaluate(x, static_cast<Graph*>(parent())->y3LineEdit->text()).toReal();

        if (!y1.isNull()) {
            painter.setPen(Qt::blue);
            if (connectPoints && !prevY1.isNull())
                painter.drawLine(transform.map(QPointF(x, prevY1.toReal())), transform.map(QPointF(x, y1.toReal())));
            else
                painter.drawPoint(transform.map(QPointF(x, y1.toReal())));
        }

        if (!y2.isNull()) {
            painter.setPen(Qt::red);
            if (connectPoints && !prevY2.isNull())
                painter.drawLine(transform.map(QPointF(x, prevY2.toReal())), transform.map(QPointF(x, y2.toReal())));
            else
                painter.drawPoint(transform.map(QPointF(x, y2.toReal())));
        }

        if (!y3.isNull()) {
            painter.setPen(Qt::green);
            if (connectPoints && !prevY3.isNull())
                painter.drawLine(transform.map(QPointF(x, prevY3.toReal())), transform.map(QPointF(x, y3.toReal())));
            else
                painter.drawPoint(transform.map(QPointF(x, y3.toReal())));
        }

        prevX = x;
        prevY1 = y1;
        prevY2 = y2;
        prevY3 = y3;
    }

    painter.end();
    repaint();
}

QList<qreal> RenderArea::marks(qreal lowerBound, qreal upperBound, int minors)
{
    QList<qreal> list;
    qreal a(0.2);
    a += std::log10((double) minors);
    qreal interval(std::pow(10.0, std::floor(std::log10(upperBound - lowerBound) - a)));
    qreal lowerMark = lowerBound - std::fmod(lowerBound, interval);
    lowerMark += lowerMark < lowerBound ? interval : 0;
    for (qreal aMark = lowerMark; aMark < upperBound; aMark += interval) {
        if (std::abs(aMark) < interval /2)
            aMark = 0;
        list << aMark;
    }
    return list;
}


void RenderArea::paintEvent(QPaintEvent* event)
{
    QPainter painter(this);
    painter.drawImage(QPoint(0, 0), image);
    painter.end();
}

void RenderArea::mousePressEvent(QMouseEvent* event)
{
    selection = QRect(event->pos(),event->pos());
    originalImage = image;
}

void RenderArea::mouseMoveEvent(QMouseEvent* event)
{
    image = originalImage;
    selection.setBottomRight(event->pos());
    QPainter painter(&image);
    //    painter.begin(this);
    painter.setPen(Qt::DashLine);
    painter.drawRect(selection);
    painter.end();
    update();
}

void RenderArea::mouseReleaseEvent(QMouseEvent* event)
{
    if (selection.width() != 0 || selection.height() != 0) {
        QTransform invertedTransform(transform.inverted());
        QRectF selF;
        if (selection.top() < selection.bottom()) {
            selF.setTop(selection.top());
            selF.setBottom(selection.bottom());
        } else {
            selF.setTop(selection.bottom());
            selF.setBottom(selection.top());

        }
        if (selection.left() < selection.right()) {
            selF.setLeft(selection.left());
            selF.setRight(selection.right());
        } else {
            selF.setLeft(selection.right());
            selF.setRight(selection.left());
        }
        //        logicalRectange.setTopLeft(invertedTransform.map(QPointF(selection.topLeft())));
        //        logicalRectange.setBottomRight(invertedTransform.map(QPointF(selection.bottomRight())));
        logicalRectange = invertedTransform.mapRect(selF);

        static_cast<Graph*>(parent())->repopulateParameterEdits();
        plot();
    }
}
