import subprocess


def convert_value(value):
    """Make output agreeable to xfconf."""
    if isinstance(value, bool):
        if value is True:
            value = 'true'
        else:
            value = 'false'
    return value


def xfconf_init_property(channel, p_name, p_type, initial_value):
    """Initialize the specified xfconf property."""
    p_type = p_type.__name__
    initial_value = str(convert_value(initial_value))
    cmd = 'xfconf-query -c %s -p %s -n -s %s -t %s' % (channel, p_name,
                                                       initial_value, p_type)
    subprocess.call(cmd.split())


def xfconf_list_properties(channel):
    """List the properties defined for the given channel."""
    settings = dict()
    cmd = 'xfconf-query -c %s -l -v' % channel
    for line in subprocess.check_output(cmd, shell=True).split('\n'):
        try:
            key, value = line.split(None, 1)
        except ValueError:
            key = line.strip()
            value = ""
        if str.isdigit(value):
            value = int(value)
        elif value.lower() in ['true', 'false']:
            value = value.lower() == 'true'
        else:
            value = str(value)
        settings[key] = value
    return settings


def xfconf_set_property(channel, prop, value):
    """Set the specified xfconf property."""
    value = str(value).lower()
    cmd = 'xfconf-query -c %s -p %s -s %s' % (channel, prop, value)
    subprocess.call(cmd.split())


class XfceSessionSync():
    """
    Class to set/get xfce4-session lock settings.
    """
    def __init__(self):
        """Initialize the XfceSessionSync instance."""
        self.settings = {'/shutdown/LockScreen': False}
        current_settings = self._get_xfce4_session_settings()
        self._update_settings(current_settings)
        self._init_xfconf_properties(current_settings)

    def _init_xfconf_properties(self, current_settings):
        """If xfce4-session has not been configured, some of its properties
        may not exist. Create any missing properties."""
        channel = 'xfce4-session'
        for key, value in list(self.settings.items()):
            if key not in list(current_settings.keys()):
                xfconf_init_property(channel, key, type(value), value)

    def _get_xfce4_session_settings(self):
        """Return a dictionary of the xfce4-session settings."""
        return xfconf_list_properties('xfce4-session')

    def _update_settings(self, settings):
        """Update the internal settings."""
        for key, value in list(settings.items()):
            if key in list(self.settings.keys()):
                self.settings[key] = value

    def get_lock(self):
        """Return True if Lock on Sleep is enabled."""
        return self.settings['/shutdown/LockScreen']

    def set_lock(self, value):
        """Set the Lock on Sleep setting."""
        xfconf_set_property('xfce4-session', '/shutdown/LockScreen', value)
        self.settings['/shutdown/LockScreen'] = value


class XfpmSync():
    """
    Class to set/get xserver dpms timings via xfpm, thus keeping xfpm in sync.
    """
    def __init__(self):
        '''Following settings should concur with xfpm defaults'''
        self.settings = {'/xfce4-power-manager/dpms-enabled': True,
                         '/xfce4-power-manager/dpms-on-battery-sleep': 5,
                         '/xfce4-power-manager/dpms-on-battery-off': 10,
                         '/xfce4-power-manager/dpms-on-ac-sleep': 10,
                         '/xfce4-power-manager/dpms-on-ac-off': 15,
                         '/xfce4-power-manager/lock-screen-suspend-hibernate':
                            False
                         }
        current_settings = self._get_xfpm_settings()
        self._update_settings(current_settings)
        self._init_xfconf_properties(current_settings)

    def _init_xfconf_properties(self, current_settings):
        """
        If xfpm has never been used, some xfconf channel properties may not be
        set. Ensures that we don't get complains about missing properties.
        """
        channel = 'xfce4-power-manager'
        for key, value in list(self.settings.items()):
            if key not in list(current_settings.keys()):
                xfconf_init_property(channel, key, type(value), value)

    def _get_power_state(self):
        """
        Get the power state from /sys/class/power_supply/AC/online if present.
        Returns:
            1: AC connected
            0: AC disconnected
        (Do we always have this file, if a battery is present?)
        """
        try:
            with open('/sys/class/power_supply/AC/online', 'rb') as f:
                pstate = int(f.readline())
        except:
            pstate = 1

        return pstate

    def _get_xfpm_settings(self):
        """Returns xfpm xfconf settings as string"""
        return xfconf_list_properties('xfce4-power-manager')

    def _update_settings(self, settings):
        """Update the internal settings."""
        for key, value in list(settings.items()):
            if key in list(self.settings.keys()):
                self.settings[key] = value

    def _get_timings_scope(self):
        """Return the timings scope (Batter or AC)."""
        if self._get_power_state() == 0:
            sleep = '/xfce4-power-manager/dpms-on-battery-sleep'
            off = '/xfce4-power-manager/dpms-on-battery-off'
        else:  # we assume on ac for all other cases
            sleep = '/xfce4-power-manager/dpms-on-ac-sleep'
            off = '/xfce4-power-manager/dpms-on-ac-off'

        return sleep, off

    def set_dpms(self, sleep_time, off_time):
        """
        For most monitors *_sleep and *_off have probably the same outcome.
        So, basically we set the lower value for the corresponding power state.
        But xfpm ignores settings not in:
            0 <= *_sleep <= *_off
        or
            *_off == 0 and *_sleep >= 0
        i.e. sometimes we have to adapt both.
        """
        sleep, off = self._get_timings_scope()

        if not (sleep_time == 0 and off_time == 0):
            if sleep_time + 1 > off_time:
                off_time = sleep_time + 1

        xfconf_set_property('xfce4-power-manager', off, off_time)
        self.settings[off] = off_time

        xfconf_set_property('xfce4-power-manager', sleep, sleep_time)
        self.settings[sleep] = sleep_time

    def get_dpms(self):
        """Return the current DPMS timeouts."""
        sleep, off = self._get_timings_scope()
        return self.settings[sleep], self.settings[off]

    def get_lock(self):
        """Return True if Lock on Suspend/Hibernate is enabled."""
        prop_name = '/xfce4-power-manager/lock-screen-suspend-hibernate'
        return self.settings[prop_name]

    def set_lock(self, value):
        """Set the Lock on Suspend/Hibernate setting."""
        prop_name = '/xfce4-power-manager/lock-screen-suspend-hibernate'
        xfconf_set_property('xfce4-power-manager', prop_name, value)
        self.settings[prop_name] = value
