/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


#ifndef PUBLIC_VERIFY_X509_DATATYPES
#define PUBLIC_VERIFY_X509_DATATYPES

#include <stdio.h>

#include <openssl/x509.h>
#include <openssl/evp.h>
#include <openssl/asn1.h>


/* used for enabling or disabling in verify_X509_setParameter() */
#define VERIFY_ENABLE  1
#define VERIFY_DISABLE 2

#define GLOBUS_PROXY_V3_OID    "1.3.6.1.4.1.3536.1.222"
#define GLOBUS_PROXY_V3_SN     "gt3ProxyCertInfo"
#define GLOBUS_PROXY_V3_LN     "GT3 Proxy Certificate Information"

#define RFC_PROXY_OID      "1.3.6.1.5.5.7.1.14"
#define RFC_PROXY_SN       "proxyCertInfo"
#define RFC_PROXY_LN       "Proxy Certificate Information"

#define ANY_LANGUAGE_OID         "1.3.6.1.5.5.7.21.0"
#define ANY_LANGUAGE_SN          "ANY_LANGUAGE"
#define ANY_LANGUAGE_LN          "Any Language"

#define IMPERSONATION_PROXY_OID         "1.3.6.1.5.5.7.21.1"
#define IMPERSONATION_PROXY_SN          "IMPERSONATION_PROXY"
#define IMPERSONATION_PROXY_LN          "GSI impersonation proxy"

#define INDEPENDENT_PROXY_OID           "1.3.6.1.5.5.7.21.2"
#define INDEPENDENT_PROXY_SN            "INDEPENDENT_PROXY"
#define INDEPENDENT_PROXY_LN            "GSI independent proxy"

#define LIMITED_PROXY_OID               "1.3.6.1.4.1.3536.1.1.1.9"
#define LIMITED_PROXY_SN                "LIMITED_PROXY"
#define LIMITED_PROXY_LN                "GSI limited proxy"

/* used for parsing whether there is a VOMS AC extension */
#define VOMS_AC_OID	"1.3.6.1.4.1.8005.100.100.5"


/* verify_x509_setParameter options */
typedef enum {
    VERIFY_X509_CA_PATH                  =(unsigned long)  11000,
    VERIFY_X509_CERTIFICATE_FILEPATH     =(unsigned long)  11001,
    VERIFY_X509_CERTIFICATE_F_HANDLE     =(unsigned long)  11002,
    VERIFY_X509_CERTIFICATE_PEM          =(unsigned long)  11003,
    VERIFY_X509_PRIVATEKEY_FILE          =(unsigned long)  11004,
    VERIFY_X509_PRIVATEKEY_PEM           =(unsigned long)  11005,
    VERIFY_X509_CRL_PATH                 =(unsigned long)  11006,
    VERIFY_X509_OCSP_RESPONDER_URI       =(unsigned long)  11007,
    VERIFY_X509_STACK_OF_X509            =(unsigned long)  12101,
    VERIFY_X509_EVP_PKEY                 =(unsigned long)  12102,

    VERIFY_X509_OPTIONS_NO_CRL_CHECK             =(unsigned long)  30001,
    VERIFY_X509_OPTIONS_ALLOW_LIMITED_PROXY      =(unsigned long)  30002,
    VERIFY_X509_OPTIONS_MUST_HAVE_PRIV_KEY       =(unsigned long)  30004,
    VERIFY_X509_OPTIONS_REQUIRE_LIMITED_PROXY    =(unsigned long)  30005,

    VERIFY_X509_OPTIONS_VERIFY_AT_NOTBEFORE      =(unsigned long)  30006
} verify_x509_option_t;


/**
 * Error codes (reasons) as described in man ERR_GET_REASON
 */
typedef enum {
    VER_R_X509_PARAMS_OK                   = 100,
    VER_R_X509_PARAMS_ALREADY_SET          = 101,
    VER_R_X509_PARAMS_UNSUPPORTED_DATATYPE = 102,
    VER_R_X509_PARAMS_ACCESS_FAILURE       = 103,
    VER_R_X509_PARAMS_DATA_INCORRECT       = 104,
    VER_R_X509_PARAMS_DATA_EMPTY           = 105,
    VER_R_X509_PARAMS_CONTAINER_FAILURE    = 106,

    VER_R_NO_CACERT			   = 201,
    VER_R_CERTSTACK_EMPTY		   = 202,
    VER_R_PARAMETER_EMPTY		   = 203,
    VER_R_LIMITED_DISABLED		   = 204,
    VER_R_NOPRIVATEKEY_DISABLED		   = 205,

    VER_R_X509_VERIFY_CERT_FAILURE	   = 301
} verify_x509_error_t;


/**
 * Proxy certificate info
 */
typedef enum {
    GT2_TYPE = 0x0004,
    GT3_TYPE = 0x0008,
    RFC_TYPE = 0x0010,
    ANY_PROXY_VERSION = 0x00FC    /* all except 1 (CA) and 2 (EEC) */
} proxy_cert_info_type_t;

/**
 * Policy language extension
 */
typedef enum {
    IMPERSONATION = 0x0100,
    LIMITED       = 0x0200,
    INDEPENDENT   = 0x0400,
    RESTRICTED    = 0x0800,
    ANYLANG       = 0x1000,
    ALL_LANGUAGE  = 0xFF00
} policy_language_type_t;

/**
 * Different type of certs and proxies. Combine proxies from proxy cert info and
 * policy language.
 */
typedef enum {
    NONE		  = 0,
    CA			  = 1,
    EEC			  = 2,
    GT2_PROXY		  = GT2_TYPE|IMPERSONATION,
    GT3_PROXY		  = GT3_TYPE|IMPERSONATION,
    RFC_PROXY		  = RFC_TYPE|IMPERSONATION,
    GT2_LIMITED_PROXY	  = GT2_TYPE|LIMITED,
    GT3_LIMITED_PROXY	  = GT3_TYPE|LIMITED,
    RFC_LIMITED_PROXY	  = RFC_TYPE|LIMITED,
    GT3_INDEPENDENT_PROXY = GT3_TYPE|INDEPENDENT,
    RFC_INDEPENDENT_PROXY = RFC_TYPE|INDEPENDENT,
    GT3_ANYLANG_PROXY     = GT3_TYPE|ANYLANG,
    RFC_ANYLANG_PROXY     = RFC_TYPE|ANYLANG,
    GT3_RESTRICTED_PROXY  = GT3_TYPE|RESTRICTED,
    RFC_RESTRICTED_PROXY  = RFC_TYPE|RESTRICTED
} proxy_type_t;



/**
 * Structure type used to pass around the data derived from the input data (e.g.
 * a private key read from the input pem
 */
typedef struct intern_discovered_verify_x509_data_s
{
    /* Derived data */
    char * certificate_pem_str;
    char * private_key_pem;
    char * ocsp_responder_uri;

    STACK_OF (X509) * stack_of_x509;
    EVP_PKEY        * evp_pkey;
} internal_discovered_verify_x509_data_t;


/**
 * Structure type used to pass around input data
 */
typedef struct internal_verify_x509_data_s
{
    /* Assigned data */
    char * capath;
    char * certificate_filepath;
    FILE * certificate_f_handle;
    char * certificate_pem_str;
    char * private_key_filepath;
    char * private_key_pem;
    char * crl_path;
    char * ocsp_responder_uri;
    unsigned int no_crl_check;
    unsigned int allow_limited_proxy;
    unsigned int require_limited_proxy;
    unsigned int must_have_priv_key;
    unsigned int verify_at_notbefore;

    STACK_OF (X509) * stack_of_x509;
    EVP_PKEY        * evp_pkey;

    unsigned short is_initialized;
    unsigned short is_derived;

    internal_discovered_verify_x509_data_t derived_data;
} internal_verify_x509_data_t;



/**
 * @ingroup proxypolicy
 *
 * @note NOTE: The API provides functions to manipulate
 * the fields of a PROXYPOLICY.  Accessing the fields
 * directly will not work.
 *
 * This typedef maintains information about the policies
 * that have been placed on a proxy certificate
 *
 * policy_language defines which policy language is to be used to define the
 * policies
 * policy the policy that determines the policies on a certificate
 */
typedef struct PROXYPOLICY_st
{
    ASN1_OBJECT *       policy_language;
    ASN1_OCTET_STRING * policy;
} PROXYPOLICY;
/* Following declares
 * PROXYPOLICY_new(), PROXYPOLICY_free(),
 * d2i_PROXYPOLICY(), i2d_PROXYPOLICY() and
 * PROXYPOLICY_it */
DECLARE_ASN1_FUNCTIONS(PROXYPOLICY)

/*
 * This typedef maintains information about a proxy
 * certificate.
 *
 * @note NOTE: The API provides functions to manipulate
 * the fields of a PROXYCERTINFO.  Accessing the fields
 * directly is not a good idea.
 *
 *
 * path_length an optional field in the ANS.1 DER encoding,
 * it specifies the maximum depth of the path of Proxy Certificates
 * that can be signed by this End Entity Certificate or Proxy Certificate.
 * policy a non-optional field in the ANS.1 DER encoding,
 * specifies policies on the use of this certificate.
 */
typedef struct PROXYCERTINFO_st
{
    ASN1_INTEGER * path_length;       /* [ OPTIONAL ] */
    PROXYPOLICY *  policy;
    int                                 version;
} PROXYCERTINFO;
/* Following declares
 * PROXYPOLICY_new(), PROXYPOLICY_free(),
 * d2i_PROXYPOLICY(), i2d_PROXYPOLICY() and
 * PROXYCERTINFO_it */
DECLARE_ASN1_FUNCTIONS(PROXYCERTINFO)

/*
 * This typedef maintains information about a draft-RFC / GT3  proxy
 * certificate. The struct is identical to that for RFC, but we need to define
 * separate ASN1_SEQUENCEs for both, see _verify_proxy_certinfo.c
 */
typedef struct PROXYCERTINFO_GT3_st
{
    ASN1_INTEGER * path_length;       /* [ OPTIONAL ] */
    PROXYPOLICY *  policy;
    int                                 version;
} PROXYCERTINFO_GT3;
/* Following declares
 * PROXYPOLICY_GT3_new(), PROXYPOLICY_GT3_free(),
 * d2i_PROXYPOLICY(), i2d_PROXYPOLICY() and
 * PROXYCERTINFO_it */
DECLARE_ASN1_FUNCTIONS(PROXYCERTINFO_GT3)

#endif /* PUBLIC_VERIFY_X509_DATATYPES */
