/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


#ifndef LCAS_PEM_C
#define LCAS_PEM_C

/* Needed for strdup and setenv */
#define _XOPEN_SOURCE 600

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcas_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/* LCAS includes */
#include "lcas_types.h"
#include "_lcas_log.h"
#include "_lcas.h"
#include "lcas_pem.h"

/******************************************************************************
                             Module specific defines
******************************************************************************/
/* Default settings */
#define LCAS_LOG_FILE "/var/log/lcas-suexec.log"
#define LCAS_LOG_STRING "pem"
#define LCAS_DEFAULT_DB_FILE "lcas.db"
#define LCAS_REQUIRED_DB_PATH LCAS_ETC_HOME

int lcas_pem(char * pem_string, lcas_request_t request)
{
    char *          lcas_log_file = NULL;
    time_t          myclock;
    struct tm  *    timestamp = NULL;
    size_t          string_size; 
    char *          lcas_db_file = NULL; 
    char *          lcas_db_file_env = NULL; 
    char *          lcas_log_string = NULL; 
    char *          lcas_log_string_default = NULL; 
    int             rc = 0;
    int             res = 0;
    /*
     * Take log file from environment variables.
     * If not defined take the default ones
     */
    lcas_log_file = getenv("LCAS_LOG_FILE");
    lcas_log_file = (lcas_log_file ? lcas_log_file : LCAS_LOG_FILE );

    /*
     * The other environment variables should in principle not be touched,
     * because reasonably defaults are provided by the LCAS core.
     * An exception is made for LCAS_LOG_STRING, because we want to enforce
     * that it is logged which interface is used.
     * For traceability also add a timestamp.
     * setenv with overwrite=0 only overwrites if the variable is non-existant
     */
    time(&myclock);
    /* timestamp = localtime(&clock); */
    timestamp = gmtime(&myclock);
    string_size = strlen(LCAS_LOG_STRING) + strlen(": ") + 19 + 1;
    lcas_log_string_default = (char *) malloc(string_size * sizeof(char));

    res = snprintf(lcas_log_string_default, string_size, "%s: %04d-%02d-%02d.%02d:%02d:%02d",
                   LCAS_LOG_STRING,
                   timestamp->tm_year + 1900, timestamp->tm_mon + 1, timestamp->tm_mday,
                   timestamp->tm_hour, timestamp->tm_min, timestamp->tm_sec);

    if ( (res > string_size) || (res < 0) )
    {
        lcas_log(0,"%s","lcas_pem(): date conversion failed\n");
    }

    setenv("LCAS_LOG_STRING", lcas_log_string_default, 0);
    lcas_log_string = getenv("LCAS_LOG_STRING");

    if (lcas_log_string_default != NULL)
    {
        free(lcas_log_string_default);
        lcas_log_string_default = NULL;
    }

    /*
     * Set the lcas_db_file. If we're very paranoid we set DONT_ALLOW_ABSOLUTE_LCAS_DB_FILE, but normally
     * this is not needed, because the calling application (glexec) should take care that the
     * environment variables are set correctly
     */
#if DONT_ALLOW_ABSOLUTE_LCAS_DB_FILE
    /*
     * "Another exception is made for the lcas.db file: This interface is used by
     *  VO services (i.e. not site-controlled services), so the VO should not be able
     *  to determine which lcas.db file (or in the future a general lcas config file)
     *  they take."
     */
    if ( (lcas_db_file_env = getenv("LCAS_DB_FILE")) != NULL)
    {
        /*
         * OK, one tries to set the location of the lcas.db file.
         * We don't allow paths to be included in the filename. LCAS will add the
         * site-controlled path to it.
         * If a path is found: error
         * If no path is found: add the default path to it and performs various checks
         */
        if (strchr(lcas_db_file_env, '/') != NULL)
        {
            fprintf(stderr, "%s: (LCAS_DB_FILE = %s)\n", lcas_log_string, lcas_db_file_env);
            fprintf(stderr,
                "%s: Attempt to include path to lcas.db file by environment variable\n", lcas_log_string);
            fprintf(stderr, "%s: This is not allowed for this lcas interface\n", lcas_log_string);
            return 1;
        }
    }
    else
    {
        lcas_db_file_env = LCAS_DEFAULT_DB_FILE;
    }
    /* prepend required path */
    lcas_db_file = (char *) malloc(strlen(LCAS_REQUIRED_DB_PATH) + 2 + strlen(lcas_db_file_env));
    sprintf(lcas_db_file, "%s/%s", LCAS_REQUIRED_DB_PATH, lcas_db_file_env);
#else /* DONT_ALLOW_ABSOLUTE_LCAS_DB_FILE */
    lcas_db_file_env = getenv("LCAS_DB_FILE");
    lcas_db_file_env = (lcas_db_file_env ? lcas_db_file_env : LCAS_DEFAULT_DB_FILE );
    lcas_db_file = strdup(lcas_db_file_env);
#endif /* DONT_ALLOW_ABSOLUTE_LCAS_DB_FILE */

    /*
     * Check the lcas_db_file.
     * Again, if we're very paranoid we set CHECK_LCAS_DB_PERMISSIONS, but normally
     * this is not needed, because the calling application (glexec) should do that.
     */
#if CHECK_LCAS_DB_PERMISSIONS
    /*
     * Check lcas.db file:
     * - Should be in a directory owned by root, which non-writable for others.
     * - Should be owned by root and non-writable for others.
     * If checks are ok reset environment variable
     */
    /*
     * Stat the directory of lcas.db and verify it is a directory, or error out.
     */
    if (((lstat(LCAS_REQUIRED_DB_PATH, &dir_info)) != 0) || !(S_ISDIR(dir_info.st_mode)))
    {
        fprintf(stderr, "%s: cannot stat directory: (%s)\n", lcas_log_string, LCAS_REQUIRED_DB_PATH);
        if (lcas_db_file) {free(lcas_db_file); lcas_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the directory is not owned by root
     */
    if ((dir_info.st_uid != 0) || (dir_info.st_gid != 0))
    {
        fprintf(stderr, "%s: directory is not owned by root: (%s)\n", lcas_log_string, LCAS_REQUIRED_DB_PATH);
        if (lcas_db_file) {free(lcas_db_file); lcas_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the directory is writable by others.
     */
    if (dir_info.st_mode & S_IWOTH)
    {
        fprintf(stderr, "%s: directory is writable by others: (%s)\n", lcas_log_string, LCAS_REQUIRED_DB_PATH);
        if (lcas_db_file) {free(lcas_db_file); lcas_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if we cannot stat the lcas.db file.
     */
    if (((lstat(lcas_db_file, &prg_info)) != 0) || (S_ISLNK(prg_info.st_mode)))
    {
        fprintf(stderr, "%s: cannot stat program: (%s)\n", lcas_log_string, lcas_db_file);
        if (lcas_db_file) {free(lcas_db_file); lcas_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the lcas.db file is not owned by root
     */
    if ((prg_info.st_uid != 0) || (prg_info.st_gid != 0))
    {
        fprintf(stderr, "%s: file is not owned by root: (%s)\n", lcas_log_string, lcas_db_file);
        if (lcas_db_file) {free(lcas_db_file); lcas_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the lcas.db file is writable by others.
     */
    if (prg_info.st_mode & S_IWOTH)
    {
        fprintf(stderr, "%s: file is writable by others: (%s)\n", lcas_log_string, lcas_db_file);
        if (lcas_db_file) {free(lcas_db_file); lcas_db_file = NULL;}
        return 1;
    }
#endif /* CHECK_LCAS_DB_PERMISSIONS*/

    /* Everything's fine, so set (overwrite!) the environment variable. */
    setenv("LCAS_DB_FILE", lcas_db_file, 1);
    if (lcas_db_file)
    {
        free(lcas_db_file);
        lcas_db_file = NULL;
    }

    /* First initialize LCAS */
    if (lcas_init_and_logfile(lcas_log_file, NULL, DO_USRLOG|DO_SYSLOG))
//    if (lcas_init_and_logfile(lcas_log_file, NULL, DO_USRLOG))
    {
        fprintf(stderr, "%s: LCAS initialization failure\n", lcas_log_string);
        goto lcas_pem_error;
    }


    /*
     * Now that we have the credential let's run (and terminate) LCAS !
     */
    rc = lcas_run_va(LCAS_ARG_PEM, pem_string, request);
    if (rc != 0)
    {
        lcas_log(0, "LCAS failed to do mapping and return account information\n");
        if (lcas_term())
        {
            fprintf(stderr, "LCAS termination failure\n");
            goto lcas_pem_error;
        }
        goto lcas_pem_error;
    }

    rc = lcas_term();
    if (rc)
    {
        fprintf(stderr, "LCAS termination failure\n");
        goto lcas_pem_error;
    }

    return 0;

 lcas_pem_error:
    return 1;
}

#endif /* LCAS_PEM_C */
