/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "ACL IMSpector filter plugin"
#define PLUGIN_SHORT_NAME "ACL"

struct aclelement
{
	bool filtered;
	std::string localid;
	std::vector<std::string> remoteids;
};

extern "C"
{
	bool initfilterplugin(struct filterplugininfo &filterplugininfo,
		class Options &options, bool debugmode);
	void closefilterplugin(void);
	bool filter(char *originalbuffer, char *modifiedbuffer, struct imevent &imevent);
};

std::vector<struct aclelement> acl;
bool localdebugmode = false;

bool matchacl(std::string localid, std::string remoteid, std::vector<struct aclelement> &acl);
bool matchid(std::string &one, std::string &two);
bool parseacl(std::vector<struct aclelement> &acl, std::string aclfilename);
void debugacl(std::vector<struct aclelement> &acl);

bool initfilterplugin(struct filterplugininfo &filterplugininfo,
	class Options &options, bool debugmode)
{
	std::string aclfilename = options["acl_filename"];

	if (aclfilename.empty()) return false;

	localdebugmode = debugmode;

	filterplugininfo.pluginname = PLUGIN_NAME;

	if (!(parseacl(acl, aclfilename))) return false;

	debugprint(localdebugmode, PLUGIN_SHORT_NAME ": List %s size: %d",
		aclfilename.c_str(), acl.size());

	debugacl(acl);

	return true;
}

void closefilterplugin(void)
{
	return;
}

/* The main plugin function. See eventplugin.cpp. */
bool filter(char *originalbuffer, char *modifiedbuffer, struct imevent &imevent)
{
	bool filtered = matchacl(imevent.localid, imevent.remoteid, acl);
	
	if (filtered) debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Filtered");
	else debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Passed");

	return filtered;
}

/* The real guts of this thing.  Returns the filtered response for a match. */
bool matchacl(std::string localid, std::string remoteid, std::vector<struct aclelement> &acl)
{
	debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Local: %s Remote: %s",
		localid.c_str(), remoteid.c_str());
	
	for (std::vector<struct aclelement>::iterator i = acl.begin();
		i != acl.end(); i++)
	{
		if (matchid(localid, (*i).localid) || (*i).localid == "all")
		{
			debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Got Local match (%s)", (*i).localid.c_str());
			
			if (!(*i).remoteids.size())
			{
				debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Remote acl empty; matching");
				return (*i).filtered;
			}		
			for (std::vector<std::string>::iterator j = (*i).remoteids.begin();
				j != (*i).remoteids.end(); j++)
			{
				if ((*j) == "groupchat")
				{
					if (remoteid.find("groupchat-", 0) == 0)
					{
						debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Got groupchat match (%s)", (*j).c_str());
						return (*i).filtered;
					}
				}
				if (matchid(remoteid, (*j)))
				{
					debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Got Remote match (%s)", (*j).c_str());
					return (*i).filtered;
				}
			}
			/* Local match, but remote match has ran off the end. */
			continue;
		}
	}
	
	debugprint(localdebugmode, PLUGIN_SHORT_NAME ": No match");
	
	return false;
}

/* Matches two strings.  Second string can be a subdomain, or full domain.
 * ie user@company.com is matched by compnay.com or com. */
bool matchid(std::string &one, std::string &two)
{
	std::string::size_type onelen = one.length();
	std::string::size_type twolen = two.length();
	int leftof = (int) onelen - (int) twolen - 1;
	
	if (leftof < 0) leftof = 0;
	
	char left = one[leftof];
	
	if (one.find(two, onelen - twolen) != std::string::npos &&
		((left == '@' || left == '.') || leftof == 0))
	{
		return true;
	}
	
	return false;
}

/* Parse in an acl. */
bool parseacl(std::vector<struct aclelement> &acl, std::string aclfilename)
{
	FILE *hfile = NULL;
	char buffer[STRING_SIZE];
	int c = 0;
	
	memset(buffer, 0, STRING_SIZE);
	
	if (!(hfile = fopen(aclfilename.c_str(), "r")))
		return false;
		
	while (fgets(buffer, STRING_SIZE, hfile))
	{
		stripnewline(buffer);

		if (!strlen(buffer)) continue;
		if (buffer[0] == '#') continue;
		
		std::string command;
		std::vector<std::string> args;
		int argc;
		
		chopline(buffer, command, args, argc);
		
		struct aclelement aclelement;
		
		if (command == "allow")
			aclelement.filtered = false;
		else if (command == "deny")
			aclelement.filtered = true;
		else
			continue;
		
		if (!argc) continue;

		aclelement.localid = args.front();
		args.erase(args.begin());
		
		aclelement.remoteids = args;
		
		acl.push_back(aclelement);
		
		c++;
	}
	
	fclose(hfile);
	
	return c;
}

/* Output a acl to the debug log. */
void debugacl(std::vector<struct aclelement> &acl)
{
	for (std::vector<struct aclelement>::iterator i = acl.begin();
		i != acl.end(); i++)
	{
		debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Action: %s", (*i).filtered ? "Deny" : "Allow");
		debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Local: %s", (*i).localid.c_str());
		for (std::vector<std::string>::iterator j = (*i).remoteids.begin();
			j != (*i).remoteids.end(); j++)
		{
			debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Remote: %s", (*j).c_str());
		}
	}
}

