// Copyright 2020 New Relic Corporation. All rights reserved.
// SPDX-License-Identifier: Apache-2.0

package internal

import (
	"bytes"
	"encoding/json"
	"strings"
	"testing"

	"github.com/newrelic/go-agent/internal/stacktracetest"
)

func TestGetStackTrace(t *testing.T) {
	stack := GetStackTrace()
	js, err := json.Marshal(stack)
	if nil != err {
		t.Fatal(err)
	}
	if nil == js {
		t.Fatal(string(js))
	}
}

func TestLongStackTraceLimitsFrames(t *testing.T) {
	st := stacktracetest.CountedCall(maxStackTraceFrames+20, func() []uintptr {
		return GetStackTrace()
	})
	if len(st) != maxStackTraceFrames {
		t.Error("Unexpected size of stacktrace", maxStackTraceFrames, len(st))
	}
	l := len(StackTrace(st).frames())
	if l != maxStackTraceFrames {
		t.Error("Unexpected number of frames", maxStackTraceFrames, l)
	}
}

func TestManyStackTraceFramesLimitsOutput(t *testing.T) {
	frames := make([]stacktraceFrame, maxStackTraceFrames+20)
	expect := `[
	{},{},{},{},{},{},{},{},{},{},
	{},{},{},{},{},{},{},{},{},{},
	{},{},{},{},{},{},{},{},{},{},	
	{},{},{},{},{},{},{},{},{},{},	
	{},{},{},{},{},{},{},{},{},{},	
	{},{},{},{},{},{},{},{},{},{},	
	{},{},{},{},{},{},{},{},{},{},	
	{},{},{},{},{},{},{},{},{},{},	
	{},{},{},{},{},{},{},{},{},{},	
	{},{},{},{},{},{},{},{},{},{}	
	]`
	estimate := 256 * len(frames)
	output := bytes.NewBuffer(make([]byte, 0, estimate))
	writeFrames(output, frames)
	if CompactJSONString(expect) != output.String() {
		t.Error("Unexpected JSON output", CompactJSONString(expect), output.String())
	}
}

func TestStacktraceFrames(t *testing.T) {
	// This stacktrace taken from Go 1.11
	inputFrames := []stacktraceFrame{
		{
			File: "/Users/will/Desktop/gopath/src/github.com/newrelic/go-agent/internal/stacktrace.go",
			Name: "github.com/newrelic/go-agent/internal.GetStackTrace",
			Line: 17,
		},
		{
			File: "/Users/will/Desktop/gopath/src/github.com/newrelic/go-agent/internal_txn.go",
			Name: "github.com/newrelic/go-agent.(*txn).NoticeError",
			Line: 696,
		},
		{
			File: "\u003cautogenerated\u003e",
			Name: "go.(*struct { github.com/newrelic/go-agent.threadWithExtras }).NoticeError",
			Line: 1,
		},
		{
			File: "/Users/will/Desktop/gopath/src/github.com/newrelic/go-agent/internal_attributes_test.go",
			Name: "github.com/newrelic/go-agent.TestAddAttributeSecurityPolicyDisablesInclude",
			Line: 68,
		},
		{
			File: "/Users/will/.gvm/gos/go1.11/src/testing/testing.go",
			Name: "testing.tRunner",
			Line: 827,
		},
		{
			File: "/Users/will/.gvm/gos/go1.11/src/runtime/asm_amd64.s",
			Name: "runtime.goexit",
			Line: 1333,
		},
	}
	buf := &bytes.Buffer{}
	writeFrames(buf, inputFrames)
	expectedJSON := `[
		{
			"name":"testing.tRunner",
			"filepath":"/Users/will/.gvm/gos/go1.11/src/testing/testing.go",
			"line":827
		},
		{
			"name":"runtime.goexit",
			"filepath":"/Users/will/.gvm/gos/go1.11/src/runtime/asm_amd64.s",
			"line":1333
		}
	]`
	testExpectedJSON(t, expectedJSON, buf.String())
}

func TestStackTraceTopFrame(t *testing.T) {
	// This test uses a separate package since the stacktrace code removes
	// the top stack frames which are in packages "newrelic" and "internal".
	stackJSON := stacktracetest.TopStackFrame(func() []byte {
		st := GetStackTrace()
		js, _ := json.Marshal(st)
		return js
	})

	stack := []struct {
		Name     string `json:"name"`
		FilePath string `json:"filepath"`
		Line     int    `json:"line"`
	}{}
	if err := json.Unmarshal(stackJSON, &stack); err != nil {
		t.Fatal(err)
	}
	if len(stack) < 2 {
		t.Fatal(string(stackJSON))
	}
	if stack[0].Name != "stacktracetest.TopStackFrame" {
		t.Error(string(stackJSON))
	}
	if stack[0].Line != 9 {
		t.Error(string(stackJSON))
	}
	if !strings.Contains(stack[0].FilePath, "go-agent/internal/stacktracetest/stacktracetest.go") {
		t.Error(string(stackJSON))
	}
}

func TestFramesCount(t *testing.T) {
	st := stacktracetest.CountedCall(3, func() []uintptr {
		return GetStackTrace()
	})
	frames := StackTrace(st).frames()
	if len(st) != len(frames) {
		t.Error("Invalid # of frames", len(st), len(frames))
	}
}
