/* $Id$
 *
 * Resolve ranges of positional and named aggregates, where the type
 * is an unconstraint array.
 *
 * Copyright (C) 2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __UNCONSTRAINT_BOUNDS_HPP_INCLUDED
#define __UNCONSTRAINT_BOUNDS_HPP_INCLUDED

#include "frontend/visitor/NullVisitor.hpp"
#include "frontend/ast/Types.hpp"

namespace ast {

//! Resolve the bounds of an unconstraint array aggregate.
/** This visitor can be used to determine the bounds of an unconstraint
  * Array Aggregate. 
  *
  * It is useful wherever a an unconstraint variable/parameter/port
  * will get its actual constraints from the initializer.
  *
  * In contrast to ResolveAggregates, this visitor will not set element
  * position number, but only determine the bounds and direction.
  *
  * This visitor should be used only on small portions of the syntax
  * tree, i.e. only Aggregates should accept it.
  * 
  * Important: This visitor assumes that the type of the Aggregate has been
  *            determined already, but that it is an unconstraint type.
  *            It is definitely wrong to use this visitor for a constraint
  *            type.
  */
class UnconstraintBounds : public NullVisitor {
public:
	//! dummy c'tor
	UnconstraintBounds();

	//! dummy d'tor
	virtual ~UnconstraintBounds() {}

	//! picked up bounds (if any) 
	DiscreteRange *bounds;

private:
	/** Visit an Aggregate node.
	 *  @param node Aggregate node that get's visited.
	 */
	virtual void visit(Aggregate &node);

	/** visit a ElementAssociation
         *  @param node node that get's visited.
         */
	virtual void visit(ElementAssociation &node);

	/** Visit an Others node.
	 *  @param node Others node that get's visited.
	 */
	virtual void visit(Others &node);

	/** Visit a ConstInteger node.
	 *  @param node ConstInteger node that get's visited.
	 */
	virtual void visit(ConstInteger &node);

	/** Visit a DiscreteRange
	 *  @param node DiscreteRange node that get's visited.
	 */
	virtual void visit(DiscreteRange &node);

	//! Process a generic AstNode.
        /** Failmatch method. Must not be called.
         *
         *  @param node AstNode
         */
	virtual void process(AstNode &node);

	/** determine the index range of the unconstraint array at by 
	  * the index type.
	  * @param at unconstraint array type.
	  * @return index range of the index subtype of at
	  */
	static DiscreteRange *
	findIndexRange(const UnconstrainedArrayType *at);

	/** low bound of determined index */
	universal_integer low;
	/** high bound of determined index */
	universal_integer high;
	/** number of elements for positional aggregates */
	size_t numElements;
	/** is it a positional aggregate? */
	bool positional;
};

}; /* namespace ast */

#endif /* __UNCONSTRAINT_BOUNDS_HPP_INCLUDED */
