/* $Id: esorex_json-test.c,v 1.6 2013/02/25 11:07:12 rpalsa Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: rpalsa $
 * $Date: 2013/02/25 11:07:12 $
 * $Revision: 1.6 $
 * $Name: esorex-3_10 $
 */

#include <string.h>

#include "er_json.h"

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#ifndef STR_LENGTH    
#define STR_LENGTH 80
#endif

/*-----------------------------------------------------------------------------
                        Private function prototypes
 -----------------------------------------------------------------------------*/

static void esorex_json_parameterlist(void);
static void esorex_json_frameset(void);

/*-----------------------------------------------------------------------------
                                  Main
 -----------------------------------------------------------------------------*/
int main(void)
{

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */
    esorex_json_parameterlist();
    esorex_json_frameset();

    /* End of tests */
    return cpl_test_end(0);
}

static void esorex_json_parameterlist(void)
{
    cpl_parameter     * par1;
    cpl_parameter     * par2;
    cpl_parameter     * par3;
    cpl_parameter     * par4;
    cpl_parameterlist * parlist;
    
    /* Export a parameterlist to a JSON file */

    par1 = cpl_parameter_new_value("recipe.par1", CPL_TYPE_DOUBLE,
                                   "Description 1","recipe", 10.0);
    
    cpl_test_nonnull( par1 );

    par2 = cpl_parameter_new_value("recipe.par2", CPL_TYPE_STRING,
                                   "Description 2","recipe", "String");
    
    cpl_test_nonnull( par2 );
    
    par3 = cpl_parameter_new_value("recipe.par3", CPL_TYPE_BOOL,
                                   "Description 3","recipe", CPL_TRUE);
    
    cpl_test_nonnull( par3 );

    par4 = cpl_parameter_new_value("recipe.par4", CPL_TYPE_INT,
                                   "Escaped chars \" / \f \n \r \t \\","recipe", -10);
    
    cpl_test_nonnull( par4 );

    parlist = cpl_parameterlist_new();
    
    cpl_test_nonnull( parlist );

    cpl_parameterlist_append(parlist, par1);
    
    /* Dumping to json */
    er_recipe_parameterlist_to_json(parlist, "recipe_name", "parlist.json");
    
    cpl_test_error(CPL_ERROR_NONE);

    /* Read the file and compare it with a static version */
    {
        char * whole_file = 0;
        long length;
        FILE * fjson = fopen("parlist.json", "rb");
        const char * expected_json="[\n"
                "  {\n"
                "    \"name\": \"recipe.par1\",\n"
                "    \"value\": \"10\",\n"
                "    \"display_name\": \"recipe.par1\",\n"
                "    \"description\": \"Description 1\",\n"
                "    \"recipe\": \"recipe_name\"\n"
                "  }\n"
                "]\n\0";
        
        cpl_test_nonnull(fjson);
        fseek(fjson, 0, SEEK_END);
        length = ftell(fjson);
        rewind(fjson);
        whole_file = cpl_malloc(length+1);
        
        cpl_test_nonnull(whole_file);
        fread(whole_file, 1, length, fjson);
        whole_file[length]='\0';
        fclose(fjson);
        
        cpl_test_zero( strcmp(whole_file, expected_json) );
        
        cpl_test_zero( remove("parlist.json") );
        cpl_free(whole_file);
    }

    /* Now with no recipe name */
    er_recipe_parameterlist_to_json(parlist, "", "parlist11.json");
    
    cpl_test_error(CPL_ERROR_NONE);

    /* Read the file and compare it with a static version */
    {
        char * whole_file = 0;
        long length;
        FILE * fjson = fopen("parlist11.json", "rb");
        const char * expected_json="[\n"
                "  {\n"
                "    \"name\": \"recipe.par1\",\n"
                "    \"value\": \"10\",\n"
                "    \"display_name\": \"recipe.par1\",\n"
                "    \"description\": \"Description 1\"\n"
                "  }\n"
                "]\n\0";
        
        cpl_test_nonnull(fjson);
        fseek(fjson, 0, SEEK_END);
        length = ftell(fjson);
        rewind(fjson);
        whole_file = cpl_malloc(length+1);
        
        cpl_test_nonnull(whole_file);
        fread(whole_file, 1, length, fjson);
        whole_file[length]='\0';
        fclose(fjson);
        
        cpl_test_zero( strcmp(whole_file, expected_json) );
        
        cpl_test_zero( remove("parlist11.json") );
        cpl_free(whole_file);
    }
    
    /* Do the same with a list that contains more than 1 parameter, including
     * a parameter with control characters */
    cpl_parameterlist_append(parlist, par2);
    cpl_parameterlist_append(parlist, par3);
    cpl_parameterlist_append(parlist, par4);
    
    cpl_test_error(CPL_ERROR_NONE);
            
    /* Dumping to json */
    er_recipe_parameterlist_to_json(parlist, "recipe_name", "parlist2.json");

    cpl_test_error(CPL_ERROR_NONE);

    /* Read the file and compare it with a static version */
    {
        char * whole_file = 0;
        long length;
        FILE * fjson = fopen("parlist2.json", "rb");
        const char * expected_json="[\n"
                "  {\n"
                "    \"name\": \"recipe.par1\",\n"
                "    \"value\": \"10\",\n"
                "    \"display_name\": \"recipe.par1\",\n"
                "    \"description\": \"Description 1\",\n"
                "    \"recipe\": \"recipe_name\"\n"
                "  },\n"
                "  {\n"
                "    \"name\": \"recipe.par2\",\n"
                "    \"value\": \"String\",\n"
                "    \"display_name\": \"recipe.par2\",\n"
                "    \"description\": \"Description 2\",\n"
                "    \"recipe\": \"recipe_name\"\n"
                "  },\n"
                "  {\n"
                "    \"name\": \"recipe.par3\",\n"
                "    \"value\": true,\n"
                "    \"display_name\": \"recipe.par3\",\n"
                "    \"description\": \"Description 3\",\n"
                "    \"recipe\": \"recipe_name\"\n"
                "  },\n"
                "  {\n"
                "    \"name\": \"recipe.par4\",\n"
                "    \"value\": \"-10\",\n"
                "    \"display_name\": \"recipe.par4\",\n"
                "    \"description\": \"Escaped chars \\\" \\/ \\f \\n \\r \\t \\\\\",\n"
                "    \"recipe\": \"recipe_name\"\n"
                "  }\n"
                "]\n\0";

        cpl_test_nonnull(fjson);
        fseek(fjson, 0, SEEK_END);
        length = ftell(fjson);
        rewind(fjson);
        whole_file = cpl_malloc(length+1);

        cpl_test_nonnull(whole_file);
        fread(whole_file, 1, length, fjson);
        whole_file[length]='\0';
        fclose(fjson);

        cpl_test_zero( strcmp(whole_file, expected_json) );

        cpl_test_zero( remove("parlist2.json") );
        cpl_free(whole_file);
    }

    /* Try to save in a non-existent path */
    {
        cpl_error_code errcode = 
                er_recipe_parameterlist_to_json(parlist, "recipe_name",
                        "/non-existent/path/parlist.json");
        cpl_test_error(CPL_ERROR_FILE_NOT_CREATED);
    }
    
    /* Trying with null inputs */
    {
        cpl_error_code errcode = 
                er_recipe_parameterlist_to_json(NULL, "recipe_name",
                        "parlist.json");
        cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
        
        errcode =  er_recipe_parameterlist_to_json(parlist, NULL,
                         "parlist.json");
        cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

        errcode =  er_recipe_parameterlist_to_json(parlist, "recipe_name",
                         NULL);
        cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    }
    
    cpl_parameterlist_delete(parlist);
}

static void esorex_json_frameset(void)
{
    cpl_frame    * fset1;
    cpl_frame    * fset2;
    cpl_frame    * fset3;
    cpl_frameset * frameset;
    
    /* Export a frameset to a JSON file */

    fset1 = cpl_frame_new();
    cpl_frame_set_filename(fset1, "filename1.fits");
    cpl_frame_set_tag(fset1, "FLAT");
    
    cpl_test_nonnull( fset1 );

    fset2 = cpl_frame_new();
    cpl_frame_set_filename(fset2, "filename2.fits");
    cpl_frame_set_tag(fset2, "BIAS");
    
    cpl_test_nonnull( fset2 );
    
    fset3 = cpl_frame_new();
    cpl_frame_set_filename(fset3, "filename3.fits");
    cpl_frame_set_tag(fset3, "SCIENCE");
    
    cpl_test_nonnull( fset3 );

    frameset = cpl_frameset_new();
    
    cpl_test_nonnull( frameset );

    cpl_frameset_insert(frameset, fset1);
    
    /* Dumping to json */
    er_frameset_to_json(frameset, "frameset.json");
    
    cpl_test_error(CPL_ERROR_NONE);

    /* Read the file and compare it with a static version */
    {
        char * whole_file = 0;
        long length;
        FILE * fjson = fopen("frameset.json", "rb");
        const char * expected_json="[\n"
                "  {\n"
                "    \"name\": \"filename1.fits\",\n"
                "    \"category\": \"FLAT\"\n"
                "  }\n"
                "]\n\0";
        
        cpl_test_nonnull(fjson);
        fseek(fjson, 0, SEEK_END);
        length = ftell(fjson);
        rewind(fjson);
        whole_file = cpl_malloc(length+1);
        
        cpl_test_nonnull(whole_file);
        fread(whole_file, 1, length, fjson);
        whole_file[length]='\0';
        fclose(fjson);
        
        cpl_test_zero( strcmp(whole_file, expected_json) );
        
        cpl_test_zero( remove("frameset.json") );
        cpl_free(whole_file);
    }

    /* Do the same with a list that contains more than 1 parameter */
    cpl_frameset_insert(frameset, fset2);
    cpl_frameset_insert(frameset, fset3);
    
    cpl_test_error(CPL_ERROR_NONE);
            
    /* Dumping to json */
    er_frameset_to_json(frameset, "frameset2.json");

    cpl_test_error(CPL_ERROR_NONE);

    /* Read the file and compare it with a static version */
    {
        char * whole_file = 0;
        long length;
        FILE * fjson = fopen("frameset2.json", "rb");
        const char * expected_json="[\n"
                "  {\n"
                "    \"name\": \"filename1.fits\",\n"
                "    \"category\": \"FLAT\"\n"
                "  },\n"
                "  {\n"
                "    \"name\": \"filename2.fits\",\n"
                "    \"category\": \"BIAS\"\n"
                "  },\n"
                "  {\n"
                "    \"name\": \"filename3.fits\",\n"
                "    \"category\": \"SCIENCE\"\n"
                "  }\n"
                "]\n\0";

        cpl_test_nonnull(fjson);
        fseek(fjson, 0, SEEK_END);
        length = ftell(fjson);
        rewind(fjson);
        whole_file = cpl_malloc(length+1);

        cpl_test_nonnull(whole_file);
        fread(whole_file, 1, length, fjson);
        whole_file[length]='\0';
        fclose(fjson);

        cpl_test_zero( strcmp(whole_file, expected_json) );

        cpl_test_zero( remove("frameset2.json") );
        cpl_free(whole_file);
    }

    /* Try to save in a non-existent path */
    {
        cpl_error_code errcode =
                er_frameset_to_json(frameset, 
                        "/non-existent/path/frameset.json");
        cpl_test_error(CPL_ERROR_FILE_NOT_CREATED);
    }
    
    /* Trying with null inputs */
    {
        cpl_error_code errcode = 
                er_frameset_to_json(NULL, "frameset.json");
        cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
        
        errcode = er_frameset_to_json(frameset, NULL);
        cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    }
    
    cpl_frameset_delete(frameset);
}
