// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

namespace Microsoft.EntityFrameworkCore.Query.Internal;

public partial class NavigationExpandingExpressionVisitor
{
    private sealed class EntityReference : Expression, IPrintableExpression
    {
        public EntityReference(IEntityType entityType, EntityQueryRootExpression? entityQueryRootExpression)
        {
            EntityType = entityType;
            IncludePaths = new IncludeTreeNode(entityType, this, setLoaded: true);
            EntityQueryRootExpression = entityQueryRootExpression;
        }

        public IEntityType EntityType { get; }

        public Dictionary<(IForeignKey, bool), Expression> ForeignKeyExpansionMap { get; } = new();

        public bool IsOptional { get; private set; }
        public IncludeTreeNode IncludePaths { get; private set; }
        public IncludeTreeNode? LastIncludeTreeNode { get; private set; }
        public EntityQueryRootExpression? EntityQueryRootExpression { get; }

        public override ExpressionType NodeType
            => ExpressionType.Extension;

        public override Type Type
            => EntityType.ClrType;

        protected override Expression VisitChildren(ExpressionVisitor visitor)
            => this;

        public EntityReference Snapshot()
        {
            var result = new EntityReference(EntityType, EntityQueryRootExpression) { IsOptional = IsOptional };
            result.IncludePaths = IncludePaths.Snapshot(result);

            return result;
        }

        public void SetLastInclude(IncludeTreeNode lastIncludeTree)
            => LastIncludeTreeNode = lastIncludeTree;

        public void MarkAsOptional()
            => IsOptional = true;

        void IPrintableExpression.Print(ExpressionPrinter expressionPrinter)
        {
            expressionPrinter.Append($"{nameof(EntityReference)}: {EntityType.DisplayName()}");
            if (IsOptional)
            {
                expressionPrinter.Append("[Optional]");
            }

            if (IncludePaths.Count > 0)
            {
                expressionPrinter.AppendLine(" | IncludePaths: ");
                using (expressionPrinter.Indent())
                {
                    expressionPrinter.AppendLine("Root");
                }

                PrintInclude(IncludePaths);
            }

            void PrintInclude(IncludeTreeNode currentNode)
            {
                if (currentNode.Count > 0)
                {
                    using (expressionPrinter.Indent())
                    {
                        foreach (var (navigationBase, includeTreeNode) in currentNode)
                        {
                            expressionPrinter.AppendLine(@"\-> " + navigationBase.Name);
                            PrintInclude(includeTreeNode);
                        }
                    }
                }
            }
        }
    }

    /// <summary>
    ///     A tree structure of includes for a given entity type in <see cref="EntityReference" />.
    /// </summary>
    private sealed class IncludeTreeNode(IEntityType entityType, EntityReference? reference, bool setLoaded)
        : Dictionary<INavigationBase, IncludeTreeNode>
    {
        private EntityReference? _reference = reference;

        public IEntityType EntityType { get; } = entityType;
        public LambdaExpression? FilterExpression { get; private set; }
        public bool SetLoaded { get; private set; } = setLoaded;

        public IncludeTreeNode AddNavigation(INavigationBase navigation, bool setLoaded)
        {
            if (TryGetValue(navigation, out var existingValue))
            {
                if (setLoaded && !existingValue.SetLoaded)
                {
                    existingValue.SetLoaded = true;
                }

                return existingValue;
            }

            IncludeTreeNode? nodeToAdd = null;
            if (_reference != null)
            {
                nodeToAdd = navigation switch
                {
                    INavigation concreteNavigation when _reference.ForeignKeyExpansionMap.TryGetValue(
                        (concreteNavigation.ForeignKey, concreteNavigation.IsOnDependent), out var expansion) => UnwrapEntityReference(
                        expansion)!.IncludePaths,
                    ISkipNavigation skipNavigation when _reference.ForeignKeyExpansionMap.TryGetValue(
                            (skipNavigation.ForeignKey, skipNavigation.IsOnDependent), out var firstExpansion)
                        // Value known to be non-null
                        && UnwrapEntityReference(firstExpansion)!.ForeignKeyExpansionMap.TryGetValue(
                            (skipNavigation.Inverse.ForeignKey, !skipNavigation.Inverse.IsOnDependent),
                            out var secondExpansion) => UnwrapEntityReference(secondExpansion)!.IncludePaths,
                    _ => nodeToAdd
                };
            }

            nodeToAdd ??= new IncludeTreeNode(navigation.TargetEntityType, null, setLoaded);

            this[navigation] = nodeToAdd;

            return this[navigation];
        }

        public IncludeTreeNode Snapshot(EntityReference? entityReference)
        {
            var result = new IncludeTreeNode(EntityType, entityReference, SetLoaded) { FilterExpression = FilterExpression };

            foreach (var (navigationBase, includeTreeNode) in this)
            {
                result[navigationBase] = includeTreeNode.Snapshot(null);
            }

            return result;
        }

        public void Merge(IncludeTreeNode includeTreeNode)
        {
            // EntityReference is intentionally ignored
            FilterExpression = includeTreeNode.FilterExpression;
            foreach (var (navigationBase, value) in includeTreeNode)
            {
                AddNavigation(navigationBase, value.SetLoaded).Merge(value);
            }
        }

        public void AssignEntityReference(EntityReference entityReference)
            => _reference = entityReference;

        public void ApplyFilter(LambdaExpression filterExpression)
            => FilterExpression = filterExpression;

        public override bool Equals(object? obj)
            => obj != null
                && (ReferenceEquals(this, obj)
                    || obj is IncludeTreeNode includeTreeNode
                    && Equals(includeTreeNode));

        private bool Equals(IncludeTreeNode includeTreeNode)
        {
            if (Count != includeTreeNode.Count)
            {
                return false;
            }

            foreach (var (navigationBase, value) in this)
            {
                if (!includeTreeNode.TryGetValue(navigationBase, out var otherIncludeTreeNode)
                    || !value.Equals(otherIncludeTreeNode))
                {
                    return false;
                }
            }

            return true;
        }

        public override int GetHashCode()
            => HashCode.Combine(base.GetHashCode(), EntityType);
    }

    /// <summary>
    ///     Stores information about the current queryable, its source, structure of projection, parameter type etc.
    ///     This is needed because once navigations are expanded we still remember these to avoid expanding again.
    /// </summary>
    private sealed class NavigationExpansionExpression : Expression, IPrintableExpression
    {
        private readonly List<(MethodInfo OrderingMethod, Expression KeySelector)> _pendingOrderings = [];

        private readonly string _parameterName;

        public NavigationExpansionExpression(
            Expression source,
            NavigationTreeNode currentTree,
            Expression pendingSelector,
            string parameterName)
        {
            Source = source;
            _parameterName = parameterName;
            CurrentTree = currentTree;
            PendingSelector = pendingSelector;
        }

        public Expression Source { get; private set; }

        public ParameterExpression CurrentParameter
            // CurrentParameter would be non-null if CurrentTree is non-null
            => CurrentTree.CurrentParameter!;

        [field: AllowNull, MaybeNull]
        public NavigationTreeNode CurrentTree
        {
            // _currentTree is always non-null. Field is to override the setter to set parameter
            get;
            private set
            {
                field = value;
                field.SetParameter(_parameterName);
            }
        }

        public Expression PendingSelector { get; private set; }
        public MethodInfo? CardinalityReducingGenericMethodInfo { get; private set; }
        public List<Expression> CardinalityReducingMethodArguments { get; } = [];

        public Type SourceElementType
            => CurrentParameter.Type;

        public IReadOnlyList<(MethodInfo OrderingMethod, Expression KeySelector)> PendingOrderings
            => _pendingOrderings;

        public void UpdateSource(Expression source)
            => Source = source;

        public void UpdateCurrentTree(NavigationTreeNode currentTree)
            => CurrentTree = currentTree;

        public void ApplySelector(Expression selector)
            => PendingSelector = selector;

        public void AddPendingOrdering(MethodInfo orderingMethod, Expression keySelector)
        {
            _pendingOrderings.Clear();
            _pendingOrderings.Add((orderingMethod, keySelector));
        }

        public void AppendPendingOrdering(MethodInfo orderingMethod, Expression keySelector)
            => _pendingOrderings.Add((orderingMethod, keySelector));

        public void ClearPendingOrderings()
            => _pendingOrderings.Clear();

        public void ConvertToSingleResult(MethodInfo genericMethod, params Expression[] arguments)
        {
            CardinalityReducingGenericMethodInfo = genericMethod;
            CardinalityReducingMethodArguments.AddRange(arguments);
        }

        public override ExpressionType NodeType
            => ExpressionType.Extension;

        public override Type Type
            => CardinalityReducingGenericMethodInfo == null
                ? typeof(IQueryable<>).MakeGenericType(PendingSelector.Type)
                : PendingSelector.Type;

        protected override Expression VisitChildren(ExpressionVisitor visitor)
            => this;

        void IPrintableExpression.Print(ExpressionPrinter expressionPrinter)
        {
            expressionPrinter.AppendLine(nameof(NavigationExpansionExpression));
            using (expressionPrinter.Indent())
            {
                expressionPrinter.Append("Source: ");
                expressionPrinter.Visit(Source);
                expressionPrinter.AppendLine();
                expressionPrinter.Append("PendingSelector: ");
                expressionPrinter.Visit(Lambda(PendingSelector, CurrentParameter));
                expressionPrinter.AppendLine();
                if (CardinalityReducingGenericMethodInfo != null)
                {
                    expressionPrinter.AppendLine("CardinalityReducingMethod: " + CardinalityReducingGenericMethodInfo.Name);
                }
            }
        }
    }

    private sealed class GroupByNavigationExpansionExpression : Expression, IPrintableExpression
    {
        public GroupByNavigationExpansionExpression(
            Expression source,
            ParameterExpression groupingParameter,
            NavigationTreeNode currentTree,
            Expression pendingSelector,
            string innerParameterName)
        {
            Source = source;
            CurrentParameter = groupingParameter;
            Type = source.Type;
            GroupingEnumerable = new NavigationExpansionExpression(
                Call(QueryableMethods.AsQueryable.MakeGenericMethod(CurrentParameter.Type.GetGenericArguments()[1]), CurrentParameter),
                currentTree,
                pendingSelector,
                innerParameterName);
        }

        public Expression Source { get; private set; }

        public ParameterExpression CurrentParameter { get; }

        public NavigationExpansionExpression GroupingEnumerable { get; }

        public Type SourceElementType
            => CurrentParameter.Type;

        public void UpdateSource(Expression source)
            => Source = source;

        public override ExpressionType NodeType
            => ExpressionType.Extension;

        public override Type Type { get; }

        protected override Expression VisitChildren(ExpressionVisitor visitor)
            => this;

        void IPrintableExpression.Print(ExpressionPrinter expressionPrinter)
        {
            expressionPrinter.AppendLine(nameof(GroupByNavigationExpansionExpression));
            using (expressionPrinter.Indent())
            {
                expressionPrinter.Append("Source: ");
                expressionPrinter.Visit(Source);
                expressionPrinter.AppendLine();
                expressionPrinter.Append("GroupingEnumerable: ");
                expressionPrinter.Visit(GroupingEnumerable);
                expressionPrinter.AppendLine();
            }
        }
    }

    /// <summary>
    ///     A leaf node on navigation tree, representing projection structures of
    ///     <see cref="NavigationExpansionExpression" />. Contains <see cref="Value" />,
    ///     which can be <see cref="NewExpression" /> or <see cref="EntityReference" />.
    /// </summary>
    private sealed class NavigationTreeExpression(Expression value) : NavigationTreeNode(null, null), IPrintableExpression
    {
        /// <summary>
        ///     Either <see cref="NewExpression" /> or <see cref="EntityReference" />.
        /// </summary>
        public Expression Value { get; private set; } = value;

        protected override Expression VisitChildren(ExpressionVisitor visitor)
        {
            Value = visitor.Visit(Value);

            return this;
        }

        public override Type Type
            => Value.Type;

        void IPrintableExpression.Print(ExpressionPrinter expressionPrinter)
        {
            expressionPrinter.AppendLine(nameof(NavigationTreeExpression));
            using (expressionPrinter.Indent())
            {
                expressionPrinter.Append("Value: ");
                expressionPrinter.Visit(Value);
                expressionPrinter.AppendLine();
                expressionPrinter.Append("Expression: ");
                expressionPrinter.Visit(GetExpression());
            }
        }
    }

    /// <summary>
    ///     A node in navigation binary tree. A navigation tree is a structure of the current parameter, which
    ///     would be transparent identifier (hence it's a binary structure). This allows us to easily condense to
    ///     inner/outer member access.
    /// </summary>
    private class NavigationTreeNode : Expression
    {
        private NavigationTreeNode? _parent;

        public NavigationTreeNode(NavigationTreeNode? left, NavigationTreeNode? right)
        {
            Left = left;
            Right = right;
            if (left != null
                && right != null)
            {
                left._parent = this;
                left.CurrentParameter = null;
                right._parent = this;
                right.CurrentParameter = null;
            }
        }

        public NavigationTreeNode? Left { get; }
        public NavigationTreeNode? Right { get; }
        public ParameterExpression? CurrentParameter { get; private set; }

        public void SetParameter(string parameterName)
            => CurrentParameter = Parameter(Type, parameterName);

        public override ExpressionType NodeType
            => ExpressionType.Extension;

        public override Type Type
            // Left/Right could be null for NavigationTreeExpression (derived type) but it overrides this property.
            => TransparentIdentifierFactory.Create(Left!.Type, Right!.Type);

        public Expression GetExpression()
        {
            if (_parent == null)
            {
                // If parent is null and CurrentParameter is non-null & vice-versa
                return CurrentParameter!;
            }

            var parentExpression = _parent.GetExpression();
            return _parent.Left == this
                ? MakeMemberAccess(parentExpression, parentExpression.Type.GetMember("Outer")[0])
                : MakeMemberAccess(parentExpression, parentExpression.Type.GetMember("Inner")[0]);
        }
    }

    /// <summary>
    ///     Owned navigations are not expanded, since they map differently in different providers.
    ///     This remembers such references so that they can still be treated like navigations.
    /// </summary>
    private sealed class OwnedNavigationReference(Expression parent, INavigation navigation, EntityReference entityReference)
        : Expression, IPrintableExpression
    {
        protected override Expression VisitChildren(ExpressionVisitor visitor)
        {
            Parent = visitor.Visit(Parent);

            return this;
        }

        public Expression Parent { get; private set; } = parent;
        public INavigation Navigation { get; } = navigation;
        public EntityReference EntityReference { get; } = entityReference;

        public override Type Type
            => Navigation.ClrType;

        public override ExpressionType NodeType
            => ExpressionType.Extension;

        void IPrintableExpression.Print(ExpressionPrinter expressionPrinter)
        {
            expressionPrinter.AppendLine(nameof(OwnedNavigationReference));
            using (expressionPrinter.Indent())
            {
                expressionPrinter.Append("Parent: ");
                expressionPrinter.Visit(Parent);
                expressionPrinter.AppendLine();
                expressionPrinter.Append("Navigation: " + Navigation.Name + " (OWNED)");
            }
        }
    }

    /// <summary>
    ///     Queryable properties are not expanded (similar to <see cref="OwnedNavigationReference" />.
    /// </summary>
    private sealed class ComplexPropertyReference(Expression parent, IComplexProperty complexProperty)
        : Expression, IPrintableExpression
    {
        protected override Expression VisitChildren(ExpressionVisitor visitor)
        {
            Parent = visitor.Visit(Parent);

            return this;
        }

        public Expression Parent { get; private set; } = parent;
        public new IComplexProperty Property { get; } = complexProperty;
        public ComplexTypeReference ComplexTypeReference { get; } = new(complexProperty.ComplexType);

        public override Type Type
            => Property.ClrType;

        public override ExpressionType NodeType
            => ExpressionType.Extension;

        void IPrintableExpression.Print(ExpressionPrinter expressionPrinter)
        {
            expressionPrinter.AppendLine(nameof(ComplexPropertyReference));
            using (expressionPrinter.Indent())
            {
                expressionPrinter.Append("Parent: ");
                expressionPrinter.Visit(Parent);
                expressionPrinter.AppendLine();
                expressionPrinter.Append("Property: " + Property.Name);
            }
        }
    }

    private sealed class ComplexTypeReference(IComplexType complexType) : Expression, IPrintableExpression
    {
        public IComplexType ComplexType { get; } = complexType;

        public override ExpressionType NodeType
            => ExpressionType.Extension;

        public override Type Type
            => ComplexType.ClrType;

        protected override Expression VisitChildren(ExpressionVisitor visitor)
            => this;

        void IPrintableExpression.Print(ExpressionPrinter expressionPrinter)
            => expressionPrinter.Append($"{nameof(ComplexTypeReference)}: {ComplexType.DisplayName()}");
    }

    /// <summary>
    ///     Queryable properties are not expanded (similar to <see cref="OwnedNavigationReference" />.
    /// </summary>
    private sealed class PrimitiveCollectionReference(Expression parent, IProperty property) : Expression, IPrintableExpression
    {
        protected override Expression VisitChildren(ExpressionVisitor visitor)
        {
            Parent = visitor.Visit(Parent);

            return this;
        }

        public Expression Parent { get; private set; } = parent;
        public new IProperty Property { get; } = property;

        public override Type Type
            => Property.ClrType;

        public override ExpressionType NodeType
            => ExpressionType.Extension;

        void IPrintableExpression.Print(ExpressionPrinter expressionPrinter)
        {
            expressionPrinter.AppendLine(nameof(OwnedNavigationReference));
            using (expressionPrinter.Indent())
            {
                expressionPrinter.Append("Parent: ");
                expressionPrinter.Visit(Parent);
                expressionPrinter.AppendLine();
                expressionPrinter.Append("Property: " + Property.Name + " (QUERYABLE)");
            }
        }
    }
}
