#!/usr/bin/env python3
# encoding: utf-8

# Unit tests for decopy
# Copyright (C) 2016 Maximiliano Curia <maxy@debian.org>
#
# License: ISC
#  Permission to use, copy, modify, and/or distribute this software for any
#  purpose with or without fee is hereby granted, provided that the above
#  copyright notice and this permission notice appear in all copies.
#  .
#  THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
#  REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
#  AND FITNESS. IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
#  INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
#  LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
#  OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
#  PERFORMANCE OF THIS SOFTWARE.

import json
import os
import re
import unittest

from decopy import matchers


class TestMatchers(unittest.TestCase):

    def testCleanComments(self):
        test = ('\n'
                '  c Fortran\n'
                '  C Idem\n'
                '  /* C // C++ */\n'
                ';; Lisp\n'
                '  <!-- HTML -->\n'
                'dnl m4\n'
                '  * continuation *\n'
                '\n')

        result = matchers.clean_comments(test)
        self.assertEqual(result, ' Fortran Idem C C++ Lisp HTML m4 '
                         'continuation ')

    def testParseCopyright(self):
        line = 'Copyright: Foo Bar <foo@example.com>'
        continuation = None
        copyrights, continuation = matchers.parse_copyright(line, continuation)
        self.assertEqual(len(copyrights), 1)
        self.assertEqual(copyrights[0], 'Foo Bar <foo@example.com>')
        line = '           2050- Baz <baz@example.com>'
        copyrights, continuation = matchers.parse_copyright(line, continuation)
        self.assertEqual(len(copyrights), 1)
        self.assertEqual(copyrights[0], '2050- Baz <baz@example.com>')
        line = 'Copyright (c) William Boo, (c) La Momia'
        copyrights, continuation = matchers.parse_copyright(line,
                                                            continuation)
        self.assertEqual(len(copyrights), 2)
        self.assertEqual(copyrights[0], 'William Boo')
        self.assertEqual(copyrights[1], 'La Momia')

    def testParseHolders(self):
        text = (' /* Copyright Foo Bar\n'
                '  *  Baz */\n'
                ' // Copyright (c) Lorem, (c) Ipsum')
        holders = matchers.parse_holders(text)
        self.assertEqual(len(holders), 4)
        self.assertEqual([holder.person for holder in holders],
                         ['Foo Bar', 'Baz', 'Lorem', 'Ipsum'])

    def _check_license_match(self, license_, matcher, expected, ignore=None):

        for key in ('licenseText', 'standardLicenseHeader',
                    'licenseExceptionText'):
            if key not in license_ or not license_[key]:
                continue
            if ignore and key in ignore:
                continue
            text = license_[key]
            result = matcher(text)
            self.assertEqual(result, expected,
                             msg='with key={}'.format(key))

    def _check_license_match_re(self, license_, matcher, expected):

        for key in ('licenseText', 'standardLicenseHeader',
                    'licenseExceptionText'):
            if key not in license_ or not license_[key]:
                continue
            text = license_[key]
            result = matcher(text)
            matches = re.match(expected, result)
            self.assertTrue(
                matches, msg='with key={}, result ({}) doesn\'t match with '
                'the expected value ({})\n{}'.format(
                    key, result, expected, text))

    @staticmethod
    def _find_licenses_matcher(text):
        licenses = matchers.find_licenses(
            matchers.clean_comments(text))
        return ' or '.join(sorted(licenses))

    def _open_license_and_check(self, filename, matcher, expected, ignore=None):

        with open(os.path.join('tests/testdata/licenses', filename)) as fp:
            license_ = json.load(fp)
        return self._check_license_match(license_, matcher, expected, ignore)

    def _open_license_and_check_re(self, filename, matcher, expected):

        with open(os.path.join('tests/testdata/licenses', filename)) as fp:
            license_ = json.load(fp)
        return self._check_license_match_re(license_, matcher, expected)

    def testParseBsd(self):

        def _bsd_matcher(text):
            return matchers.parse_bsd(
                matchers.clean_comments(text),
                None,
                'BSD')

        self._open_license_and_check('BSD-2-Clause.json',
                                     _bsd_matcher,
                                     'BSD-2-clause')
        self._open_license_and_check('BSD-2-Clause-FreeBSD.json',
                                     _bsd_matcher,
                                     'BSD-2-clause-FreeBSD')
        self._open_license_and_check('BSD-2-Clause-NetBSD.json',
                                     _bsd_matcher,
                                     'BSD-2-clause-NetBSD')
        self._open_license_and_check('BSD-3-Clause.json',
                                     _bsd_matcher,
                                     'BSD-3-clause')
        self._open_license_and_check('BSD-3-Clause-LBNL.json',
                                     _bsd_matcher,
                                     'BSD-3-clause')
        self._open_license_and_check('BSD-3-Clause-Clear.json',
                                     _bsd_matcher,
                                     'BSD-3-clause')
        self._open_license_and_check('BSD-3-Clause-No-Nuclear-License.json',
                                     _bsd_matcher,
                                     'BSD-3-clause-no-nuclear')
        self._open_license_and_check('BSD-3-Clause-No-Nuclear-License-2014.json',
                                     _bsd_matcher,
                                     'BSD-3-clause-no-nuclear')
        self._open_license_and_check('BSD-3-Clause-No-Nuclear-Warranty.json',
                                     _bsd_matcher,
                                     'BSD-3-clause-no-nuclear')
        self._open_license_and_check('BSD-4-Clause.json',
                                     _bsd_matcher,
                                     'BSD-4-clause')
        self._open_license_and_check('BSD-4-Clause-UC.json',
                                     _bsd_matcher,
                                     'BSD-4-clause')
        self._open_license_and_check('BSD-Protection.json',
                                     _bsd_matcher,
                                     'BSD-protection')
        self._open_license_and_check('BSD-Source-Code.json',
                                     _bsd_matcher,
                                     'BSD-source-code')
        self._open_license_and_check('BSD-3-Clause-Attribution.json',
                                     _bsd_matcher,
                                     'BSD-3-clause-attribution')
        self._open_license_and_check('0BSD.json',
                                     _bsd_matcher,
                                     '0BSD')

    def testParseCCBY(self):

        def _cc_by_matcher(text):
            return matchers.parse_cc_by(
                matchers.clean_comments(text),
                None,
                'CC-BY')

        self._open_license_and_check('CC-BY-1.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-1')
        self._open_license_and_check('CC-BY-2.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-2')
        self._open_license_and_check('CC-BY-2.5.json',
                                     _cc_by_matcher,
                                     'CC-BY-2.5')
        self._open_license_and_check('CC-BY-3.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-3')
        self._open_license_and_check('CC-BY-4.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-4')
        self._open_license_and_check('CC-BY-NC-1.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-1')
        self._open_license_and_check('CC-BY-NC-2.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-2')
        self._open_license_and_check('CC-BY-NC-2.5.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-2.5')
        self._open_license_and_check('CC-BY-NC-3.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-3')
        self._open_license_and_check('CC-BY-NC-4.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-4')
        self._open_license_and_check('CC-BY-NC-ND-1.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-ND-1')
        self._open_license_and_check('CC-BY-NC-ND-2.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-ND-2')
        self._open_license_and_check('CC-BY-NC-ND-2.5.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-ND-2.5')
        self._open_license_and_check('CC-BY-NC-ND-3.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-ND-3')
        self._open_license_and_check('CC-BY-NC-ND-4.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-ND-4')
        self._open_license_and_check('CC-BY-NC-SA-1.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-SA-1')
        self._open_license_and_check('CC-BY-NC-SA-2.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-SA-2')
        self._open_license_and_check('CC-BY-NC-SA-2.5.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-SA-2.5')
        self._open_license_and_check('CC-BY-NC-SA-3.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-SA-3')
        self._open_license_and_check('CC-BY-NC-SA-4.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-NC-SA-4')
        self._open_license_and_check('CC-BY-ND-1.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-ND-1')
        self._open_license_and_check('CC-BY-ND-2.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-ND-2')
        self._open_license_and_check('CC-BY-ND-2.5.json',
                                     _cc_by_matcher,
                                     'CC-BY-ND-2.5')
        self._open_license_and_check('CC-BY-ND-3.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-ND-3')
        self._open_license_and_check('CC-BY-ND-4.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-ND-4')
        self._open_license_and_check('CC-BY-SA-1.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-SA-1')
        self._open_license_and_check('CC-BY-SA-2.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-SA-2')
        self._open_license_and_check('CC-BY-SA-2.5.json',
                                     _cc_by_matcher,
                                     'CC-BY-SA-2.5')
        self._open_license_and_check('CC-BY-SA-3.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-SA-3')
        self._open_license_and_check('CC-BY-SA-4.0.json',
                                     _cc_by_matcher,
                                     'CC-BY-SA-4')

        self._open_license_and_check('CC0-1.0.json',
                                     self._find_licenses_matcher,
                                     'CC0')

    def testParseGPL(self):

        def _gpl_matcher(text):
            return matchers.parse_gpl(
                matchers.clean_comments(text),
                None,
                'GPL')

        self._open_license_and_check('AGPL-1.0.json',
                                     self._find_licenses_matcher,
                                     'AGPL-1')
        self._open_license_and_check_re('AGPL-3.0.json',
                                        self._find_licenses_matcher,
                                        r'AGPL-3\+?$')
        self._open_license_and_check_re('GPL-1.0.json',
                                        self._find_licenses_matcher,
                                        r'GPL-1\+?$')
        self._open_license_and_check('GPL-1.0+.json',
                                     self._find_licenses_matcher,
                                     'GPL-1+')
        self._open_license_and_check_re('GPL-2.0.json',
                                        self._find_licenses_matcher,
                                        r'GPL-2\+?$')
        self._open_license_and_check('GPL-2.0+.json',
                                     self._find_licenses_matcher,
                                     'GPL-2+')
        self._open_license_and_check_re('GPL-3.0.json',
                                        self._find_licenses_matcher,
                                        r'GPL(?:-3\+)?$')
        self._open_license_and_check('GPL-3.0+.json',
                                     self._find_licenses_matcher,
                                     'GPL-3+')

        self._open_license_and_check('GPL-2.0-with-autoconf-exception.json',
                                     _gpl_matcher,
                                     'GPL with AutoConf exception')
        self._open_license_and_check('GPL-2.0-with-bison-exception.json',
                                     _gpl_matcher,
                                     'GPL with Bison exception')
        self._open_license_and_check('GPL-2.0-with-classpath-exception.json',
                                     _gpl_matcher,
                                     'GPL with ClassPath exception')
        self._open_license_and_check('GPL-2.0-with-GCC-exception.json',
                                     _gpl_matcher,
                                     'GPL with GCC exception')

        self._open_license_and_check('GPL-3.0-with-autoconf-exception.json',
                                     _gpl_matcher,
                                     'GPL-3 with AutoConf exception')
        self._open_license_and_check('GPL-3.0-with-GCC-exception.json',
                                     _gpl_matcher,
                                     'GPL-3 with GCC exception')

        self._open_license_and_check('389-exception.json',
                                     _gpl_matcher,
                                     'GPL-2 with 389 exception')

        self._open_license_and_check('Autoconf-exception-2.0.json',
                                     _gpl_matcher,
                                     'GPL with AutoConf exception')
        self._open_license_and_check('Autoconf-exception-3.0.json',
                                     _gpl_matcher,
                                     'GPL-3 with AutoConf exception')
        self._open_license_and_check('Bison-exception-2.2.json',
                                     _gpl_matcher,
                                     'GPL with Bison exception')
        self._open_license_and_check('Classpath-exception-2.0.json',
                                     _gpl_matcher,
                                     'GPL with ClassPath exception')
        self._open_license_and_check('CLISP-exception-2.0.json',
                                     _gpl_matcher,
                                     'GPL-2 with CLISP exception')
        self._open_license_and_check('DigiRule-FOSS-exception.json',
                                     _gpl_matcher,
                                     'GPL with DigiRule-FOSS exception')
        self._open_license_and_check('eCos-exception-2.0.json',
                                     _gpl_matcher,
                                     'GPL with eCos-2 exception')
        self._open_license_and_check('Fawkes-Runtime-exception.json',
                                     _gpl_matcher,
                                     'GPL with ClassPath_MIF exception')
        self._open_license_and_check('FLTK-exception.json',
                                     _gpl_matcher,
                                     'GPL with FLTK exception')
        self._open_license_and_check('Font-exception-2.0.json',
                                     _gpl_matcher,
                                     'GPL with Font exception')
        self._open_license_and_check('freertos-exception-2.0.json',
                                     _gpl_matcher,
                                     'GPL with FreeRTOS exception')
        self._open_license_and_check('GCC-exception-2.0.json',
                                     _gpl_matcher,
                                     'GPL with GCC exception')
        self._open_license_and_check('GCC-exception-3.1.json',
                                     _gpl_matcher,
                                     'GPL-3 with GCC exception')
        self._open_license_and_check('gnu-javamail-exception.json',
                                     _gpl_matcher,
                                     'GPL with GNU-JavaMail exception')
        self._open_license_and_check('i2p-gpl-java-exception.json',
                                     _gpl_matcher,
                                     'GPL with i2p-Java exception')
        self._open_license_and_check('Libtool-exception.json',
                                     _gpl_matcher,
                                     'GPL with LibTool exception')
        self._open_license_and_check('LZMA-exception.json',
                                     _gpl_matcher,
                                     'GPL with LZMA exception')
        self._open_license_and_check('mif-exception.json',
                                     _gpl_matcher,
                                     'GPL with MIF exception')
        self._open_license_and_check('Nokia-Qt-exception-1.1.json',
                                     _gpl_matcher,
                                     'GPL-2.1 with Qt1.1 exception')
        self._open_license_and_check('OCCT-exception-1.0.json',
                                     _gpl_matcher,
                                     'GPL-2.1 with OCCT-1 exception')
        self._open_license_and_check('openvpn-openssl-exception.json',
                                     _gpl_matcher,
                                     'GPL with openvpn-openssl exception')
        # handled in the find licenses part
        # FIXME: ?
        self._open_license_and_check('Qwt-exception-1.0.json',
                                     self._find_licenses_matcher,
                                     'LGPL-2.1 with Qwt-1 exception')
        self._open_license_and_check('u-boot-exception-2.0.json',
                                     _gpl_matcher,
                                     'GPL with u-boot exception')
        self._open_license_and_check('WxWindows-exception-3.1.json',
                                     _gpl_matcher,
                                     'GPL with WxWindows-3.1+ exception')

    def testParseLGPL(self):

        self._open_license_and_check_re('LGPL-2.0.json',
                                        self._find_licenses_matcher,
                                        r'LGPL-2\+?')
        self._open_license_and_check('LGPL-2.0+.json',
                                     self._find_licenses_matcher,
                                     'LGPL-2+')
        self._open_license_and_check_re('LGPL-2.1.json',
                                        self._find_licenses_matcher,
                                        r'LGPL-2.1\+?')
        self._open_license_and_check('LGPL-2.1+.json',
                                     self._find_licenses_matcher,
                                     'LGPL-2.1+')
        self._open_license_and_check_re('LGPL-3.0.json',
                                        self._find_licenses_matcher,
                                        r'LGPL-3\+?')
        self._open_license_and_check('LGPL-3.0+.json',
                                     self._find_licenses_matcher,
                                     'LGPL-3+')
        self._open_license_and_check('LGPLLR.json',
                                     self._find_licenses_matcher,
                                     'LGPLLR')

    def testParseGFDL(self):

        self._open_license_and_check('GFDL-1.1.json',
                                     self._find_licenses_matcher,
                                     'GFDL-1.1')
        self._open_license_and_check('GFDL-1.2.json',
                                     self._find_licenses_matcher,
                                     'GFDL-NIV-1.2')
        self._open_license_and_check('GFDL-1.3.json',
                                     self._find_licenses_matcher,
                                     'GFDL-NIV-1.3')

    def testParseLPPL(self):

        self._open_license_and_check('LPPL-1.0.json',
                                     self._find_licenses_matcher,
                                     'LPPL-1+')
        self._open_license_and_check('LPPL-1.1.json',
                                     self._find_licenses_matcher,
                                     'LPPL-1.1+')
        self._open_license_and_check('LPPL-1.2.json',
                                     self._find_licenses_matcher,
                                     'LPPL-1.2+')
        self._open_license_and_check('LPPL-1.3a.json',
                                     self._find_licenses_matcher,
                                     'LPPL-1.3+')
        self._open_license_and_check('LPPL-1.3c.json',
                                     self._find_licenses_matcher,
                                     'LPPL-1.3+')

    def testParseMIT(self):

        self._open_license_and_check('MIT.json',
                                     self._find_licenses_matcher,
                                     'Expat')
        self._open_license_and_check('MIT-advertising.json',
                                     self._find_licenses_matcher,
                                     'MIT-advertising')
        self._open_license_and_check('MIT-CMU.json',
                                     self._find_licenses_matcher,
                                     'MIT-CMU')
        self._open_license_and_check('MIT-enna.json',
                                     self._find_licenses_matcher,
                                     'MIT-enna')
        self._open_license_and_check('MIT-feh.json',
                                     self._find_licenses_matcher,
                                     'MIT-feh')
        self._open_license_and_check('MITNFA.json',
                                     self._find_licenses_matcher,
                                     'MITNFA')

    def testParseZPL(self):

        self._open_license_and_check('ZPL-1.1.json',
                                     self._find_licenses_matcher,
                                     'ZPL-1.1')
        self._open_license_and_check('ZPL-2.0.json',
                                     self._find_licenses_matcher,
                                     'ZPL-2')
        self._open_license_and_check('ZPL-2.1.json',
                                     self._find_licenses_matcher,
                                     'ZPL-2.1')

    def testParseSPDX(self):

        self._open_license_and_check('AAL.json',
                                     self._find_licenses_matcher,
                                     'AAL')
        self._open_license_and_check('Abstyles.json',
                                     self._find_licenses_matcher,
                                     'Abstyles')
        self._open_license_and_check('Adobe-2006.json',
                                     self._find_licenses_matcher,
                                     'Adobe-2006')
        self._open_license_and_check('Adobe-Glyph.json',
                                     self._find_licenses_matcher,
                                     'Adobe-Glyph')
        self._open_license_and_check('ADSL.json',
                                     self._find_licenses_matcher,
                                     'ADSL')
        self._open_license_and_check('AFL-1.1.json',
                                     self._find_licenses_matcher,
                                     'Academic-1.1')
        self._open_license_and_check('AFL-1.2.json',
                                     self._find_licenses_matcher,
                                     'Academic-1.2')
        self._open_license_and_check('AFL-2.0.json',
                                     self._find_licenses_matcher,
                                     'Academic-2')
        self._open_license_and_check('AFL-2.1.json',
                                     self._find_licenses_matcher,
                                     'Academic-2.1')
        self._open_license_and_check('AFL-3.0.json',
                                     self._find_licenses_matcher,
                                     'Academic-3')
        self._open_license_and_check('Afmparse.json',
                                     self._find_licenses_matcher,
                                     'Afmparse')
        self._open_license_and_check('Aladdin.json',
                                     self._find_licenses_matcher,
                                     'Aladdin')
        self._open_license_and_check('AMDPLPA.json',
                                     self._find_licenses_matcher,
                                     'AMDPLPA')
        self._open_license_and_check('AML.json',
                                     self._find_licenses_matcher,
                                     'AML')
        self._open_license_and_check('AMPAS.json',
                                     self._find_licenses_matcher,
                                     'BSD-2-clause-AMPAS')
        self._open_license_and_check('ANTLR-PD.json',
                                     self._find_licenses_matcher,
                                     'ANTLR-PD')
        self._open_license_and_check('Apache-1.0.json',
                                     self._find_licenses_matcher,
                                     'Apache-1')
        self._open_license_and_check('Apache-1.1.json',
                                     self._find_licenses_matcher,
                                     'Apache-1.1')
        self._open_license_and_check('Apache-2.0.json',
                                     self._find_licenses_matcher,
                                     'Apache-2')
        self._open_license_and_check('APAFML.json',
                                     self._find_licenses_matcher,
                                     'APAFML')
        self._open_license_and_check('APL-1.0.json',
                                     self._find_licenses_matcher,
                                     'APL-1',
                                     ignore={'standardLicenseHeader'})
        self._open_license_and_check('APSL-1.0.json',
                                     self._find_licenses_matcher,
                                     'APSL-1')
        self._open_license_and_check('APSL-1.1.json',
                                     self._find_licenses_matcher,
                                     'APSL-1.1')
        self._open_license_and_check('APSL-1.2.json',
                                     self._find_licenses_matcher,
                                     'APSL-1.2')
        self._open_license_and_check('APSL-2.0.json',
                                     self._find_licenses_matcher,
                                     'APSL-2')
        self._open_license_and_check('Artistic-1.0-cl8.json',
                                     self._find_licenses_matcher,
                                     'Artistic-1-cl8')
        self._open_license_and_check('Artistic-1.0-Perl.json',
                                     self._find_licenses_matcher,
                                     'Artistic-1-Perl')
        self._open_license_and_check('Artistic-1.0.json',
                                     self._find_licenses_matcher,
                                     'Artistic-1')
        self._open_license_and_check('Artistic-2.0.json',
                                     self._find_licenses_matcher,
                                     'Artistic-2')
        self._open_license_and_check('Bahyph.json',
                                     self._find_licenses_matcher,
                                     'Bahyph')
        self._open_license_and_check('Barr.json',
                                     self._find_licenses_matcher,
                                     'Barr')
        self._open_license_and_check('BitTorrent-1.0.json',
                                     self._find_licenses_matcher,
                                     'BitTorrent-1')
        # The header has the wrong version in it
        self._open_license_and_check('BitTorrent-1.1.json',
                                     self._find_licenses_matcher,
                                     'BitTorrent-1')
        self._open_license_and_check('Borceux.json',
                                     self._find_licenses_matcher,
                                     'Borceux')
        self._open_license_and_check('BSL-1.0.json',
                                     self._find_licenses_matcher,
                                     'BSL-1')
        self._open_license_and_check('bzip2-1.0.5.json',
                                     self._find_licenses_matcher,
                                     'BSD-like-bzip2')
        self._open_license_and_check('bzip2-1.0.6.json',
                                     self._find_licenses_matcher,
                                     'BSD-like-bzip2')
        self._open_license_and_check('Caldera.json',
                                     self._find_licenses_matcher,
                                     'Caldera')
        self._open_license_and_check('CATOSL-1.1.json',
                                     self._find_licenses_matcher,
                                     'CATOSL-1.1')
        self._open_license_and_check('CDDL-1.0.json',
                                     self._find_licenses_matcher,
                                     'CDDL-1')
        self._open_license_and_check('CDDL-1.1.json',
                                     self._find_licenses_matcher,
                                     'CDDL-1.1')
        self._open_license_and_check('CECILL-1.0.json',
                                     self._find_licenses_matcher,
                                     'CeCILL-1')
        self._open_license_and_check('CECILL-1.1.json',
                                     self._find_licenses_matcher,
                                     'CeCILL-1.1')
        self._open_license_and_check('CECILL-2.0.json',
                                     self._find_licenses_matcher,
                                     'CeCILL-2')
        self._open_license_and_check('CECILL-2.1.json',
                                     self._find_licenses_matcher,
                                     'CeCILL-2.1')
        self._open_license_and_check('CECILL-B.json',
                                     self._find_licenses_matcher,
                                     'CeCILL-B')
        self._open_license_and_check('CECILL-C.json',
                                     self._find_licenses_matcher,
                                     'CeCILL-C')

        self._open_license_and_check('ClArtistic.json',
                                     self._find_licenses_matcher,
                                     'ClArtistic')

        self._open_license_and_check('CNRI-Jython.json',
                                     self._find_licenses_matcher,
                                     'CNRI-Jython')
        self._open_license_and_check('CNRI-Python-GPL-Compatible.json',
                                     self._find_licenses_matcher,
                                     'CNRI-Python-GPL-Compatible')
        self._open_license_and_check('CNRI-Python.json',
                                     self._find_licenses_matcher,
                                     'CNRI-Python')

        self._open_license_and_check('Condor-1.1.json',
                                     self._find_licenses_matcher,
                                     'Condor-1.1+')

        self._open_license_and_check('CPAL-1.0.json',
                                     self._find_licenses_matcher,
                                     'CPAL-1')
        self._open_license_and_check('CPL-1.0.json',
                                     self._find_licenses_matcher,
                                     'CPL-1')
        self._open_license_and_check('CPOL-1.02.json',
                                     self._find_licenses_matcher,
                                     'CPOL-1.02')
        self._open_license_and_check('Crossword.json',
                                     self._find_licenses_matcher,
                                     'Crossword')
        self._open_license_and_check('CrystalStacker.json',
                                     self._find_licenses_matcher,
                                     'CrystalStacker')
        self._open_license_and_check('CUA-OPL-1.0.json',
                                     self._find_licenses_matcher,
                                     'CUA-OPL-1')
        self._open_license_and_check('Cube.json',
                                     self._find_licenses_matcher,
                                     'Cube')
        self._open_license_and_check('curl.json',
                                     self._find_licenses_matcher,
                                     'curl')

        self._open_license_and_check('D-FSL-1.0.json',
                                     self._find_licenses_matcher,
                                     'D-FSL')
        self._open_license_and_check('diffmark.json',
                                     self._find_licenses_matcher,
                                     'diffmark')
        self._open_license_and_check('DOC.json',
                                     self._find_licenses_matcher,
                                     'DOC')
        self._open_license_and_check('Dotseqn.json',
                                     self._find_licenses_matcher,
                                     'Dotseqn')
        self._open_license_and_check('DSDP.json',
                                     self._find_licenses_matcher,
                                     'DSDP')
        self._open_license_and_check('dvipdfm.json',
                                     self._find_licenses_matcher,
                                     'dvipdfm')

        self._open_license_and_check('ECL-1.0.json',
                                     self._find_licenses_matcher,
                                     'ECL-1')
        self._open_license_and_check('ECL-2.0.json',
                                     self._find_licenses_matcher,
                                     'ECL-2')
        self._open_license_and_check('eCos-2.0.json',
                                     self._find_licenses_matcher,
                                     'GPL-2+ with eCos-2 exception')
        self._open_license_and_check('EFL-1.0.json',
                                     self._find_licenses_matcher,
                                     'EFL-1')
        self._open_license_and_check('EFL-2.0.json',
                                     self._find_licenses_matcher,
                                     'EFL-2')
        self._open_license_and_check('eGenix.json',
                                     self._find_licenses_matcher,
                                     'eGenix')
        self._open_license_and_check('Entessa.json',
                                     self._find_licenses_matcher,
                                     'BSD-like-Entessa')
        self._open_license_and_check('EPL-1.0.json',
                                     self._find_licenses_matcher,
                                     'EPL-1')
        self._open_license_and_check('ErlPL-1.1.json',
                                     self._find_licenses_matcher,
                                     'ErlPL-1.1')
        self._open_license_and_check('EUDatagrid.json',
                                     self._find_licenses_matcher,
                                     'EUDatagrid')
        self._open_license_and_check('EUPL-1.0.json',
                                     self._find_licenses_matcher,
                                     'EUPL-1')
        self._open_license_and_check('EUPL-1.1.json',
                                     self._find_licenses_matcher,
                                     'EUPL-1.1')
        self._open_license_and_check('Eurosym.json',
                                     self._find_licenses_matcher,
                                     'Eurosym')

        self._open_license_and_check('Fair.json',
                                     self._find_licenses_matcher,
                                     'Fair')
        self._open_license_and_check('Frameworx-1.0.json',
                                     self._find_licenses_matcher,
                                     'Frameworx-1')
        self._open_license_and_check('FreeImage.json',
                                     self._find_licenses_matcher,
                                     'FreeImage-1')
        self._open_license_and_check('FSFAP.json',
                                     self._find_licenses_matcher,
                                     'FSFAP')
        self._open_license_and_check('FSFUL.json',
                                     self._find_licenses_matcher,
                                     'FSFUL')
        self._open_license_and_check('FSFULLR.json',
                                     self._find_licenses_matcher,
                                     'FSFULLR')
        self._open_license_and_check('FTL.json',
                                     self._find_licenses_matcher,
                                     'FTL')

        self._open_license_and_check('GFDL-1.1.json',
                                     self._find_licenses_matcher,
                                     'GFDL-1.1')
        self._open_license_and_check('GFDL-1.2.json',
                                     self._find_licenses_matcher,
                                     'GFDL-NIV-1.2')
        self._open_license_and_check('GFDL-1.3.json',
                                     self._find_licenses_matcher,
                                     'GFDL-NIV-1.3')
        self._open_license_and_check('Giftware.json',
                                     self._find_licenses_matcher,
                                     'Giftware')
        self._open_license_and_check('GL2PS.json',
                                     self._find_licenses_matcher,
                                     'GL2PS')
        self._open_license_and_check('Glide.json',
                                     self._find_licenses_matcher,
                                     'Glide')
        self._open_license_and_check('Glulxe.json',
                                     self._find_licenses_matcher,
                                     'Glulxe')
        self._open_license_and_check('gnuplot.json',
                                     self._find_licenses_matcher,
                                     'gnuplot')
        self._open_license_and_check('gSOAP-1.3b.json',
                                     self._find_licenses_matcher,
                                     'gSOAP-1.3')

        self._open_license_and_check('HaskellReport.json',
                                     self._find_licenses_matcher,
                                     'HaskellReport')
        self._open_license_and_check('HPND.json',
                                     self._find_licenses_matcher,
                                     'HPND')


if __name__ == '__main__':
    unittest.main()
