# SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
# SPDX-License-Identifier: GPL-2.0-or-later
"""The base class definition for invoking OpenPGP implementations."""

from __future__ import annotations

import abc
import typing


if typing.TYPE_CHECKING:
    import pathlib

    from testsigs import defs


class PGPCli(abc.ABC):
    """An OpenPGP implementation."""

    # Abstract dataclasses still have some quirks, so do this the hard way.

    cfg: defs.Config
    """The runtime configuration settings, e.g. the paths to the tools to invoke."""

    def __init__(self, cfg: defs.Config) -> None:
        """Initialize a `PGPCli` object."""
        self.cfg = cfg

    @abc.abstractmethod
    def generate_keys(self, uid: str) -> defs.KeyFiles:
        """Generate an OpenPGP key to sign with."""
        raise NotImplementedError

    @abc.abstractmethod
    def import_secret_key(self, keys: defs.KeyFiles, uid: str) -> defs.PublicKey:
        """Import a secret key, return the parts of the public key we care about."""
        raise NotImplementedError

    @abc.abstractmethod
    def sign_detached(self, keys: defs.KeyFiles, datafile: pathlib.Path) -> pathlib.Path:
        """Create an ASCII-armored detached signature, return the path to the new file."""
        raise NotImplementedError

    @abc.abstractmethod
    def verify_detached(
        self,
        keys: defs.KeyFiles,
        pub_key: defs.PublicKey,
        datafile: pathlib.Path,
        signature: pathlib.Path,
    ) -> None:
        """Verify an OpenPGP detached signature in a file."""
        raise NotImplementedError
