/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "JobThumbnail.h"

#include <QFileInfo>
#include <QImage>
#include <QMatrix>
#include <QMutexLocker>

#include <klocale.h>

#include <threadweaver/ThreadWeaver.h>

#include <libkexiv2/kexiv2.h>

#include "RawImageInfo.h"
#include "ExportCodec.h"
#include "libraw/libraw/libraw.h"
#include "MessagesModel.h"

struct JobThumbnail::Private
{
  RawImageInfoSP rawImageInfo;
  static ThreadWeaver::Weaver* weaver;
};

ThreadWeaver::Weaver* JobThumbnail::Private::weaver = 0;

JobThumbnail::JobThumbnail( RawImageInfoSP _rawImageInfo) : d(new Private)
{
  Q_ASSERT( _rawImageInfo );
  d->rawImageInfo = _rawImageInfo;
}

JobThumbnail::~JobThumbnail()
{
  delete d;
}

ThreadWeaver::Weaver* JobThumbnail::weaver()
{
  if(not Private::weaver)
  {
    Private::weaver = new ThreadWeaver::Weaver;
  }
  return Private::weaver;
}

void JobThumbnail::run()
{
  QImage img;
  QByteArray imgData;

  LibRaw raw;
  raw.imgdata.params.use_auto_wb    = 1; // Use automatic white balance.
  raw.imgdata.params.use_camera_wb  = 1; // Use camera white balance, if possible.
  raw.imgdata.params.half_size      = 1; // Half-size color image (3x faster than -q).
  raw.imgdata.params.filtering_mode = LIBRAW_FILTERING_AUTOMATIC;
  int ret = raw.open_file(QFile::encodeName(d->rawImageInfo->fileInfo().absoluteFilePath()));
  if(ret != LIBRAW_SUCCESS) {
    MessagesModel::instance()->tellError(i18n("Thumbnail"), i18n("Failed to open %1.", d->rawImageInfo->fileInfo().absoluteFilePath()));
    return;
  }
  raw.unpack_thumb();
  if(ret == LIBRAW_SUCCESS) {
    // There is an embedded thumbnail
    libraw_processed_image_t *thumb = raw.dcraw_make_mem_thumb(&ret);
    if(thumb) {
      QByteArray imgData;
      if(thumb->type == LIBRAW_IMAGE_BITMAP) {
        QString header = QString("P6\n%1 %2\n%3\n").arg(thumb->width).arg(thumb->height).arg((1 << thumb->bits)-1);
        imgData.append(header.toAscii());
      }
      imgData.append(QByteArray((const char*)thumb->data, (int)thumb->data_size));
      img.loadFromData(imgData);
    }
  }
  
  // Try to load half of the image
  if(img.isNull()) {
    ret = raw.unpack();
    if (ret == LIBRAW_SUCCESS)
    {
      ret = raw.dcraw_process();
      if (ret == LIBRAW_SUCCESS)
      {
        libraw_processed_image_t *halfImg = raw.dcraw_make_mem_image(&ret);
        if(halfImg)
        {
          QByteArray imgData;
          QString header = QString("P6\n%1 %2\n%3\n").arg(halfImg->width).arg(halfImg->height).arg((1 << halfImg->bits)-1);
          imgData.append(header.toAscii());
          imgData.append(QByteArray((const char*)halfImg->data, (int)halfImg->data_size));
          img.loadFromData(imgData);
        }
      }
    }
  }
  if(img.isNull()) {
    d->rawImageInfo->setStatus( RawImageInfo::ERROR );
    MessagesModel::instance()->tellError(i18n("Thumbnail"), i18n("Failed to extract thumbnail from %1.", d->rawImageInfo->fileInfo().absoluteFilePath()));
    return;
  }
  
  QMutexLocker locker(ExportCodec::exiv2Mutex());
  KExiv2Iface::KExiv2 exiv2;
  if(exiv2.load( d->rawImageInfo->fileInfo().absoluteFilePath()))
  {
    if(exiv2.hasExif()) {
    	KExiv2Iface::KExiv2::ImageOrientation orient = exiv2.getImageOrientation ();
    	if (orient == KExiv2Iface::KExiv2::ORIENTATION_ROT_90)
    	{
    	  QMatrix m_90;
    	  m_90.rotate (90.0);
    	  img = img.transformed (m_90);
    	} else if (orient == KExiv2Iface::KExiv2::ORIENTATION_ROT_270)
    	{
    	  QMatrix m_270;	
    	  m_270.rotate (270.0);
    	  img = img.transformed (m_270);
	}
    }
  }
  img = img.scaled ( 100, 100, Qt::KeepAspectRatio);
  d->rawImageInfo->setThumbnail(img );
}
