# Copyright (C) 2010 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA

"""Widgets for browsing sets of interesting locations."""

from PyQt4 import QtCore, QtGui

from bzrlib.plugins.explorer.lib import (
    kinds,
    )
from bzrlib.plugins.explorer.lib.extensions import bookmarks
from bzrlib.plugins.explorer.lib.i18n import gettext, N_
from bzrlib.plugins.explorer.widgets import (
    conditional_dataview,
    )


class LocationSetBrowser(QtGui.QWidget):

    def __init__(self, open_callback, *args):
        self._open_callback = open_callback
        QtGui.QWidget.__init__(self, *args)

    def refresh_view(self):
        """Refresh the view."""

    def count(self):
        """Return the number of locations in the set."""
        return 0


class BookmarksBrowser(LocationSetBrowser):

    def __init__(self, stores, open_callback, *args):
        LocationSetBrowser.__init__(self, open_callback, *args)
        self._stores = stores
        self.setLayout(self._build_layout())
        self.bookmarks_by_row = []

    def _build_layout(self):
        dataview = conditional_dataview.QBzrConditionalDataView("list",
                True, "", None)
        dataview.connect(dataview.view(),
            QtCore.SIGNAL("activated(QModelIndex)"),
            self._do_selected)
        self.dataview = dataview
        layout = QtGui.QVBoxLayout()
        layout.addWidget(dataview)
        layout.setContentsMargins(0, 0, 0, 0)
        return layout

    def _do_selected(self, model_index):
        model = model_index.model()
        row = model_index.row()
        bm = self.bookmarks_by_row[row]
        self._open_callback(bm.location)

    def refresh_view(self):
        self.bookmarks_by_row = []
        tuple_list = []
        icon_list = []
        for store in self._stores:
            self._append_bookmarks(store.root(), tuple_list, icon_list)
        decorator_provider = lambda row, record: icon_list[row]
        self.dataview.setData(tuple_list, decorator_provider)

    def _append_bookmarks(self, folder, tuple_list, icon_list):
        for entry in folder:
            if isinstance(entry, bookmarks.Bookmark):
                self.bookmarks_by_row.append(entry)
                tuple_list.append((entry.title,))
                icon_list.append(kinds.icon_for_kind(entry.kind))
            elif isinstance(entry, bookmarks.BookmarkFolder):
                # TODO: Nest folders, don't flatten them
                self._append_bookmarks(entry, tuple_list, icon_list)

    def count(self):
        """Return the number of locations in the set."""
        return len(self.bookmarks_by_row)


class HistoryBrowser(LocationSetBrowser):

    def __init__(self, history_manager, open_callback, *args):
        LocationSetBrowser.__init__(self, open_callback, *args)
        self._history_manager = history_manager
        self.setLayout(self._build_layout())
        self.locations_by_row = []

    def _build_layout(self):
        dataview = conditional_dataview.QBzrConditionalDataView("list",
                True, "", None)
        dataview.connect(dataview.view(),
            QtCore.SIGNAL("activated(QModelIndex)"),
            self._do_selected)
        self.dataview = dataview
        layout = QtGui.QVBoxLayout()
        layout.addWidget(dataview)
        layout.setContentsMargins(0, 0, 0, 0)
        return layout

    def _do_selected(self, model_index):
        model = model_index.model()
        row = model_index.row()
        location = self.locations_by_row[row]
        self._open_callback(location)

    def refresh_view(self):
        self.locations_by_row = []
        tuple_list = []
        icon_list = []
        for location, kind, title in self._history_manager.recent_locations():
            self.locations_by_row.append(location)
            tuple_list.append((title,))
            icon_list.append(kinds.icon_for_kind(kind))
        decorator_provider = lambda row, record: icon_list[row]
        self.dataview.setData(tuple_list, decorator_provider)

    def count(self):
        """Return the number of locations in the set."""
        return len(self.locations_by_row)
