//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Job/ParameterTreeItems.h
//! @brief     Defines classes for ParameterTreeItems
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_JOB_PARAMETERTREEITEMS_H
#define BORNAGAIN_GUI_MODEL_JOB_PARAMETERTREEITEMS_H

#include "GUI/Model/Descriptor/DoubleProperty.h"
#include <QMap>
#include <QXmlStreamReader>

//! ParameterTreeItems is a collection of items necessary to form a tuning tree for
//! real time widget.

//! The ParameterLabelItem class represents a label (string without value, like 'Layer',
//! 'MultiLayer') in a parameter tuning tree.

class ParameterLabelItem : public QObject {
    Q_OBJECT
public:
    ParameterLabelItem(QObject* parent);
    ParameterLabelItem(const QString& title, QObject* parent);

    QString title() const;
    void setTitle(const QString& title);

private:
    QString m_title;
};

//! The ParameterItem class represent a tuning value in a parameter tuning tree.

class ParameterItem : public QObject {
    Q_OBJECT
public:
    ParameterItem(QObject* parent);

    QString title() const;
    void setTitle(const QString& title);

    double valueOfLink() const;
    void propagateValueToLink(double newValue);

    //! Unique string to identify this ParameterItem.
    //!
    //! The link is arbitrary. It cannot be used for finding the linked item (therefore it does
    //! not have to be a model path). However, it is used for comparison, also across project
    //! load/save. Therefore the link is always the same, not e.g. an always generated UUID.
    //! This link is used for setting backup values and for finding this ParameterItem when
    //! referring from fit parameters.
    QString link() const;

    QString titleForFitItem() const;

    //! Links this item to the given value defined by a property.
    void linkToProperty(DoubleProperty& d);

    RealLimits limitsOfLink() const;
    int decimalsOfLink() const;

private:
    QString m_title;
    DoubleProperty m_d; // a copy for the access to limits, uid, decimals
    std::function<double()> getPropertyValue = nullptr;
    std::function<void(double)> setPropertyValue = nullptr;
};

//! The ParameterContainerItem is a top item to hold all ParameterItem, represents an entry
//! point to parameter tuning tree. Part of JobItem.

class ParameterContainerItem {
public:
    ParameterContainerItem();

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    void setBackupValue(const QString& link, double d);
    void restoreBackupValues();

    ParameterItem* findParameterItem(const QString& link) const;

    QObject* parameterTreeRoot();

private:
    void restoreBackupValue(QObject* item);

    QMap<QString, double> m_backupValues;
    std::unique_ptr<QObject> m_parameterTreeRoot;
};

#endif // BORNAGAIN_GUI_MODEL_JOB_PARAMETERTREEITEMS_H
