/*
 * Standardized error code module.
 * Header file
 * ---------------------------------------------------------------------------
 * DESIGN
 *
 * The purpose of this module is to unify error codes and error messages.
 *
 * Traditionally, errors are assigned an arbitrary integer value (or possibly
 * an enumerated value). These codes are unique only within the module that
 * defines them; when error codes are returned from several different modules
 * it is difficult to distinguish which error came from where. Also, if the
 * error code needs to be propagated through several modules, a series of
 * conversions are required. To obtain a human-readable message, a special
 * error-to-message mapping function is usually needed. After the error code
 * has been converted several times, it is difficult to know which mapping
 * function should be used.
 *
 * This module tries to fix these problems. Error codes are defined such that
 * every error is unique throughout the entire program. Furthermore, an error
 * code equals the error message, and thus the message can be accessed directly
 * without any need of locating and using the correct mapping function.
 *
 * This is accomplished by defining an error code to be a pointer to the
 * message describing the error. Since all strings are stored at unique memory
 * locations, all error codes will be unique. There is not even a need to
 * manually assign unique error codes across all modules (which is impractical,
 * especially if library modules are used). Also, the error code itself is
 * the pointer to the error message: the message associated with the error is
 * right at your fingertips ready to be accessed.
 *
 * IMPLEMENTATION
 *
 * By convention, all error codes will have uppercase names, with the first
 * part of the name containing a prefix that identifies which module the
 * error is defined in. Errors are defined using the 'error_d' typedef:
 *	error_d STK_CANNOT_POP="Nothing to pop from stack";
 * Error codes are stored in variables of type 'error_t', but assigned an
 * initial value from 'error_d':
 *	error_t e;
 *	e = STK_CANNOT_POP;
 *	printf("Error: %s\n", e);
 *
 * By convention, error codes should be declared in the header file of the
 * module it belongs to, but defined in the implementation files. So, the
 * header file should contain something like:
 *	extern error_d STK_CANNOT_POP;
 *	extern error_d STK_INTERNAL_ERROR;
 *	... etc.
 */

#ifndef ERROR_H
#define ERROR_H

typedef char error_d[];
typedef char *error_t;
#define NO_ERROR			((error_t)0)


#endif /* ERROR_H */
