/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.view.translation;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.android.internal.util.DataClass;

import java.util.Objects;

/**
 * A translated response value from {@link android.service.translation.TranslationService}.
 */
@DataClass(genBuilder = true, genToString = true, genEqualsHashCode = true,
        genHiddenConstDefs = true)
public final class TranslationResponseValue implements Parcelable {

    /**
     * This value was successfully translated.
     */
    public static final int STATUS_SUCCESS = 0;
    /**
     * This value was not successfully translated. No value can be obtained with {@link #getText()}.
     */
    public static final int STATUS_ERROR = 1;

    /**
     * Name in the result of {@link #getExtras()} to pass dictionary definitions of the text
     * categorized by parts of speech.
     *
     * <p>The dictionary definitions consists of groups of terms keyed by their corresponding parts
     * of speech. This map-like structure is stored in a {@link Bundle}. The individual parts of
     * speech can be traversed by {@link Bundle#keySet()} and used to get the corresponding list
     * of terms as {@link CharSequence}s.
     *
     * <ul>
     *     <li>"noun" -> ["def1", "def2", ...]</li>
     *     <li>"verb" -> ["def3", "def4", ...]</li>
     *     <li>...</li>
     * </ul>
     *
     * The set of parts of speech can then be used by
     * {@link Bundle#getCharSequenceArrayList(String)} to get the list of terms.
     *
     * <b>Example</b>:
     *
     * {@code for (String partOfSpeech : extras.getBundle(EXTRA_DEFINITIONS).keySet()) {
     *    ArrayList<CharSequence> terms =
     *            extras.getBundle(EXTRA_DEFINITIONS).getCharSequenceArrayList(partOfSpeech);
     *    ...
     * }}</p>
     */
    public static final String EXTRA_DEFINITIONS = "android.view.translation.extra.DEFINITIONS";

    /**
     * The status code of this {@link TranslationResponseValue}.
     *
     * <p>If the status code is {@link #STATUS_ERROR}, no values are attached, and all getters will
     * return {@code null}.
     */
    private final @Status int mStatusCode;

    /**
     * The translated text result.
     */
    @Nullable
    private final CharSequence mText;

    /**
     * Extra results associated with the translated text.
     *
     * <p>The bundle includes {@link #EXTRA_DEFINITIONS}, obtained by {@link Bundle#getBundle}.
     * </p>
     */
    @NonNull
    private final Bundle mExtras;

    // TODO: Add example of transliteration.
    /**
     * The transliteration result of the translated text.
     *
     * <p>This returns a CharSequence representation of the transliteration of the translated text.
     */
    @Nullable
    private final CharSequence mTransliteration;

    /**
     * Creates a {@link TranslationResponseValue} with the {@link #STATUS_ERROR} result;
     */
    @NonNull
    public static TranslationResponseValue forError() {
        return new TranslationResponseValue(STATUS_ERROR, null, Bundle.EMPTY, null);
    }

    private static CharSequence defaultText() {
        return null;
    }

    private static Bundle defaultExtras() {
        return Bundle.EMPTY;
    }

    private boolean extrasEquals(Bundle other) {
        // TODO: Also compare the contents.
        return Objects.equals(mExtras, other) || (mExtras.isEmpty() && other.isEmpty());
    }

    private static CharSequence defaultTransliteration() {
        return null;
    }

    @DataClass.Suppress("setStatusCode")
    abstract static class BaseBuilder {

    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/view/translation/TranslationResponseValue.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /** @hide */
    @android.annotation.IntDef(prefix = "STATUS_", value = {
        STATUS_SUCCESS,
        STATUS_ERROR
    })
    @java.lang.annotation.Retention(java.lang.annotation.RetentionPolicy.SOURCE)
    @DataClass.Generated.Member
    public @interface Status {}

    /** @hide */
    @DataClass.Generated.Member
    public static String statusToString(@Status int value) {
        switch (value) {
            case STATUS_SUCCESS:
                    return "STATUS_SUCCESS";
            case STATUS_ERROR:
                    return "STATUS_ERROR";
            default: return Integer.toHexString(value);
        }
    }

    @DataClass.Generated.Member
    /* package-private */ TranslationResponseValue(
            @Status int statusCode,
            @Nullable CharSequence text,
            @NonNull Bundle extras,
            @Nullable CharSequence transliteration) {
        this.mStatusCode = statusCode;

        if (!(mStatusCode == STATUS_SUCCESS)
                && !(mStatusCode == STATUS_ERROR)) {
            throw new java.lang.IllegalArgumentException(
                    "statusCode was " + mStatusCode + " but must be one of: "
                            + "STATUS_SUCCESS(" + STATUS_SUCCESS + "), "
                            + "STATUS_ERROR(" + STATUS_ERROR + ")");
        }

        this.mText = text;
        this.mExtras = extras;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mExtras);
        this.mTransliteration = transliteration;

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * The status code of this {@link TranslationResponseValue}.
     *
     * <p>If the status code is {@link #STATUS_ERROR}, no values are attached, and all getters will
     * return {@code null}.
     */
    @DataClass.Generated.Member
    public @Status int getStatusCode() {
        return mStatusCode;
    }

    /**
     * The translated text result.
     */
    @DataClass.Generated.Member
    public @Nullable CharSequence getText() {
        return mText;
    }

    /**
     * Extra results associated with the translated text.
     *
     * <p>The bundle includes {@link #EXTRA_DEFINITIONS}, obtained by {@link Bundle#getBundle}.
     * </p>
     */
    @DataClass.Generated.Member
    public @NonNull Bundle getExtras() {
        return mExtras;
    }

    /**
     * The transliteration result of the translated text.
     *
     * <p>This returns a CharSequence representation of the transliteration of the translated text.
     */
    @DataClass.Generated.Member
    public @Nullable CharSequence getTransliteration() {
        return mTransliteration;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "TranslationResponseValue { " +
                "statusCode = " + statusToString(mStatusCode) + ", " +
                "text = " + mText + ", " +
                "extras = " + mExtras + ", " +
                "transliteration = " + mTransliteration +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public boolean equals(@Nullable Object o) {
        // You can override field equality logic by defining either of the methods like:
        // boolean fieldNameEquals(TranslationResponseValue other) { ... }
        // boolean fieldNameEquals(FieldType otherValue) { ... }

        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        @SuppressWarnings("unchecked")
        TranslationResponseValue that = (TranslationResponseValue) o;
        //noinspection PointlessBooleanExpression
        return true
                && mStatusCode == that.mStatusCode
                && Objects.equals(mText, that.mText)
                && extrasEquals(that.mExtras)
                && Objects.equals(mTransliteration, that.mTransliteration);
    }

    @Override
    @DataClass.Generated.Member
    public int hashCode() {
        // You can override field hashCode logic by defining methods like:
        // int fieldNameHashCode() { ... }

        int _hash = 1;
        _hash = 31 * _hash + mStatusCode;
        _hash = 31 * _hash + Objects.hashCode(mText);
        _hash = 31 * _hash + Objects.hashCode(mExtras);
        _hash = 31 * _hash + Objects.hashCode(mTransliteration);
        return _hash;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        byte flg = 0;
        if (mText != null) flg |= 0x2;
        if (mTransliteration != null) flg |= 0x8;
        dest.writeByte(flg);
        dest.writeInt(mStatusCode);
        if (mText != null) dest.writeCharSequence(mText);
        dest.writeBundle(mExtras);
        if (mTransliteration != null) dest.writeCharSequence(mTransliteration);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ TranslationResponseValue(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        byte flg = in.readByte();
        int statusCode = in.readInt();
        CharSequence text = (flg & 0x2) == 0 ? null : (CharSequence) in.readCharSequence();
        Bundle extras = in.readBundle();
        CharSequence transliteration = (flg & 0x8) == 0 ? null : (CharSequence) in.readCharSequence();

        this.mStatusCode = statusCode;

        if (!(mStatusCode == STATUS_SUCCESS)
                && !(mStatusCode == STATUS_ERROR)) {
            throw new java.lang.IllegalArgumentException(
                    "statusCode was " + mStatusCode + " but must be one of: "
                            + "STATUS_SUCCESS(" + STATUS_SUCCESS + "), "
                            + "STATUS_ERROR(" + STATUS_ERROR + ")");
        }

        this.mText = text;
        this.mExtras = extras;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mExtras);
        this.mTransliteration = transliteration;

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<TranslationResponseValue> CREATOR
            = new Parcelable.Creator<TranslationResponseValue>() {
        @Override
        public TranslationResponseValue[] newArray(int size) {
            return new TranslationResponseValue[size];
        }

        @Override
        public TranslationResponseValue createFromParcel(@NonNull Parcel in) {
            return new TranslationResponseValue(in);
        }
    };

    /**
     * A builder for {@link TranslationResponseValue}
     */
    @SuppressWarnings("WeakerAccess")
    @DataClass.Generated.Member
    public static final class Builder extends BaseBuilder {

        private @Status int mStatusCode;
        private @Nullable CharSequence mText;
        private @NonNull Bundle mExtras;
        private @Nullable CharSequence mTransliteration;

        private long mBuilderFieldsSet = 0L;

        /**
         * Creates a new Builder.
         *
         * @param statusCode
         *   The status code of this {@link TranslationResponseValue}.
         *
         *   <p>If the status code is {@link #STATUS_ERROR}, no values are attached, and all getters will
         *   return {@code null}.
         */
        public Builder(
                @Status int statusCode) {
            mStatusCode = statusCode;

            if (!(mStatusCode == STATUS_SUCCESS)
                    && !(mStatusCode == STATUS_ERROR)) {
                throw new java.lang.IllegalArgumentException(
                        "statusCode was " + mStatusCode + " but must be one of: "
                                + "STATUS_SUCCESS(" + STATUS_SUCCESS + "), "
                                + "STATUS_ERROR(" + STATUS_ERROR + ")");
            }

        }

        /**
         * The translated text result.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setText(@NonNull CharSequence value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x2;
            mText = value;
            return this;
        }

        /**
         * Extra results associated with the translated text.
         *
         * <p>The bundle includes {@link #EXTRA_DEFINITIONS}, obtained by {@link Bundle#getBundle}.
         * </p>
         */
        @DataClass.Generated.Member
        public @NonNull Builder setExtras(@NonNull Bundle value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x4;
            mExtras = value;
            return this;
        }

        /**
         * The transliteration result of the translated text.
         *
         * <p>This returns a CharSequence representation of the transliteration of the translated text.
         */
        @DataClass.Generated.Member
        public @NonNull Builder setTransliteration(@NonNull CharSequence value) {
            checkNotUsed();
            mBuilderFieldsSet |= 0x8;
            mTransliteration = value;
            return this;
        }

        /** Builds the instance. This builder should not be touched after calling this! */
        public @NonNull TranslationResponseValue build() {
            checkNotUsed();
            mBuilderFieldsSet |= 0x10; // Mark builder used

            if ((mBuilderFieldsSet & 0x2) == 0) {
                mText = defaultText();
            }
            if ((mBuilderFieldsSet & 0x4) == 0) {
                mExtras = defaultExtras();
            }
            if ((mBuilderFieldsSet & 0x8) == 0) {
                mTransliteration = defaultTransliteration();
            }
            TranslationResponseValue o = new TranslationResponseValue(
                    mStatusCode,
                    mText,
                    mExtras,
                    mTransliteration);
            return o;
        }

        private void checkNotUsed() {
            if ((mBuilderFieldsSet & 0x10) != 0) {
                throw new IllegalStateException(
                        "This Builder should not be reused. Use a new Builder instance instead");
            }
        }
    }

    @DataClass.Generated(
            time = 1631057245846L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/view/translation/TranslationResponseValue.java",
            inputSignatures = "public static final  int STATUS_SUCCESS\npublic static final  int STATUS_ERROR\npublic static final  java.lang.String EXTRA_DEFINITIONS\nprivate final @android.view.translation.TranslationResponseValue.Status int mStatusCode\nprivate final @android.annotation.Nullable java.lang.CharSequence mText\nprivate final @android.annotation.NonNull android.os.Bundle mExtras\nprivate final @android.annotation.Nullable java.lang.CharSequence mTransliteration\npublic static @android.annotation.NonNull android.view.translation.TranslationResponseValue forError()\nprivate static  java.lang.CharSequence defaultText()\nprivate static  android.os.Bundle defaultExtras()\nprivate  boolean extrasEquals(android.os.Bundle)\nprivate static  java.lang.CharSequence defaultTransliteration()\nclass TranslationResponseValue extends java.lang.Object implements [android.os.Parcelable]\nclass BaseBuilder extends java.lang.Object implements []\n@com.android.internal.util.DataClass(genBuilder=true, genToString=true, genEqualsHashCode=true, genHiddenConstDefs=true)\nclass BaseBuilder extends java.lang.Object implements []")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
