#include <math.h>
#include <QtWidgets>
#include "model.h"
#include "scaladialog.h"

ScalaDialog::ScalaDialog(Model *p_model, QWidget *parent, Qt::WindowFlags f) : QWidget(parent, f) {

  model = p_model;
  QVBoxLayout *mainLayout = new QVBoxLayout;
  setLayout(mainLayout);

  QHBoxLayout *scalaFileLayout = new QHBoxLayout;
  QWidget *scalaFileContainer = new QWidget;
  scalaFileContainer->setLayout(scalaFileLayout);
  scalaFileLayout->addWidget(new QLabel("Scala File: "));
  scalaFileNameLabel = new QLabel("<NO FILE LOADED>");
  scalaFileLayout->addWidget(scalaFileNameLabel);
  mainLayout->addWidget(scalaFileContainer);

  QHBoxLayout *scalaNameLayout = new QHBoxLayout;
  QWidget *scalaNameContainer = new QWidget;
  scalaNameContainer->setLayout(scalaNameLayout);
  scalaNameLayout->addWidget(new QLabel("Scala Name: "));
  scalaNameEdit = new QLineEdit();
  scalaNameEdit->setReadOnly(true);
  scalaNameEdit->setMaximumWidth(400);
  scalaNameEdit->setText("NONE");
  scalaNameLayout->addWidget(scalaNameEdit);
  mainLayout->addWidget(scalaNameContainer);

  QHBoxLayout *scalaLenLayout = new QHBoxLayout;
  QWidget *scalaLenContainer = new QWidget;
  scalaLenContainer->setLayout(scalaLenLayout);
  scalaLenLayout->addWidget(new QLabel("Scala Length: "));
  scalaLenLabel = new QLabel("0");
  scalaLenLayout->addWidget(scalaLenLabel);
  scalaLenLayout->addWidget(new QLabel("Octave Period: "));
  octavePeriodLabel = new QLabel("no");
  scalaLenLayout->addWidget(octavePeriodLabel);
  mainLayout->addWidget(scalaLenContainer);

  QHBoxLayout *scalaLayout = new QHBoxLayout;
  QWidget *scalaContainer = new QWidget;
  scalaContainer->setLayout(scalaLayout);
  scalaModeCheck = new QCheckBox("Activate Scala Mode");
  scalaLayout->addWidget(scalaModeCheck);
  scalaModeCheck->setChecked(model->getScalaMode());
  QObject::connect(scalaModeCheck, SIGNAL(toggled(bool)), this, SLOT(setScalaMode(bool)));
  loadScalaButton = new QPushButton("Load Scala");
  scalaLayout->addWidget(loadScalaButton);
  QObject::connect(loadScalaButton, SIGNAL(clicked()), this, SLOT(loadScala()));
  //mainLayout->addSpacing(40);
  mainLayout->addWidget(scalaContainer);

  QHBoxLayout *paramLayout = new QHBoxLayout;
  QWidget *paramContainer = new QWidget;
  paramContainer->setLayout(paramLayout);
  paramLayout->addWidget(new QLabel("Base Frequency: "));
  baseFreqEdit = new QLineEdit;
  baseFreqEdit->setFixedWidth(120);
  baseFreqEdit->setText(QString::number(model->getScalaBaseFreq()));
  paramLayout->addWidget(baseFreqEdit);
  paramLayout->addSpacing(40);
  QObject::connect(baseFreqEdit, SIGNAL(returnPressed()), this, SLOT(updateScala()));
  paramContainer->setLayout(paramLayout);
  paramLayout->addWidget(new QLabel("Base Note: "));
  baseNoteEdit = new QLineEdit;
  baseNoteEdit->setFixedWidth(40);
  baseNoteEdit->setText(QString::number(model->getScalaBaseNote()));
  paramLayout->addWidget(baseNoteEdit);
  QObject::connect(baseNoteEdit, SIGNAL(returnPressed()), this, SLOT(updateScala()));
  //mainLayout->addSpacing(40);
  mainLayout->addWidget(paramContainer);

  scalaOutput = new QTextEdit;
  scalaOutput->setReadOnly(true);
  mainLayout->addWidget(scalaOutput);

  QHBoxLayout *buttonLayout = new QHBoxLayout;
  QWidget *buttonContainer = new QWidget;
  buttonContainer->setLayout(buttonLayout);
  closeButton = new QPushButton("Close");
  closeButton->setFixedWidth(250);
  QObject::connect(closeButton, SIGNAL(clicked()), this, SLOT(hide()));
  buttonLayout->addWidget(closeButton);
  //mainLayout->addSpacing(40);
  mainLayout->addWidget(buttonContainer);
}

ScalaDialog::~ScalaDialog() {

}

void ScalaDialog::setScala(QString p_scalaName) {

  scalaName = p_scalaName;
}

QString ScalaDialog::getScala() {

  return(scalaName);
}

void ScalaDialog::updateScalaOutput() {

  int i;
  QString qs;

  scalaOutput->clear();
  for (i = 0; i < 128; i++) {
    qs = QString::number(i) + ":" + "  Note: " + QString::number(model->getScalaNote(i)) + "  Freq: " + QString::number(model->getScalaFreq(i));
    scalaOutput->insertPlainText(qs + "\n");
  }
}

void ScalaDialog::updateScala() {

  int scalaLen;
  bool octavePeriod;

  if (baseNoteEdit == sender()) {
    baseFreqEdit->setText(QString::number(freqConst * exp(log2div12 * (double)baseNoteEdit->text().toInt())));
  }
  model->setScalaBaseFreq(baseFreqEdit->text().toDouble());
  model->setScalaBaseNote(baseNoteEdit->text().toInt());
  model->updateScala(scalaLen, octavePeriod);
  scalaLenLabel->setText(QString::number(scalaLen));
  octavePeriodLabel->setText(octavePeriod ? "yes" : "no");
  updateScalaOutput();
}

void ScalaDialog::setScalaMode(bool on) {

  int scalaLen;
  bool octavePeriod;

  model->setScalaMode(on);
  model->updateScala(scalaLen, octavePeriod);
  scalaLenLabel->setText(QString::number(scalaLen));
  octavePeriodLabel->setText(octavePeriod ? "yes" : "no");
  updateScalaOutput();
}

void ScalaDialog::loadScala() {

  QString fn, l, s1, s2, s3;
  QFile f;
  int pos;
  double value;
  bool isCent;
  int scalaLen;
  bool octavePeriod;

  fn = QFileDialog::getOpenFileName(this);
  if (fn.length()) {
    f.setFileName(fn);
  } else {
    return;
  }
  if (!f.open(QIODevice::ReadOnly | QIODevice::Text)) {
    return;
  }
  scalaFileNameLabel->setText(fn);

  model->clearScalaData();
  QTextStream t(&f);

  while (!t.atEnd()) {
    l = t.readLine().simplified();
    if (!l.contains("!"))
      break;
  }
  scalaNameEdit->setText(l);
  scalaNameEdit->setCursorPosition(0);
  while (!t.atEnd()) {
    l = t.readLine().simplified();
    if (!l.contains("!"))
      break;
  }
  while(!t.atEnd()) {
    l = t.readLine().simplified();
    if (l.contains("!")) {
      continue;
    }
    if (l.contains(".")) {
      s1 = ((pos = l.indexOf(" ")) > 0) ? l.left(pos) : l;
      isCent = true;
      value = s1.toFloat();
      model->appendScalaItem(value, isCent);
    } else {
      isCent = false;
      if (l.contains("/")) {
        s1 = l.left(l.indexOf("/"));
        s2 = l.mid(l.indexOf("/") + 1);
        s3 = ((pos = s2.indexOf(" ")) > 0) ? s2.left(pos) : s2;
        value = s1.toFloat() / s3.toFloat();
      } else {
        s1 = ((pos = l.indexOf(" ")) > 0) ? l.left(pos) : l;
        value = s1.toFloat();
      }
      model->appendScalaItem(value, isCent);
    }
  }
  f.close();
  scalaModeCheck->setChecked(true);
  model->setScalaMode(true);
  model->updateScala(scalaLen, octavePeriod);
  scalaLenLabel->setText(QString::number(scalaLen));
  octavePeriodLabel->setText(octavePeriod ? "yes" : "no");
  updateScalaOutput();
}
