#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <QtWidgets/QWidget>
#include <QtWidgets/QHBoxLayout>
#include <QtWidgets/QVBoxLayout>
#include <QtGui/QPainter>
#include <QtGui/QMouseEvent>
#include <QtWidgets>
#include <QtCore/QString>
#include <QtCore/QTime>
#include <QCursor>
//#include <qDebug>
#include "model.h"
#include "harmonics.h"

Harmonics::Harmonics(Model *p_model, QString p_unitLabel, int p_index, QWidget *parent, int w, int h) : QWidget(parent) {

  int i1, o, n;

  model = p_model;
  unitLabel = p_unitLabel;
  index = p_index;
  mousePos = -1;
  mouseVal = 0;
  setMinimumSize(w, h);
  leftBorder = 20;
  rightBorder = 10;
  topBorder = 20;
  bottomBorder = 60;
  showPolygonFlag = true;
  for (i1 = 0; i1 < MAX_HARMONIC_GROUPS; i1++) {
    o = model->getHarmonicSplit(i1);
    n = model->getHarmonicSplit(i1 + 1) - o;
    scaleX = 10.0;
    upperControlPoints[i1][0] = QPoint((int)(9.0 * scaleX) - 10,  0);
    upperControlPoints[i1][1] = QPoint((int)(12.0 * scaleX) - 10,  0);
    upperControlPoints[i1][2] = QPoint((int)(16.0 * scaleX) - 10,  0);
    upperControlPoints[i1][3] = QPoint((int)(24.0 * scaleX) - 10,  0);
    upperControlPoints[i1][4] = QPoint((int)(32.0 * scaleX) - 10,  0);
    upperControlPoints[i1][5] = QPoint((int)(48.0 * scaleX) - 10,  0);
    upperControlPoints[i1][6] = QPoint((int)(64.0 * scaleX) - 10,  0);
    upperControlPoints[i1][7] = QPoint(0,  0);
    upperControlPoints[i1][8] = QPoint(0,  0);
  }
  scaleX = (double)(width() - leftBorder - rightBorder) / (double)n;
  scaleY = (height() - topBorder - bottomBorder) / model->getMax();
  setPalette(QPalette(QColor(200, 200, 200), QColor(20, 0, 70)));
  setAutoFillBackground(true);
  setMouseTracking(true);
  activeControlPoint = -1;
}

Harmonics::~Harmonics() {
  
}

QPoint Harmonics::getControlPoint(int index1, int index2) {

  return(upperControlPoints[index1][index2]);
}

void Harmonics::setControlPointY(int index1, int index2, int value) {

  upperControlPoints[index1][index2].setY(value);
}

void Harmonics::updateUpperHarmonics() {

  int i, e, o, n, vStart, vEnd, val, controlPointIndex;
  double dx, dy;
  
  model->getEditRange(vStart, vEnd);
  e = model->getEditGroup();
  o = model->getHarmonicSplit(model->getEditGroup());
  n = model->getHarmonicSplit(model->getEditGroup() + 1) - o;
  controlPointIndex = 0;
  for (i = 8; i < n; i++) {
    dx = (double)(upperControlPoints[e][controlPointIndex + 1].x() - upperControlPoints[e][controlPointIndex].x()) / 10.0;
    dy = (double)(upperControlPoints[e][controlPointIndex + 1].y() - upperControlPoints[e][controlPointIndex].y()) / 10.0;
    val = (double)upperControlPoints[e][controlPointIndex].y() / 10.0 + dy/dx * ((double)i - (double)upperControlPoints[e][controlPointIndex].x() / 10.0);
    for (int vl = vStart; vl <= vEnd; vl++) {
      model->getVoiceData(vl)->setHarmonic(index, i + o, val);
    }  
    if ((controlPointIndex < NUM_CONTROLPOINTS - 1) && ((double)(i + 1) > (double)upperControlPoints[e][controlPointIndex + 1].x()/10.0)) {
      controlPointIndex++;
    }
  }
}

void Harmonics::paintEvent(QPaintEvent *) {

  QPainter p;
  QString qs;
  int i, o, n, e; // TODO read data from VoiceData throughout this method
  int vStart, vEnd;
  
//  qDebug() << "M1";
  model->getEditRange(vStart, vEnd);
  e = model->getEditGroup();
  o = model->getHarmonicSplit(model->getEditGroup());
  n = model->getHarmonicSplit(model->getEditGroup() + 1) - o;
  scaleX = (double)(width() - leftBorder - rightBorder) / (double)n;
  scaleY = (height() - topBorder - bottomBorder) / model->getMax();
  p.begin(this);
  p.setPen(QPen(QBrush(QColor(175, 131, 0)), 10));
  for (i=0; i < n; i++) {  
    p.drawLine(leftBorder + (int)((double)i*scaleX), 
               height() - bottomBorder - scaleY * model->getVoiceData(vStart)->getHarmonic(index, i + o), 
               leftBorder + (int)((double)i*scaleX), 
               height() - bottomBorder); 
  }
  p.setPen(QPen(QBrush(QColor(194, 146, 0)), 8));
  for (i=0; i < n; i++) {  
    p.drawLine(leftBorder + (int)((double)i*scaleX), 
               height() - bottomBorder - scaleY * model->getVoiceData(vStart)->getHarmonic(index, i + o), 
               leftBorder + (int)((double)i*scaleX), 
               height() - bottomBorder); 
  }
  p.setPen(QPen(QBrush(QColor(216, 162, 0)), 6));
  for (i=0; i < n; i++) {  
    p.drawLine(leftBorder + (int)((double)i*scaleX), 
               height() - bottomBorder - scaleY * model->getVoiceData(vStart)->getHarmonic(index, i + o), 
               leftBorder + (int)((double)i*scaleX), 
               height() - bottomBorder); 
  }
  p.setPen(QPen(QBrush(QColor(240, 180, 0)), 4));
  for (i=0; i < n; i++) {  
    p.drawLine(leftBorder + (int)((double)i*scaleX), 
               height() - bottomBorder - scaleY * model->getVoiceData(vStart)->getHarmonic(index, i + o), 
               leftBorder + (int)((double)i*scaleX), 
               height() - bottomBorder); 
  }
  if ((mousePos > -1) && (mousePos < n) && (mouseVal >= 0) && (mouseVal <= model->getMax())) {
    p.setFont(QFont("Helvetica", 14)); 
    qs = "Harmonic " + QString::number(mousePos + 1) + " :  "  
       + QString::number(model->getVoiceData(vStart)->getHarmonic(index, mousePos + o), 'f', 1) + unitLabel 
       + " <-- " + QString::number(mouseVal, 'f', 1) + unitLabel;
    p.drawText(leftBorder, height() - bottomBorder * 0.3, qs);
  }
  if ((n > 8) && showPolygonFlag) {
    int num_controls = NUM_CONTROLPOINTS;
    if (n < 64) num_controls -= 2;
    if (n < 32) num_controls = 0;
    for (i=0; i < num_controls; i++) {
      if (i == activeControlPoint) {
        p.setPen(QPen(QBrush(QColor(220, 0, 0)), 4));
      } else {
        p.setPen(QPen(QBrush(QColor(40, 180, 40)), 4));
      }  
      p.drawEllipse(leftBorder + (int)((double)upperControlPoints[e][i].x() / 10.0 * scaleX) - 5, height() - bottomBorder - (int)((double)upperControlPoints[e][i].y() / 10.0 * scaleY) - 5, 10, 10);
      p.setPen(QPen(QBrush(QColor(40, 180, 40)), 4));
      if (i < num_controls - 1) {
        p.drawLine(leftBorder + (int)((double)upperControlPoints[e][i].x() / 10.0 * scaleX), height() - bottomBorder - (int)((double)upperControlPoints[e][i].y() / 10.0 * scaleY), 
                   leftBorder + (int)((double)upperControlPoints[e][i+1].x() / 10.0 * scaleX), height() - bottomBorder - (int)((double)upperControlPoints[e][i+1].y() / 10.0 * scaleY));
      }
    }
  }  
//  qDebug() << "M2";
  p.end();
}

void Harmonics::viewportResizeEvent(QResizeEvent *ev) {

  scaleX = (double)(width() - leftBorder - rightBorder) / (double)model->getNumHarmonics();
  scaleY = (height() - topBorder - bottomBorder) / model->getMax();
  update(0, 0, width(), height());
}

void Harmonics::mousePressEvent (QMouseEvent *ev) {

  int i, o, n, e;
  double val;
  int vStart, vEnd, x, y;
  
  e = model->getEditGroup();
  o = model->getHarmonicSplit(model->getEditGroup());
  n = model->getHarmonicSplit(model->getEditGroup() + 1) - o;
  if ((n > 8) && showPolygonFlag) {
    for (i = 0; i < NUM_CONTROLPOINTS; i++) {
      x = leftBorder + (int)((double)upperControlPoints[e][i].x() / 10.0 * scaleX); 
      y = height() - bottomBorder - (int)((double)upperControlPoints[e][i].y() / 10.0 * scaleY);
      if ((abs(x - ev->x()) <= 5) && (abs(y - ev->y()) <= 5)) {
        activeControlPoint = i;
        break;
      }
    }
  }  
  if (activeControlPoint < 0) {
    model->getEditRange(vStart, vEnd);
    i = round((ev->x() - leftBorder) / scaleX);
    if ((i >= 0) && (i < n)) { 
      val = (double)(height() - bottomBorder - ev->y()) / scaleY;
      for (int vl = vStart; vl <= vEnd; vl++) {
        model->getVoiceData(vl)->setHarmonic(index, i + o, val);  
      }
      update(0, 0, width(), height());
    }
  } else {
    update(0, 0, width(), height());
  }  
}

void Harmonics::mouseReleaseEvent (QMouseEvent *ev) {

  if (activeControlPoint >= 0) {
    activeControlPoint = -1;
    updateUpperHarmonics();
  }  
  update(0, 0, width(), height());
}

void Harmonics::mouseMoveEvent (QMouseEvent *ev) {

  int o, n, e;
  int vStart, vEnd;

  e = model->getEditGroup();
  model->getEditRange(vStart, vEnd);
  mousePos = round((ev->x() - leftBorder) / scaleX);
  o = model->getHarmonicSplit(model->getEditGroup());
  n = model->getHarmonicSplit(model->getEditGroup() + 1) - o;
  if ((mousePos >= 0) && (mousePos < 8)) {
    mouseVal = (double)(height() - bottomBorder - ev->y()) / scaleY;
    if (ev->buttons() == Qt::LeftButton) {
      for (int vl = vStart; vl <= vEnd; vl++) {
        model->getVoiceData(vl)->setHarmonic(index, mousePos + o, mouseVal);  
      }
      update(0, 0, width(), height());
    } else { 
      update(0, height() - bottomBorder, width(), height());
    } 
  } else {
    if ((mousePos >= 0) && (mousePos < n)) {
      mouseVal = (double)(height() - bottomBorder - ev->y()) / scaleY;
      if (ev->buttons() == Qt::LeftButton) {
        if (activeControlPoint < 0) {
          for (int vl = vStart; vl <= vEnd; vl++) {
            model->getVoiceData(vl)->setHarmonic(index, mousePos + o, mouseVal);  
          }
        } else {
          if (mouseVal < 0) mouseVal = 0;
          if (mouseVal > model->getMax()) mouseVal = model->getMax();
          if ((activeControlPoint >= 0) && (activeControlPoint < NUM_CONTROLPOINTS)) {
            upperControlPoints[e][activeControlPoint].setY(10.0 * mouseVal);
          } 
          updateUpperHarmonics();
        }    
        update(0, 0, width(), height());
      } else { 
        update(0, height() - bottomBorder, width(), height());
      } 
    }
  }  
}
                  
void Harmonics::setHarmonic(int objectType, int objectIndex, int voiceIndex, int value) {

  int vStart, vEnd;
  int e, o, n, index1, index2, i, m, controls;

  model->getEditRange(vStart, vEnd);
  e = objectType - 15;
  o = model->getHarmonicSplit(e);
  n = model->getHarmonicSplit(e + 1) - o;
  controls = 8 + NUM_CONTROLPOINTS + 2;
  if (n < 64) controls -= 2;
  if (n < 32) controls = 8 + 2;
  index1 = objectIndex / controls;
  index2 = objectIndex % controls;
  if (index2 < 8) {
    for (int vl = vStart; vl <= vEnd; vl++) {
      model->getVoiceData(vl)->setHarmonic(index1, index2 + o, (int)round((double)value * 100.0 / 127.0));
    }

  } else {
    m = index2 - 8;
    if ((m >= 0) && (m  < NUM_CONTROLPOINTS + 2)) {
      upperControlPoints[e][m].setY((int)round((double)value * 1000.0 / 127.0));
    }
    updateUpperHarmonics();
  }
//  qDebug() << index1 << index2 << value;
  update(0, 0, width(), height());
//  qDebug() << width() << height();
}

int Harmonics::getHarmonic(int objectType, int objectIndex, int voiceIndex) {

  int e, o, n, index1, index2, m, controls;
  int returnVal;

  e = objectType - 15;
  o = model->getHarmonicSplit(e);
  n = model->getHarmonicSplit(e + 1) - o;
  controls = 8 + NUM_CONTROLPOINTS + 2;
  if (n < 64) controls -= 2;
  if (n < 32) controls = 8 + 2;
  index1 = objectIndex / controls;
  index2 = objectIndex % controls;
  returnVal = 0;
  if (index2 < 8) {
    returnVal = (int)round((double)model->getVoiceData(voiceIndex)->getHarmonic(index1, index2 + o) * 127.0 / 100.0);
  } else {
    m = index2 - 8;
    if ((m >= 0) && (m  < NUM_CONTROLPOINTS + 2)) {
      returnVal = (int)round((double)upperControlPoints[e][m].y() * 127.0 / 1000.0);
    }
  }
  return(returnVal);
}

void Harmonics::setShowPolygon(bool on) {

    showPolygonFlag = on;
    update(0, 0, width(), height());
}

void Harmonics::randomizeHarmonics(int lowerRandomLevel, int upperRandomLevel, bool addRandomFlag) {

    int vStart, vEnd;
    int e, o, n, i1;
    double h;

    QTime time = QTime::currentTime();
    qsrand((uint)time.msec());
    model->getEditRange(vStart, vEnd);
    e = model->getEditGroup();
    o = model->getHarmonicSplit(e);
    n = model->getHarmonicSplit(e + 1) - o;
    for (int vl = vStart; vl <= vEnd; vl++) {
      if (addRandomFlag) {
        if (lowerRandomLevel) {
          for (i1 = 0; i1 < 8; i1++) {
            h = model->getVoiceData(vl)->getHarmonic(index, o + i1);
            model->getVoiceData(vl)->setHarmonic(index, o + i1, h + qrand() % lowerRandomLevel);
          }
        }
        if (upperRandomLevel) {
          for (i1 = 8; i1 < n; i1++) {
            h = model->getVoiceData(vl)->getHarmonic(index, o + i1);
            model->getVoiceData(vl)->setHarmonic(index, o + i1, h + qrand() % upperRandomLevel);
          }
        }
      } else {
        if (lowerRandomLevel) {
          for (i1 = 0; i1 < 8; i1++) {
            model->getVoiceData(vl)->setHarmonic(index, o + i1, qrand() % lowerRandomLevel);
          }
        }
        if (upperRandomLevel) {
          for (i1 = 8; i1 < n; i1++) {
            model->getVoiceData(vl)->setHarmonic(index, o + i1, qrand() % upperRandomLevel);
          }
        }
      }
    }
    update(0, 0, width(), height());
}
