;; Cuong Chau <ckc8687@gmail.com>

;; May 2022

;; Direct reasoning about complex functions is often unachievable in most
;; existing verification tools.  Decomposition is a common technique for
;; overcoming this situation.  For instance, the correctness of a function can
;; be validated throughout a sequence of steps that reason about that
;; function's local variables.  This book implements tools that assist such
;; approach.  In particular, the presented tools extract terms that compute
;; values of local variables bound by LET/LET*/B*/MV-LET/MV?-LET bindings,
;; which are declared in function definitions.  Two tools are provided:
;; CONST-FNS-GEN and LOOP-FNS-GEN.  Depending on whether the definition of the
;; input function is recursive, the appropriate tool may be applied:
;; CONST-FNS-GEN for functions with non-recursive definitions, and LOOP-FNS-GEN
;; for functions with recursive definitions.  The application of LOOP-FNS-GEN
;; is fairly restrictive; it requires that one of the arguments of the
;; recursive input function is the counter that increases/decreases by one at
;; each recursive call, and the termination of the input function is based on
;; this argument.

;; CONST-FNS-GEN takes a function name and produces a list constant,
;; i.e. zero-arity, function definitions that each computes the value of the
;; corresponding local variable declared in a LET/LET*/B*/MV-LET/MV?-LET
;; binding of the input function.  A top-level constant function corresponding
;; to the input function is also generated.  CONST-FNS-GEN also outputs a lemma
;; confirming the equivalence between the input function and the generated
;; top-level constant function.

;; Before applying CONST-FNS-GEN, the user needs to introduce constant
;; functions representing the arguments of the input function.  This can be
;; done via an ENCAPSULATE event.  For example, suppose an input function has
;; two arguments, we can introduce two constant functions representing these
;; two arguments as follows.

;; (encapsulate
;;   (((a) => *)
;;    ((b) => *))

;;   (local (defun a () <a-value>))
;;   (local (defun b () <b-value>))

;;   (defthm input-constraints-lemma
;;     <constraints-on-the-arguments>))

;; LOOP-FNS-GEN takes a function name and produces a set of mutually recursive
;; functions.  Each generated function computes the value of the corresponding
;; local variable declared in a LET/LET*/B*/MV-LET/MV?-LET binding of the input
;; function.  Unlike CONST-FNS-GEN, neither a top-level function nor an
;; equivalence lemma is generated by LOOP-FNS-GEN.  The user may verify
;; correctness of the generated mutually recursive functions on their own.

(in-package "ACL2")

(include-book "std/strings/decimal" :dir :system)
(include-book "std/strings/isubstrp" :dir :system)

;; ======================================================================

(defmacro defundd (&rest def)
  (declare (xargs :guard (and (true-listp def)
                              (symbolp (car def))
                              (symbol-listp (cadr def)))))
  `(progn
     (defun ,@def)
     (in-theory (disable ,(car def) (,(car def))))))

(defmacro strings-to-symbol (pkg-name &rest strs)
  `(intern$ (concatenate 'string ,@strs)
            ,pkg-name))

(defund assocs-vals (keys alist)
  (if (atom keys)
      nil
    (append (cdr (assoc-equal (car keys) alist))
            (assocs-vals (cdr keys) alist))))

(defund max-nats (x)
  (declare (xargs :guard (nat-listp x)))
  (if (atom x)
      0
    (max (car x)
         (max-nats (cdr x)))))

(defund remove-all (x y)
  ;; Remove all elements in 'x' from 'y'
  (declare (xargs :guard (true-listp y)))
  (if (atom x)
      y
    (remove-all (cdr x)
                (remove-equal (car x) y))))

(defund remove-assocs-equal (keys alist)
  ;; Remove all pairs with the given 'keys' from 'alist'
  (declare (xargs :guard (and (true-listp keys)
                              (alistp alist))))
  (cond ((endp alist) nil)
        ((member-equal (car (car alist)) keys)
         (remove-assocs-equal keys (cdr alist)))
        (t (cons (car alist)
                 (remove-assocs-equal keys (cdr alist))))))

(defnd replace-all (x y tree)
  ;; Replace all occurrences of 'x' in 'tree' with 'y'.
  (if (atom tree)
      tree
    (if (equal x (car tree))
        (cons y (replace-all x y (cdr tree)))
      (cons (replace-all x y (car tree))
            (replace-all x y (cdr tree))))))

(defund replace-all-except-bound-vars (x y tree)
  ;; Replace all occurrences of 'x' in 'tree' with 'y' except for the variables
  ;; bound by the LET/LET*/B* forms.
  (cond
   ((atom tree) tree)
   ((member-equal (car tree) '(let let* b*))
    (list* (car tree)
           (pairlis$
            (strip-cars (cadr tree))
            (replace-all-except-bound-vars x y (strip-cdrs (cadr tree))))
           (if (member-equal x (strip-cars (cadr tree)))
               (cddr tree)
             (replace-all-except-bound-vars x y (cddr tree)))))
   ((equal x (car tree))
    (cons y (replace-all-except-bound-vars x y (cdr tree))))
   (t (cons (replace-all-except-bound-vars x y (car tree))
            (replace-all-except-bound-vars x y (cdr tree))))))

(defund replace-list-all (xs ys tree)
  ;; For each symbol X in 'xs', replace all of its occurrences in 'tree' with
  ;; either:
  ;; 1. (Y) if the pair (X Y) appears in 'ys', or
  ;; 2. (X) otherwise
  (if (atom xs)
      tree
    (let ((pair-x-y (assoc-equal (car xs) ys)))
      (if pair-x-y
          (replace-list-all (cdr xs)
                            (remove1-equal pair-x-y ys)
                            (replace-all-except-bound-vars
                             (car xs) (cdr pair-x-y) tree))
        (replace-list-all (cdr xs)
                          ys
                          (replace-all-except-bound-vars
                           (car xs) (list (car xs)) tree))))))

(defund strip-cars-hack (x)
  ;; This function behaves like STRIP-CARS, except that it also collects atom
  ;; elements when walking through the list 'x'. For example,
  ;; (strip-cars-hack '((A) B (C))) would return '(A B C).
  (declare (xargs :guard (true-listp x)))
  (cond ((endp x) nil)
        (t (cons (if (atom (car x))
                     (car x)
                     (car (car x)))
                 (strip-cars-hack (cdr x))))))

(mutual-recursion
 (defund rename-in-term-of-mv-let (key old-key alist i n pkg-name)
   ;; Replace all occurrences of 'key' in the <term> of an MV-LET form with
   ;; 'old-key'
   (declare (xargs :guard (alistp alist)))
   (if (endp alist)
       nil
     (let ((pair (car alist)))
       (if (equal pair '(*** (e (e)))) ;; Reached the end of an MV-LET clique
           (cons pair
                 (rename-key-in-alist key (cdr alist) i n pkg-name))
         (cons (cons (car pair)
                     (replace-all-except-bound-vars
                      key old-key (cdr pair)))
               (rename-in-term-of-mv-let
                key old-key (cdr alist) i n pkg-name))))))

 (defund rename-key-in-alist (key alist i n pkg-name)
   ;; Rename 'key' in the binding list 'alist' if necessary

   ;; 'i' is the current index.

   ;; 'n' is the number of bindings of the given 'key'.
   (declare (xargs :mode :program))
   (cond ((endp alist) nil)
         ((or (<= n 1) (<= n i)) alist)
         (t (let* ((pair (car alist))
                   (val (cadr pair))
                   (old-key (if (posp i)
                                (strings-to-symbol
                                 pkg-name
                                 (symbol-name key)
                                 "-"
                                 (str::nat-to-dec-string (1- i)))
                              key))
                   (renamed-key (strings-to-symbol
                                 pkg-name
                                 (symbol-name key)
                                 "-"
                                 (str::nat-to-dec-string i))))
              (if (equal key (car pair))
                  (if (= i (1- n))
                      ;; Reach the last binding of 'key' in 'alist'
                      (cons
                       (cons key
                             (replace-all-except-bound-vars
                              key old-key (cdr pair)))
                       (if (and (consp val)
                                (equal (car val) 'mv-nth))
                           ;; Enter an MV-LET clique
                           (rename-in-term-of-mv-let key old-key
                                                     (cdr alist)
                                                     (1+ i) n
                                                     pkg-name)
                         (cdr alist)))
                    (cons
                     (cons renamed-key
                           (append
                            (replace-all-except-bound-vars
                             key old-key (cdr pair))
                            (list i))) ;; Add this extra index to the end of
                                       ;; this binding as an annotation of an
                                       ;; intermediate function
                     (if (and (consp val)
                              (equal (car val) 'mv-nth))
                         ;; Enter an MV-LET clique
                         (rename-in-term-of-mv-let key old-key
                                                   (cdr alist)
                                                   (1+ i) n
                                                   pkg-name)
                       (rename-key-in-alist key
                                            (cdr alist)
                                            (1+ i) n
                                            pkg-name))))
                (if (zp i)
                    ;; Haven't reached the first binding of 'key' yet.
                    ;; 'pair' remains unchanged.
                    (cons pair
                          (rename-key-in-alist key (cdr alist) i n pkg-name))
                  (cons
                   (cons (car pair)
                         (replace-all-except-bound-vars
                          key old-key (cdr pair)))
                   (rename-key-in-alist key (cdr alist) i n pkg-name))))))))
 )

(defund rename-keys-in-alist (distinct-keys keys alist pkg-name)
  ;; Rename 'distinct-keys' in the binding list 'alist' if necessary
  (declare (xargs :mode :program))
  (if (atom distinct-keys)
      alist
    (rename-keys-in-alist
     (cdr distinct-keys)
     keys
     (rename-key-in-alist (car distinct-keys)
                          alist
                          0
                          (count (car distinct-keys) keys)
                          pkg-name)
     pkg-name)))

(defund inter-names-extract (alist)
  ;; Extract all intermediate keys from 'alist'
  (declare (xargs :guard (symbol-alistp alist)))
  (if (endp alist)
      nil
    (if (or (< 2 (len (car alist)))
            (str::isubstrp "++" (symbol-name (caar alist))))
        (cons (caar alist)
              (inter-names-extract (cdr alist)))
      (inter-names-extract (cdr alist)))))

(defund collect-bound-vars (x)
  ;; Collect all bound variables appearing in term 'x'.  Note, the result can
  ;; contain duplicate variables.
  (cond ((atom x) nil)
        ((member-equal (car x) '(let let* b*))
         (union$ (strip-cars (cadr x))
                 (collect-bound-vars (car (last x)))))
        ((member-equal (car x) '(mv-let mv?-let))
         (union$ (cadr x)
                 (collect-bound-vars (caddr x))
                 (collect-bound-vars (car (last x)))))
        (t (union$ (collect-bound-vars (car x))
                   (collect-bound-vars (cdr x))))))

(defund rename-key-in-bindings (key alist i n pkg-name)
  ;; Rename 'key' in the binding list 'alist' under an IF/IF1 statement

  ;; 'i' is the current index.

  ;; 'n' is the number of bindings of the given 'key'.
  (cond ((endp alist) nil)
        ((<= n i)
         (let ((old-key (strings-to-symbol
                         pkg-name
                         (symbol-name key)
                         "--"
                         (str::nat-to-dec-string (1- i)))))
           (replace-all-except-bound-vars key old-key alist)))
        (t (let* ((pair (car alist))
                  (old-key (if (posp i)
                               (strings-to-symbol
                                pkg-name
                                (symbol-name key)
                                "--"
                                (str::nat-to-dec-string (1- i)))
                             key))
                  (renamed-key (strings-to-symbol
                                pkg-name
                                (symbol-name key)
                                "--"
                                (str::nat-to-dec-string i))))
             (if (equal key (car pair))
                 (cons
                  (cons renamed-key
                        (replace-all-except-bound-vars
                         key old-key (cdr pair)))
                  (rename-key-in-bindings
                   key (cdr alist) (1+ i) n pkg-name))
               (if (zp i)
                   ;; Haven't reached the first binding of 'key' yet.
                   ;; 'pair' remains unchanged.
                   (cons pair
                         (rename-key-in-bindings
                          key (cdr alist) i n pkg-name))
                 (cons
                  (cons (car pair)
                        (replace-all-except-bound-vars
                         key old-key (cdr pair)))
                  (rename-key-in-bindings
                   key (cdr alist) i n pkg-name))))))))

(defund rename-keys-in-bindings (distinct-keys keys alist pkg-name)
  ;; Rename 'distinct-keys' in the binding list 'alist' under an IF/IF1
  ;; statement
  (if (atom distinct-keys)
      alist
    (rename-keys-in-bindings
     (cdr distinct-keys)
     keys
     (rename-key-in-bindings (car distinct-keys)
                             alist
                             0
                             (count (car distinct-keys) keys)
                             pkg-name)
     pkg-name)))

(defund rename-keys-in-body (distinct-keys keys body pkg-name)
  ;; Rename 'distinct-keys' in 'body' under an IF/IF1 statement
  (if (atom distinct-keys)
      body
    (let* ((key (car distinct-keys))
           (renamed-key (strings-to-symbol
                         pkg-name
                         (symbol-name key)
                         "--"
                         (str::nat-to-dec-string (1- (count key keys))))))
      (rename-keys-in-body
       (cdr distinct-keys)
       keys
       (replace-all-except-bound-vars key renamed-key body)
       pkg-name))))

(defund rename-with-count (x replaced-vars mv-let-vars pkg-name)
  (cond ((null x) nil)
        ((atom x)
         (let ((n (count x mv-let-vars)))
           (if (or (not (symbolp x))
                   (zp n)
                   (not (member-equal x replaced-vars)))
               x
             (strings-to-symbol pkg-name
                                (symbol-name x)
                                "++"
                                (str::nat-to-dec-string n)))))
        (t (cons (rename-with-count
                  (car x) replaced-vars mv-let-vars pkg-name)
                 (rename-with-count
                  (cdr x) replaced-vars mv-let-vars pkg-name)))))

(mutual-recursion
 (defund rename-keys-in-if-statement (x mv-let-vars pkg-name)
   (cond ((atom x) x)
         ((member-equal (car x) '(let let* b*))
          (let* ((alist (cadr x))
                 (body (caddr x))
                 (vars (strip-cars alist))
                 (distinct-vars (remove-duplicates vars))
                 (renamed-alist (rename-keys-in-bindings
                                 distinct-vars vars alist pkg-name))
                 (renamed-body (rename-keys-in-body
                                distinct-vars vars body pkg-name)))
            (list (car x) renamed-alist renamed-body)))
         ((member-equal (car x) '(mv-let mv?-let))
          (rename-mv-let-vars x mv-let-vars pkg-name))
         (t x)))

 (defund rename-mv-let-vars (x mv-let-vars pkg-name)
   ;; Rename variables in the MV-LET forms appearing in term 'x'
   (declare (xargs :mode :program))
   (cond
    ((atom x) x)
    ((member-equal (car x) '(mv-let mv?-let))
     (let* ((vars (cadr x))
            (term (caddr x))
            (term (if (member-equal (car term)
                                    (list (intern$ "IF" "RTL")
                                          (intern$ "IF1" "RTL")))
                      (list (car term)
                            (cadr term)
                            (rename-keys-in-if-statement
                             (caddr term)
                             (append mv-let-vars vars vars)
                             pkg-name)
                            (cadddr term))
                    term))
            (body (cdddr x))
            (renamed-vars (rename-with-count vars vars mv-let-vars pkg-name))
            (renamed-body (rename-with-count body vars mv-let-vars pkg-name)))
       (cons (car x)
             (cons renamed-vars
                   (cons (rename-mv-let-vars
                          term
                          (append mv-let-vars vars)
                          pkg-name)
                         (rename-mv-let-vars renamed-body
                                             mv-let-vars
                                             pkg-name))))))
    (t (cons (rename-mv-let-vars (car x) mv-let-vars pkg-name)
             (rename-mv-let-vars (cdr x) mv-let-vars pkg-name)))))
 )

(defund shrink-alist (preserved-keys alist)
  ;; Remove all elements from the association list 'alist' except for the
  ;; keys in 'preserved-keys'
  (if (atom alist)
      nil
    (let* ((x (car alist)))
      (if (member-equal (car x) preserved-keys)
          (cons x (shrink-alist preserved-keys (cdr alist)))
        (shrink-alist preserved-keys (cdr alist))))))

(defund shrink-bindings (preserved-vars x)
  ;; Remove all variable bindings from term 'x' except for the variables in
  ;; 'preserved-vars'
  (cond ((atom x) x)
        ((member-equal (car x) '(let let* b*))
         (let* ((shrunk-bindingds (shrink-alist preserved-vars (cadr x)))
                (shrunk-body (shrink-bindings preserved-vars (cddr x))))
           (if shrunk-bindingds
               (list* (car x)
                      shrunk-bindingds
                      shrunk-body)
             (car (last shrunk-body)))))
        ((member-equal (car x) '(mv-let mv?-let))
         (shrink-bindings preserved-vars (car (last x))))
        (t (cons (shrink-bindings preserved-vars (car x))
                 (shrink-bindings preserved-vars (cdr x))))))

(defund mv-let-process (vars term alist i)
  ;; Extract variable bindings from an MV-LET form
  (if (atom vars)
      (append alist
              '((*** (e (e))))) ;; Add this annotation to the end of each MV-LET
                                ;; clique for later renaming
    (mv-let-process (cdr vars)
                    term
                    (append alist
                            (list (list (car vars)
                                        `(mv-nth ,i ,term))))
                    (1+ i))))

(defund bindings-extract (x preserved-vars alist)
  ;; Extract variable bindings occurring in term 'x'.  Argument 'alist' acts as
  ;; an association list of cumulative bindings.
  (cond ((atom x) alist)
        ((member-equal (car x) '(let let* b*))
         (bindings-extract (cddr x)
                           preserved-vars
                           (append alist
                                   (remove-assocs-equal preserved-vars
                                                        (cadr x)))))
        ((member-equal (car x) '(mv-let mv?-let))
         (let* ((vars (cadr x))
                (term (caddr x))
                (alist (bindings-extract term
                                         preserved-vars
                                         alist)))
           (bindings-extract
            (cdddr x)
            preserved-vars
            (mv-let-process vars
                            `(mv-list ,(len vars)
                                      ,(shrink-bindings preserved-vars term))
                            (append alist
                                    ;; The annotation of the beginning of an
                                    ;; MV-LET clique
                                    '((*** (b (b)))))
                            0))))
        (t (bindings-extract (cdr x)
                             preserved-vars
                             (bindings-extract (car x)
                                               preserved-vars
                                               alist)))))

(defund const-p (x consts)
  ;; Detect a constant term
  (cond ((null x) t)
        ((atom x) (acl2-numberp x))
        ((atom (car x)) (const-p (cdr x) consts))
        ((and (= (len (car x)) 1)
              (symbolp (caar x)))
         (and (member-equal (caar x) consts)
              (const-p (cdr x) consts)))
        (t (and (const-p (car x) consts)
                (const-p (cdr x) consts)))))

(defund elim-bindings-extract (alist const-bindings unary-bindings)
  ;; Extract constant and unary bindings
  (let* ((pair (car alist))
         (consts (strip-cars const-bindings)))
    (cond ((atom alist)
           (append const-bindings unary-bindings))
          ;; Extract a constant binding
          ((const-p (cdr pair) consts)
           (let ((updated-pair
                  (cons (car pair)
                        (car (replace-list-all
                              (pairlis$ consts nil)
                              (pairlis$ (pairlis$ consts nil)
                                        (strip-cdrs const-bindings))
                              (cdr pair))))))
             (elim-bindings-extract
              (cdr alist)
              (append const-bindings
                      (list updated-pair))
              unary-bindings)))
          ;; Extract a unary binding
          ((and (= (len (cadr pair)) 1)
                (symbolp (caadr pair)))
           (let* ((subst (cdr (assoc-equal (caadr pair) unary-bindings)))
                  (updated-pair
                   (cons (car pair)
                         (if subst subst (cadr pair)))))
             (elim-bindings-extract
              (cdr alist)
              const-bindings
              (append unary-bindings
                      (list updated-pair)))))
          (t (elim-bindings-extract (cdr alist)
                                    const-bindings
                                    unary-bindings)))))

(defund rewrite-mv-nth-form (mv-nth-form)
  (let* ((n (nth 1 mv-nth-form))
         (mv-list-form (nth 2 mv-nth-form))
         (mv-form? (nth 2 mv-list-form)))
    (if (equal (car mv-form?) 'mv)
        (nth n (cdr mv-form?))
      mv-nth-form)))

(defund simplify-mv-nth-forms (lst)
  (declare (xargs :mode :program))
  (cond
   ((atom lst) lst)
   ((atom (car lst))
    (cons (car lst)
          (simplify-mv-nth-forms (cdr lst))))
   ((equal (caar lst) 'mv-nth)
    (let* ((x (car lst))
           (n? (cadr x))
           (mv-list-form? (caddr x)))
      (if (equal (car mv-list-form?) 'mv-list)
          (let ((if1/mv-form? (caddr mv-list-form?))
                (rtl-if1 (intern$ "IF1" "RTL")))
            (cond
             ;; Simplify an MV form
             ((equal (car if1/mv-form?) 'mv)
              (cons (nth n? (cdr if1/mv-form?))
                    (simplify-mv-nth-forms (cdr lst))))
             ;; Simplify a LET/LET*/B* binding form
             ((member-equal (car if1/mv-form?) '(let let* b*))
              (cons (append (butlast if1/mv-form? 1)
                            (simplify-mv-nth-forms
                             (list (rewrite-mv-nth-form
                                    `(mv-nth
                                      ,n?
                                      (mv-list ,(cadr mv-list-form?)
                                               ,@(last if1/mv-form?)))))))
                    (simplify-mv-nth-forms (cdr lst))))
             ;; Simplify an IF1 form
             ((equal (car if1/mv-form?) rtl-if1)
              (cons `(,rtl-if1 ,(cadr if1/mv-form?)
                          ,@(simplify-mv-nth-forms
                             (list (rewrite-mv-nth-form
                                    `(mv-nth
                                      ,n?
                                      (mv-list ,(cadr mv-list-form?)
                                               ,(caddr if1/mv-form?))))))
                          ,@(simplify-mv-nth-forms
                             (list (rewrite-mv-nth-form
                                    `(mv-nth
                                      ,n?
                                      (mv-list ,(cadr mv-list-form?)
                                               ,(cadddr if1/mv-form?)))))))
                    (simplify-mv-nth-forms (cdr lst))))
             (t (cons x (simplify-mv-nth-forms (cdr lst))))))
        (cons x (simplify-mv-nth-forms (cdr lst))))))
   (t (cons (car lst)
            (simplify-mv-nth-forms (cdr lst))))))

(defund symbols-extract (term)
  ;; Extract all symbols from 'term'
  (cond ((null term) nil)
        ((atom term)
         (if (symbolp term) (list term) nil))
        (t (union$ (symbols-extract (car term))
                   (symbols-extract (cdr term))))))

(defund used-syms-extract (term alist acc)
  ;; Extract all keys in 'alist' that are used in 'term' recursively
  (declare (xargs :mode :program))
  (if (atom term)
      acc
    (let* ((keys (strip-cars alist))
           (syms (cons '*** ;; LOOP-FNS-GEN needs this symbol.
                       (symbols-extract term)))
           (common-syms (intersection$ keys syms)))
      (used-syms-extract (assocs-vals common-syms alist)
                         (remove-assocs-equal common-syms alist)
                         (append acc common-syms)))))

(defund is-contained (x term)
  (declare (xargs :guard t))
  (if (atom term)
      nil
    (or (equal x (car term))
        (is-contained x (car term))
        (is-contained x (cdr term)))))

(defund term-subst (vars term alist)
  (if (atom vars)
      term
    (if (is-contained (car vars) term)
        (term-subst (cdr vars)
                    (replace-all-except-bound-vars
                     (car vars)
                     (cdr (assoc-equal (car vars) alist))
                     term)
                    alist)
      (term-subst (cdr vars) term alist))))

(defund terms-substs (vars alist vars-acc alist-acc)
  (if (atom vars)
      alist-acc
    (let* ((x (car vars))
           (new-term (term-subst vars-acc
                                 (cadr (assoc-equal x alist))
                                 alist-acc)))
      (terms-substs (cdr vars)
                    alist
                    (append vars-acc (list (list x)))
                    (append alist-acc
                            (list (cons (list x) new-term)))))))

(defund mem-count (x terms)
  (declare (xargs :guard t))
  (if (atom terms)
      0
    (+ (if (is-contained x (car terms))
           1
         0)
       (mem-count x (cdr terms)))))

(defund single-used-vars-extract (vars exps acc)
  (declare (xargs :guard (and (symbol-listp vars)
                              (true-listp acc))))
  (if (atom vars)
      acc
    (single-used-vars-extract
     (cdr vars)
     exps
     (if (and (= (mem-count (car vars) exps) 1)
              (or (str::isubstrp "-" (symbol-name (car vars)))
                  (str::isubstrp "++" (symbol-name (car vars)))))
         (append acc (list (car vars)))
       acc))))

(defund const-fns-gen-from-alist (alist)
  ;; Generate a set of constant functions from the binding list 'alist'
  (if (atom alist)
      nil
    (let* ((x (car alist))
           (name (car x))
           (body (cadr x)))
      (cons `(defundd ,name () ,body)
            (const-fns-gen-from-alist (cdr alist))))))

(define const-fns-gen (fn new-fn state
                          &key
                          (optimized 't)
                          (sub-pairs 'nil)
                          (body-sub 'nil)
                          (excluded-vars 'nil)
                          (preserved-vars 'nil)
                          (retained-vars 'nil)
                          (rules 'nil)
                          (inter-fns-enabledp 'nil))
  ;; Generate a set of constant functions from the bindings declared in the
  ;; non-recursive definition of function 'fn'.  It is required that 'fn' is
  ;; already defined.  CONST-FNS-GEN also generates the top-level function
  ;; 'new-fn' that corresponds to 'fn'.

  ;; If 'optimized' is non-nil, all variables bound to either a constant or a
  ;; single variable will be excluded from generating their corresponding
  ;; constant functions.

  ;; 'sub-pairs' contains pairs of terms (mostly variables) that are used in
  ;; substitutions performed by function REPLACE-LIST-ALL.

  ;; 'body-sub': like 'sub-pairs' but it applies only to the body of the
  ;; function's top-level binding.

  ;; 'excluded-vars': a list of bounded variables that are excluded from
  ;; generating their corresponding constant functions.

  ;; 'preserved-vars': a list of variables whose bindings are preserved, i.e.,
  ;; CONST-FNS-GEN won't process those bindings.

  ;; 'retained-vars': By default, CONST-FNS-GEN does not generate functions for
  ;; variables that it deems redundant. The variables declared in
  ;; 'retained-vars' will be treated as non-redundant, and hence, their
  ;; functions will be generated if their bindings exist.

  ;; 'rules': a list of rules that will be enabled when proving the final
  ;; equivalence lemma.

  ;; If 'inter-fns-enabledp' is non-nil, the generated intermediate functions
  ;; will be enabled.
  :mode :program
  (b* ((pkg-name (symbol-package-name fn))
       (all-fns (strings-to-symbol pkg-name (symbol-name new-fn) "-ALL-FNS"))
       (equiv-lemma (strings-to-symbol pkg-name (symbol-name fn) "-LEMMA"))
       (fn-def (cltl-def-from-name fn (w state)))
       (formal-args (caddr fn-def))
       (updated-formals (replace-list-all formal-args
                                          sub-pairs
                                          formal-args))
       (body (car (last fn-def)))
       ;; Rename variables in the MV-LET forms appearing in 'body'
       (renamed-body (rename-mv-let-vars body formal-args pkg-name))
       ;; 'alist' contains all variable bindings extracted from 'renamed-body'
       ;; except for the variables in 'preserved-vars'.
       (alist (bindings-extract renamed-body preserved-vars nil))
       (vars (remove-equal '*** (strip-cars alist)))
       (shrunk-body (shrink-bindings preserved-vars renamed-body))
       (distinct-vars (remove-duplicates vars))
       (renamed-alist (remove-assoc-equal
                       '***
                       (rename-keys-in-alist distinct-vars
                                             (append formal-args vars)
                                             alist
                                             pkg-name)))
       (renamed-vars (strip-cars renamed-alist))
       ;; Simplify the MV-NTH forms in 'renamed-alist'
       (simplified-alist (pairlis$ renamed-vars
                                   (pairlis$
                                    (simplify-mv-nth-forms
                                     (strip-cars
                                      (strip-cdrs renamed-alist)))
                                    nil)))
       ;; Extract variables that are not used in 'new-fn'
       (redundant-vars
        (remove-all (used-syms-extract (list retained-vars shrunk-body)
                                       simplified-alist
                                       nil)
                    renamed-vars))
       (no-redundant-alist (remove-assocs-equal
                            redundant-vars
                            simplified-alist))
       (tmp-inter-names (inter-names-extract (remove-assocs-equal
                                              redundant-vars
                                              renamed-alist)))
       (used-vars (strip-cars no-redundant-alist))
       (sub-pairs-with-dots (pairlis$ (strip-cars sub-pairs)
                                      (strip-cars-hack
                                       (strip-cdrs sub-pairs))))
       (updated-vars (replace-list-all (strip-cars sub-pairs)
                                       sub-pairs-with-dots
                                       used-vars))
       (replaced-vars (remove-duplicates
                       (append formal-args
                               used-vars
                               (strip-cars sub-pairs))))
       (updated-alist
        (pairlis$ updated-vars
                  (pairlis$
                   (replace-list-all replaced-vars
                                     sub-pairs
                                     (strip-cars
                                      (strip-cdrs no-redundant-alist)))
                   nil)))
       ;; Extract constant and unary bindings that will be eliminated from
       ;; 'updated-alist'
       (elim-bindings (if optimized
                          (elim-bindings-extract
                           updated-alist
                           nil nil)
                        nil))
       (elim-vars (strip-cars elim-bindings))
       (elim-fns (pairlis$ elim-vars nil))
       (excluded-fns (pairlis$ excluded-vars nil))
       ;; Extract bindings that are excluded from generating their
       ;; corresponding constant functions
       (substd-bindings (replace-list-all
                         elim-fns
                         (pairlis$ elim-fns
                                   (strip-cdrs elim-bindings))
                         (terms-substs excluded-vars
                                       updated-alist
                                       nil
                                       nil)))
       ;; Eliminate 'elim-bindings' and bindings with 'excluded-vars' from
       ;; 'updated-alist'
       (final-alist
        (replace-list-all (append elim-fns excluded-fns)
                          (append (pairlis$ elim-fns
                                            (strip-cdrs elim-bindings))
                                  substd-bindings)
                          (remove-assocs-equal
                           (append elim-vars excluded-vars)
                           updated-alist)))
       (single-used-vars
        (single-used-vars-extract (strip-cars final-alist)
                                  (strip-cars (strip-cdrs final-alist))
                                  nil))
       (- (cw "Single used functions: ~x0~%" single-used-vars))
       ;; The body of the top-level function 'new-fn'
       (new-body (replace-list-all
                  (remove-duplicates (append elim-vars excluded-vars
                                             formal-args vars))
                  (append elim-bindings
                          (pairlis$ excluded-vars
                                    (strip-cdrs substd-bindings))
                          sub-pairs
                          body-sub)
                  shrunk-body))
       (new-body (if (and (atom new-body) new-body)
                     (list new-body)
                   new-body))
       (inter-names (remove-all
                     (append elim-vars excluded-vars)
                     (replace-list-all (strip-cars sub-pairs)
                                       sub-pairs-with-dots
                                       tmp-inter-names)))
       (all-fn-names (cons new-fn
                           (strip-cars final-alist))))

    `(encapsulate
       ()

       (set-ignore-ok t) ;; will be treated as LOCAL

       ,@(append
          (const-fns-gen-from-alist final-alist)
          `((defundd ,new-fn ()
              ,new-body)

            ,@(if inter-fns-enabledp
                  `((in-theory (enable ,@inter-names))
                    (deftheory ,all-fns ',all-fn-names))
                `((deftheory ,all-fns ',all-fn-names)))

            (defthmd ,equiv-lemma
              (equal (,new-fn)
                     (,fn ,@updated-formals))
              :hints (("Goal"
                       :do-not '(preprocess)
                       :expand :lambdas
                       :in-theory '(,all-fns ,fn ,@rules)))))))))

;; ======================================================================

(defund insert-poset (x nodes alist)
  (declare (xargs :measure (acl2-count alist)
                  :guard (and (true-listp nodes)
                              (alistp alist))))
  (if (or (atom nodes)
          (atom alist))
      (cons x alist)
    (cons (car alist)
          (insert-poset x
                        (remove1-equal (caar alist) nodes)
                        (cdr alist)))))

(defund construct-depend-lst (x nodes syms mv-clique depend-acc)
  ;; Construct a list of nodes from 'nodes' that 'x' depends on.  These nodes
  ;; have to be present in 'syms', but not in 'mv-clique'.
  (declare (xargs :guard (and (true-listp syms)
                              (true-listp mv-clique))))
  (cond ((atom nodes) depend-acc)
        ((equal x (car nodes))
         (cons x depend-acc))
        ((member-equal (car nodes) mv-clique)
         (construct-depend-lst x
                               (cdr nodes)
                               syms
                               mv-clique
                               depend-acc))
        ((member-equal (car nodes) syms)
         (construct-depend-lst x
                               (cdr nodes)
                               syms
                               mv-clique
                               (cons (car nodes) depend-acc)))
        (t (construct-depend-lst x
                                 (cdr nodes)
                                 syms
                                 mv-clique
                                 depend-acc))))

(defund construct-depend-graph (alist names mv-letp mv-clique depend-graph)
  ;; Construct a dependency graph from the nodes 'names'.  Note that the
  ;; resulting graph is a DAG.
  (if (atom alist)
      depend-graph
    (let ((x (car alist)))
      (cond
       ((equal x '(*** (b (b)))) ;; Enter an MV-LET clique
        (construct-depend-graph
         (cdr alist) names t nil depend-graph))
       ((equal x '(*** (e (e)))) ;; Exit an MV-LET clique
        (construct-depend-graph
         (cdr alist) names nil nil depend-graph))
       (mv-letp ;; Inside an MV-LET clique
        (let* ((name (car x))
               (syms (remove-duplicates (symbols-extract (cadr x))))
               (depend-lst
                (construct-depend-lst name names syms mv-clique nil)))
          (if (< 1 (len depend-lst))
              (construct-depend-graph
               (cdr alist)
               names
               mv-letp
               ;; Update the current MV-LET clique
               (cons name mv-clique)
               ;; Update the dependency graph
               (insert-poset depend-lst (cdr depend-lst) depend-graph))
            (construct-depend-graph
             (cdr alist)
             names
             mv-letp
             ;; Update the current MV-LET clique
             (cons name mv-clique)
             depend-graph))))
       (t (let* ((name (car x))
                 (syms (remove-duplicates (symbols-extract (cadr x))))
                 (depend-lst
                  (construct-depend-lst name names syms nil nil)))
            (if (< 1 (len depend-lst))
                (construct-depend-graph
                 (cdr alist)
                 names
                 mv-letp
                 nil
                 ;; Update the dependency graph
                 (insert-poset depend-lst (cdr depend-lst) depend-graph))
              (construct-depend-graph
               (cdr alist)
               names
               mv-letp
               nil
               depend-graph))))))))

(defund depend-graph-to-weighted-nodes (alist acc)
  (if (atom alist)
      acc
    (let* ((x (car alist))
           (key (car x))
           (val (1+ (max-nats (assocs-vals (cdr x) acc)))))
      (depend-graph-to-weighted-nodes
       (cdr alist)
       (append acc (list (list key val)))))))

(defund replace-const-with-loop (loop-var x l term mv-clique flg)
  ;; Replace the constant functions 'l' called in 'term' with their loop
  ;; versions.

  ;; 'x' is bound to 'term'.

  ;; 'l' is a list of appearance-ordered bounded variables.

  ;; 'flg' indicates whether the entire MV-LET clique that 'x' belongs to is in
  ;; 'l'.  T means no, NIL means yes.
  (declare (xargs :guard (true-listp mv-clique)))
  (if (atom l)
      term
    (if (or flg
            (member-equal (car l) mv-clique)
            (equal x (car l)))
        (replace-const-with-loop
         loop-var
         x
         (cdr l)
         (replace-all (list (car l))
                      (cons (car l) `((1- ,loop-var)))
                      term)
         mv-clique
         t)
      (replace-const-with-loop
       loop-var
       x
       (cdr l)
       (replace-all (list (car l))
                    (cons (car l) `(,loop-var))
                    term)
       mv-clique
       nil))))

(defund loop-fns-gen-from-alist
  (loop-var alist names weight-alist base-cond init-alist
            mv-letp mv-clique pkg-name)
  ;; Generate a clique of mutually recursive functions from the binding list
  ;; 'alist'

  ;; 'weight-alist' provides a weight for each generated function.  We use this
  ;; weight to construct the measure for the corresponding function in a
  ;; lexicographic order.

  ;; 'mv-letp' indicates whether we are currently inside of an MV-LET clique.
  (if (atom alist)
      nil
    (let ((x (car alist)))
      (cond ((equal x '(*** (b (b)))) ;; Enter an MV-LET clique
             (loop-fns-gen-from-alist loop-var (cdr alist) names weight-alist
                                      base-cond init-alist t nil pkg-name))
            ((equal x '(*** (e (e)))) ;; Exit an MV-LET clique
             (loop-fns-gen-from-alist loop-var (cdr alist) names weight-alist
                                      base-cond init-alist nil nil pkg-name))
            (mv-letp ;; Inside an MV-LET clique
             (let* ((name (car x))
                    (body (cadr x))
                    (loop-body
                     (replace-const-with-loop
                      loop-var name names body mv-clique nil))
                    (weight (nfix (cadr (assoc-equal name weight-alist))))
                    (measure `(list (acl2-count ,loop-var) ,weight))
                    (init-pair (assoc-equal name init-alist))
                    (init (if init-pair
                              (cadr init-pair)
                            (list (strings-to-symbol pkg-name
                                                     (symbol-name name)
                                                     "-1")))))
               (cons `(defund ,name (,loop-var)
                        (declare (xargs :measure ,measure))
                        (if ,base-cond
                            ,init
                          ,loop-body))
                     (loop-fns-gen-from-alist
                      loop-var
                      (cdr alist)
                      names
                      (remove-assoc-equal name weight-alist)
                      base-cond
                      (remove-assoc-equal name init-alist)
                      mv-letp
                      ;; Update the current MV-LET clique
                      (cons name mv-clique)
                      pkg-name))))
            (t (let* ((name (car x))
                      (body (cadr x))
                      (loop-body
                       (replace-const-with-loop
                        loop-var name names body nil nil))
                      (weight (nfix (cadr (assoc-equal name weight-alist))))
                      (measure `(list (acl2-count ,loop-var) ,weight))
                      (init-pair (assoc-equal name init-alist))
                      (init (if init-pair
                              (cadr init-pair)
                              (list (strings-to-symbol pkg-name
                                                       (symbol-name name)
                                                       "-1")))))
                 (cons `(defund ,name (,loop-var)
                          (declare (xargs :measure ,measure))
                          (if ,base-cond
                              ,init
                            ,loop-body))
                       (loop-fns-gen-from-alist
                        loop-var
                        (cdr alist)
                        names
                        (remove-assoc-equal name weight-alist)
                        base-cond
                        (remove-assoc-equal name init-alist)
                        mv-letp
                        nil
                        pkg-name))))))))

(define loop-fns-gen (fn state
                         &key
                         (optimized 't)
                         (loop-var ''i)
                         (bogus-mutual-recursion 'nil)
                         (base-cond 'nil)
                         (init-alist 'nil)
                         (sub-pairs 'nil)
                         (excluded-vars 'nil)
                         (preserved-vars 'nil)
                         (retained-vars 'nil)
                         (inter-fns-enabledp 't))
  ;; Generate a set of mutually recursive functions from the bindings declared
  ;; in the recursive definition of function 'fn'.  It is required that 'fn' is
  ;; already defined.

  ;; If 'optimized' is non-nil, all variables bound to either a constant or a
  ;; single variable will be excluded from generating their corresponding
  ;; constant functions.

  ;; 'base-cond' is the base condition appeared in the definitions of the
  ;; generated mutually recursive functions.

  ;; 'init-alist': the initial values of the generated mutually recursive
  ;; functions.

  ;; 'sub-pairs' contains pairs of terms (mostly variables) that are used in
  ;; substitutions performed by function REPLACE-LIST-ALL.

  ;; 'excluded-vars': a list of bounded variables that are excluded from
  ;; generating their corresponding mutually recursive functions.

  ;; 'preserved-vars': a list of variables whose bindings are preserved, i.e.,
  ;; LOOP-FNS-GEN won't process those bindings.

  ;; 'retained-vars': By default, LOOP-FNS-GEN does not generate functions for
  ;; variables that it deems redundant. The variables declared in
  ;; 'retained-vars' will be treated as non-redundant, and hence, their
  ;; functions will be generated if their bindings exist.

  ;; If 'inter-fns-enabledp' is non-nil, the generated intermediate functions
  ;; will be enabled.
  :mode :program
  (b* ((pkg-name (symbol-package-name fn))
       (all-fns (strings-to-symbol pkg-name (symbol-name fn) "-ALL-FNS"))
       (?equiv-lemma (strings-to-symbol pkg-name (symbol-name fn) "-LEMMA"))
       (fn-def (cltl-def-from-name fn (w state)))
       (formal-args (caddr fn-def))
       (body (car (last fn-def)))
       ;; Rename variables in the MV-LET forms appearing in 'body'
       (renamed-body (rename-mv-let-vars body nil pkg-name))
       ;; 'alist' contains all variable bindings extracted from 'renamed-body'
       ;; except for the variables in 'preserved-vars'.
       (alist (bindings-extract renamed-body preserved-vars nil))
       (vars (remove-equal '*** (strip-cars alist)))
       (shrunk-body (shrink-bindings preserved-vars renamed-body))
       (distinct-vars (remove-duplicates vars))
       (renamed-alist (rename-keys-in-alist distinct-vars
                                            vars
                                            alist
                                            pkg-name))
       (renamed-vars (strip-cars renamed-alist))
       ;; Simplify the MV-NTH forms in 'renamed-alist'
       (simplified-alist (pairlis$ renamed-vars
                                   (pairlis$
                                    (simplify-mv-nth-forms
                                     (strip-cars
                                      (strip-cdrs renamed-alist)))
                                    nil)))
       ;; Extract variables that are not used in 'fn'
       (redundant-vars
        (remove-all (used-syms-extract (list retained-vars shrunk-body)
                                       simplified-alist
                                       nil)
                    renamed-vars))
       (no-redundant-alist (remove-assocs-equal
                            redundant-vars
                            simplified-alist))
       (tmp-inter-names (inter-names-extract (remove-assocs-equal
                                              redundant-vars
                                              renamed-alist)))
       (used-vars (strip-cars no-redundant-alist))
       (sub-pairs-with-dots (pairlis$ (strip-cars sub-pairs)
                                      (strip-cars-hack
                                       (strip-cdrs sub-pairs))))
       (updated-vars (replace-list-all (strip-cars sub-pairs)
                                       sub-pairs-with-dots
                                       used-vars))
       (replaced-vars (remove-duplicates
                       (append formal-args
                               used-vars
                               (strip-cars sub-pairs))))
       (updated-alist
        (pairlis$ updated-vars
                  (pairlis$
                   (replace-list-all replaced-vars
                                     sub-pairs
                                     (strip-cars
                                      (strip-cdrs no-redundant-alist)))
                   nil)))
       ;; Extract constant and unary bindings that will be eliminated from
       ;; 'updated-alist'
       (elim-bindings (if optimized
                          (elim-bindings-extract
                           updated-alist
                           nil nil)
                        nil))
       (elim-vars (strip-cars elim-bindings))
       (elim-fns (pairlis$ elim-vars nil))
       (excluded-fns (pairlis$ excluded-vars nil))
       ;; Extract bindings that are excluded from generating their
       ;; corresponding mutually recursive functions
       (substd-bindings (replace-list-all
                         elim-fns
                         (pairlis$ elim-fns
                                   (strip-cdrs elim-bindings))
                         (terms-substs excluded-vars
                                       updated-alist
                                       nil
                                       nil)))
       ;; Eliminate 'elim-bindings' and bindings with 'excluded-vars' from
       ;; 'updated-alist'
       (final-alist
        (replace-list-all (append elim-fns excluded-fns)
                          (append (pairlis$ elim-fns
                                            (strip-cdrs elim-bindings))
                                  substd-bindings)
                          (remove-assocs-equal
                           (append elim-vars excluded-vars)
                           updated-alist)))
       (final-vars (strip-cars final-alist))
       (depend-graph (construct-depend-graph final-alist final-vars
                                             nil nil nil))
       (weight-alist (depend-graph-to-weighted-nodes depend-graph nil))
       (mutually-recur-fns (cons 'mutual-recursion
                                 (loop-fns-gen-from-alist
                                  loop-var
                                  final-alist final-vars weight-alist
                                  base-cond init-alist nil nil pkg-name)))
       (single-used-vars
        (single-used-vars-extract final-vars
                                  (strip-cars (strip-cdrs final-alist))
                                  nil))
       (- (cw "Single used functions: ~x0~%" single-used-vars))
       (inter-names (remove-all
                     (append elim-vars excluded-vars)
                     (replace-list-all (strip-cars sub-pairs)
                                       sub-pairs-with-dots
                                       tmp-inter-names)))
       (all-fn-names (remove-equal '*** final-vars)))

    `((include-book "ordinals/lexicographic-book" :dir :system)

      (encapsulate
        ()

        (set-ignore-ok t)
        (set-well-founded-relation l<)
        (set-bogus-mutual-recursion-ok ,bogus-mutual-recursion)
        ,mutually-recur-fns)

      (deftheory ,all-fns ',all-fn-names)

      ,@(if inter-fns-enabledp
            `((in-theory (disable ,@(pairlis$ all-fn-names nil)))
              (in-theory (enable ,@inter-names)))
          `((in-theory (disable ,@(pairlis$ all-fn-names nil))))))))
