#|

  A Simple Random Number Generator               Version 0.2 
  David Rager <ragerdl@cs.utexas.edu>         April 27, 2005 

  This file is in the public domain.  You can freely use it for any
  purpose without restriction. 
 
  This work is modified from a random.lisp generated by Jared Davis
  <jared@cs.utexas.edu>. 

  This is just a basic pure multiplicative pseudorandom number gen-
  erator.  *M31* is the 31st mersenne prime, and *P1* is 7^5 which
  is a primitive root of *M31*, ensuring that our period is M31 - 1.
  This idea is described in "Elementary Number Theory and its Appli-
  cations" by Kenneth H. Rosen, Fourth Edition, Addison Wesley 1999,
  in Chapter 10.1, pages 356-358.
 
  This random number generator does not store the seed. Rather, it 
  relies upon the user providing a fresh value each time. This
  should not be considered a truely random number generator, but
  it will suffice for providing executability and a logical grounds
  for proofs about numbers conceptually random to us.
 
  Normally we are not particularly interested in reasoning about ran-
  dom numbers.  However, we can say that the number k generated is an
  an integer, and that 0 <= k < max when max is a positive integer. 
  See the theorems genrandom-aux-minimum and genrandom-maximum. 

|#

(in-package "CRYPTO")

(encapsulate 
 nil
 (local (include-book "arithmetic-3/floor-mod/floor-mod" :dir :system))
 
 (defconst *M31* 2147483647)    
 (defconst *P1* 16807)          
 (defconst *max-random* 2010017) ; a large yet readable prime number
 
 
 (defun genrandom-aux (seed max)
   (mod (mod (* *P1* seed) *M31*) max))
 
 (defun genrandom (prev-number)
   (genrandom-aux prev-number *max-random*))
 
 
 (defthm genrandom-aux-integer
   (implies (and (integerp seed)
                 (integerp max))
            (integerp (genrandom-aux seed max))))

 (defthm genrandom-aux-minimum 
   (implies (and (integerp seed)
                 (integerp max)
                 (< 0 max))
            (<= 0 (genrandom-aux seed max))))
 
 
 (defthm genrandom-aux-maximum
   (implies (and (integerp seed)
                 (integerp max)
                 (< 0 max))
            (< (genrandom-aux seed max) max)))
 
 (defthm genrandom-integer
   (implies (and (integerp seed))
            (integerp (genrandom seed))))
 
 (in-theory (disable genrandom-aux))
 (in-theory (disable genrandom))
)
           
